%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%
| Many of the methods for refining intersection types and their members
| are integrated into the tactic library.  The intersection type works
| much like a dependent function type, and many of the ML functions that
| work on dependent function types can be modified slightly to work
| on intersection types.  However, some of the functions are unique to the
| intersection type.  We include them here.
%

%[
********************************************************************************
Intersection type tactics
********************************************************************************
]%

%
| We have a special case for decomposing the intersection.
| H >> a = b in isect x:A.B
|
| BY ISectEqTypeCD
|
| `main`: H, z:A >> a = b in B[z]
| `wf`: H >> A = A in Ui
|
| ISectEqTypeCD takes care of choosing variable names and level expressions.
|
%
let ISectEqTypeCD p =
  let T, a, b = dest_equal (concl p) in
  let x, A, B = dest_isect T in
  let z = maybe_new_proof_var (get_var_arg `v1` p ? x) p in
  let le_arg, WFTac = mk_le_arg_and_wf_tac p A
  in
      (Refine `isect_memberEquality` [le_arg ; mk_var_arg z] 
       THENL [Id; WFTac]) p
;;

update_EqTypeCD_additions `ISectEqTypeCD` ISectEqTypeCD;;

%
| We can also special-case the intersection in a hypothesis.
| H, i:a = b in isect x:A.B, J >> C
|
| BY With 'c' (ISectEqTypeHD i)
|
| `main`: H, i:a = b in isect B[c], J >> C
| `wf`: H, i:a = b in isect x:A.B, J >> c = c in A
%
let ISectEqTypeHD i p =
    let T, a, b = dest_member_or_equal (h i p) in
    let x, A, B = dest_isect T in
    let c = get_term_arg `t1` p ? failwith `ISectEqTypeHD: requires argument`
    in
	(AssertAtHyp (i + 1) (mk_equal_term (subst [(x, c)] B) a b)
	 THENL [Refine `isect_member_caseEquality` [mk_term_arg T; mk_term_arg c]
		THENL [NthHyp i; AddHiddenLabel `wf`]
		;Thin i]) p;;

update_EqTypeHD_additions `ISectEqTypeHD` ISectEqTypeHD;;

%
| We also provide for case analysis on membership in an intersection
| type.  This is along the lines of lambdaEquality.
|
| H >> a = b in B[c/x]
|
| BY GenTypeCD 'c = z in C'
|
| `main`: H >> a = b in isect z:C.B
| `wf`: H >> c = c in C
%
let GenTypeCD eq p =
    let T, a, b = dest_member_or_equal (concl p) in
    let c, zt, C = dest_equal eq in
    let z = dest_var zt ? failwith `GenTypeCD: replacement is not a var` in
    let z' = maybe_new_proof_var z p in
    let T_of_z' = replace_subterm c (mk_var_term z') T in
    let isect_term = mk_isect_term z' C T_of_z'
    in
	(Refine `isect_member_caseEquality` [mk_term_arg isect_term; mk_term_arg c]
	 THENL [Id; AddHiddenLabel `wf`]) p
;;

%
| We also add ISectEqTypeCD to Auto, since it is really the
| most logical step that Auto can take given an intersection type.
%
let ISectMemTypeCD p =
    let goal = concl p in
    let T, () = dest_member_or_equal goal in
	if is_isect_term T then
	    (if is_equal_term goal then
		 ISectEqTypeCD
	     else
		 EqToMemberEq (\i.ISectEqTypeCD) 0) p
	else
	    failwith `ISectMemTypeCD: type must be intersection`;;

update_Auto_additions `ISectMemTypeCD` ISectMemTypeCD;;
