%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%
%[
******************************************************************************
******************************************************************************
JPROVER.ML
******************************************************************************
******************************************************************************
]%

%%
% This file implements code to run the Jprover tactic both with and without 
  multi/single type conversion.  The Jprover tactic requires connection to 
  metaprl, and the jprover_hook running in metaprl, which are outside this file.

  First, nuprl tacticals are defined to allow rebuilding of the nuprl sequent tree, 
  Next is basic jprover tactic code, which calls jprover and maps its result to a nuprl
  tactic.
  Further, is jprover tactic code that does the type conversions.
%
%%

% Defines nuprl tacticals and tactics needed for the mapping of the jprover result 
  to a nuprl sequent. %

let quote = string_to_tok "`" ;;
let quote_name name       = quote ^ name ^ quote ;;
let quote_names names  = 
   let name.remaining = names 
   in
     ` [` ^ (quote_name name) 
          ^ (concatl (map (\n. `;` ^ quote_name n) remaining))
          ^ `]`
;;

%
Ax , Andr,  Andl, Orr, Orr1, Orr2, Orl, Impr, Impl, Negr, Negl,
Allr, Alll, Exr, Exl
%
let Run  tac_tok     = refine (make_tactic_rule (mk_text_term 
						    (tok_to_string tac_tok)));;
let RunT tac_tok term  = 
  refine (make_tactic_rule 
	  (make_term (`!ml_text_cons`,[]) 
		      [ [], mk_text_term (tok_to_string (tac_tok ^ ` `))
                      ; [], term
                      ]
		     ))
;;
let RunTT tac_tok term1 term2  = 
  refine (make_tactic_rule 
	  (make_term (`!ml_text_cons`,[]) 
		      [ [], mk_text_term (tok_to_string (tac_tok ^ ` `))
                      ; [], term1
		      ; [], mk_text_term " "
                      ; [], term2
                      ]
		     ))
;;
let RunH tac_tok hyp        = Run  (tac_tok ^ ` (` ^(tok_of_int hyp)^ `)`);;
let RunHT tac_tok hyp term  = RunT (tac_tok ^ ` (` ^(tok_of_int hyp)^ `)`) term;;
let RunNames tac names       = Run (tac ^ ` ` ^ (quote_names names));;
let RunNamesH tac names hyp  = RunH (tac ^ ` ` ^ (quote_names names)) hyp;;

let RunN tac name            = Run (tac ^ ` ` ^ (quote_name name));;

let impliesR        = impR  
and rev_impliesR    = pmiR 

and andI            = Run     `andR`       
and orI1            = Run     `orR1`       
and orI2            = Run     `orR2`       
and impI            = Run     `impR`       
and pmiI            = Run     `pmiR`       
and iffI            = Run     `iffR`       
and falseI          = Run     `falseR`     
and notI            = Run     `notR`       
and allI            = Run     `allR`       
and exI term        = RunT    `exR` term     

and andE   hyp      = RunH    `andL`       hyp      
and orE    hyp      = RunH    `orL`        hyp      
and impE   hyp      = RunH    `impL`       hyp      
and pmiE   hyp      = RunH    `pmiL`       hyp      
and iffE   hyp      = RunH    `iffL`       hyp       
and iffE_b hyp      = RunH    `iffL_back`  hyp       
and notE   hyp      = RunH    `notL`       hyp      
and falseE hyp      = RunH    `falseL`     hyp           
and allE   hyp term = RunHT   `allL`       hyp term    
and exE    hyp      = RunH    `exL`        hyp       
		      
and hyp    hyp      = RunH    `hypothesis`  hyp      
and thin   hyp      = RunH    `Thin`        hyp      
;;

let get_quantified_type term =
  let sub = find_subterm (\y. let x = term_of_bterm y 
                              in (is_all_term x) or (is_exists_term x)) term in
  let opid, (l,type).bterms = dest_std_term sub in
  type
;;

let quantified_var_of term =
  if is_all_term term or is_exists_term term 
    then let opid, [[],T; [x],prop] = dest_std_term term in x 
    else failwith `unexpected opid`
;;

% returns int n where term is the nth hyp in hyps, %%or 0 if term matches concl,%% or fails%
let find_hyp_number term p =
  let hyps = map snd (dest_hyps p) %and
      c = concl p% in
  letrec f l i =
    if null l then failwith `find hyp number`
    else let hyp.tail = l in
         if alpha_equal_terms term hyp then i else f tail (1+i) in
  f %(c .)% hyps 1 %0%
;; 

% alist of boolean function names (as variable terms) and the type they test%
letref Type_Function_alist = [] : (term # term) list;;
letref Function_Index = 0;;
let f_id = "Type_Identifier_Function";;

let is_type_function term =
  (let term, value = dest_apply term in
  let v = dest_var term in
  let s = tok_to_string (var_to_tok v) in
  is_prefix (string_to_toks f_id) (string_to_toks s)) ?
  false
;;

let convert_untyped_term_aux term dest_f mk_f1 dest_f2 =
 (let var, type, prop = dest_f term in
 let t1,t2 = dest_f2 prop in
 if is_type_function t1 then
 let t3, value = dest_apply t1 in
 let type = rev_apply_alist Type_Function_alist t3 %alpha_equal_terms% in
 mk_f1 var type t2
 else term) ? term
;;

let convert_untyped_term term =
 if is_all_term term then
 convert_untyped_term_aux term dest_all mk_all_term dest_implies

 else if is_exists_term term then
 convert_untyped_term_aux term dest_exists mk_exists_term dest_and

 else failwith `convert untyped term not`
;;          

%transform uni-typed term to multi typed%
let nuprl_map_type term =
  map_term_bottom_up (\t. (is_all_term t) or (is_exists_term t)) convert_untyped_term term
;;

let OnHypOf term tac p =  
     tac (find_hyp_number (nuprl_map_apply (nuprl_map_var term)) p) p
;;

let OnHypOfT term tac p =  
     tac (find_hyp_number (nuprl_map_type (nuprl_map_apply (nuprl_map_var term))) p) p
;;

let RenameFirstIn term1 term2 = 
    RenameBVars [quantified_var_of (nuprl_map_apply (nuprl_map_var term1)),
		 dest_var term2]
;;

% For testing and debugging %
%%
letref temp_proofs = [] : proof list;;
let Assign p = 
  temp_proofs := p.temp_proofs;
  Id p
;;
%%

% Jprover functionality %

% Default multiplicity limit for jprover %
letref jprover_limit = 5;;
letref jresult_list = [] : (string # term # term) list;;
let pop_jresult_list () =
 if jresult_list = [] then fail
 else
 let val = hd jresult_list in
 jresult_list := tl jresult_list;
 val
;;

% jprover introduces vars as it finds a solution, these are accounted for when 
  building the  nuprl proof %
let jprover_default_var = tok_to_var `v0_jprover`;;
let v0_jprover_term = mk_var_term jprover_default_var;;
let v0_jprover_term_p = alpha_equal_terms v0_jprover_term;;

% maps jprover result to corresponding nuprl tactic represented as a term list
note: jprover result is string#term#term list, first 3 args%

let jname_map name term1 term2 =

  if name = "Ax" then OnHypOf term1 hypothesis
  else if name = "Andr" then andR
  else if name = "Andl" then OnHypOf term1 andL
      
  else if name = "Orr" then fail "orr?" %"TryOnC (D 0)"%  
  else if name = "Orr1" then orR1
  else if name = "Orr2" then orR2
  else if name = "Orl" then OnHypOf term1 orL
      
  else if name = "Impr" then impR
  else if name = "Impl" then OnHypOf term1 impL
      
  else if name = "Negr" then notR
  else if name = "Negl" then OnHypOf term1 notL

  else if name = "Allr" then RenameFirstIn term1 term2 0 THEN allR
  else if name = "Alll" then OnHypOf term1 (\t. allL t term2)
 
  else if name = "Exr" then exR term2
  else if name = "Exl" then
   OnHypOf term1 (\i.RenameFirstIn term1 term2 i THEN exL i)   
      
  else fail (J "jname_map: inapproprate jname " name)
;;
let itext_cons_term term1 term2 = make_icons_term itext_cons_op term1 term2
;;

let jname_map_ttt name term1 term2 =

  if name = "Ax" then itext_cons_term (itext_term " OnHypOf ")
    (itext_cons_term (mk_prl_term term1) (itext_term " hypothesis "))
  else if name = "Andr" then itext_cons_term (itext_term " OnHypOf ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " andR "))
  else if name = "Andl" then itext_cons_term (itext_term " OnHypOf ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " andL "))
      
  else if name = "Orr" then fail "orr?" %"TryOnC (D 0)"%  
  else if name = "Orr1" then itext_term " orR1 "
  else if name = "Orr2" then itext_term " orR2 "
  else if name = "Orl" then itext_cons_term (itext_term " OnHypOf ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " orL "))
      
  else if name = "Impr" then itext_term " impR "
  else if name = "Impl" then itext_cons_term (itext_term " OnHypOf ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " impL "))
      
  else if name = "Negr" then itext_term " notR "
  else if name = "Negl" then itext_cons_term (itext_term " OnHypOf ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " notL "))

  else if name = "Allr" then itext_cons_term
		    (itext_term "RenameFirstIn ")
		    (itext_cons_term (mk_prl_term term1)
				     (itext_cons_term (mk_prl_term term2)
								  (itext_term " 0 THEN allR ")))
  else if name = "Alll" then itext_cons_term (itext_term " OnHypOf ")
		    (itext_cons_term (mk_prl_term term1)
				     (itext_cons_term (itext_term " (\\t. allL t ")
						      (itext_cons_term (mk_prl_term term2)
								       (itext_term " ) "))))
 
  else if name = "Exr" then itext_cons_term (itext_term " exR ") (mk_prl_term term2)
  else if name = "Exl" then
   itext_cons_term (itext_term " OnHypOf ")
		   (itext_cons_term
		     (mk_prl_term term1)
		     (itext_cons_term (itext_term " (\\i.RenameFirstIn ")
				      (itext_cons_term (mk_prl_term term1)
						       (itext_cons_term
							(mk_prl_term term2)
							(mk_text_term " i THEN exL i )")))))
      
  else fail (J "jname_map: inapproprate jname " name)
;;
		    
let jname_map_tree name term1 term2 =

  if name = "Ax" then OnHypOf term1 hyp
  else if name = "Andr" then andI
  else if name = "Andl" then OnHypOf term1 andL
      
  else if name = "Orr" then fail "orr?" %"TryOnC (D 0)"%  
  else if name = "Orr1" then orI1
  else if name = "Orr2" then orI2
  else if name = "Orl" then OnHypOf term1 orE
      
  else if name = "Impr" then impI
  else if name = "Impl" then OnHypOf term1 impE
      
  else if name = "Negr" then notI
  else if name = "Negl" then OnHypOf term1 notE

  else if name = "Allr" then RenameFirstIn term1 term2 0 THEN allI    
  else if name = "Alll" then OnHypOf term1 (\t. allE t term2)
 
  else if name = "Exr" then exI term2
  else if name = "Exl" then    
    OnHypOf term1 (\i.RenameFirstIn term1 term2 i THEN exE i) 
      
  else fail (J "jname_map_tree: inapproprate jname " name)
;;
		  
% these tactics split the proof into 2 subgoals %
letref tactic_split_list = ["Andr"; "Orl"; "Impl"] ;;
let tactic_split_p name = member name tactic_split_list ;;

%
let hyp_member_p term hyps =
   member_p term (map_isexpr_to_list icons_op (\x. x) hyps) alpha_equal_terms;;
%

let dest_jprover_term jt =
 let ps, bts = dest_term_with_opid `jprover` jt in
   if null bts then failwith `jprover failed`
   else let [((), s); ((), t1); ((), t2)] = bts in
   ((string_of_istring_term s), t1, t2)
;;
	     
let dest_jprover_list term = map_isexpr_to_list icons_op dest_jprover_term term
;;

%
 jprover only cares about hyps that are propositions, which
 have invisible vars. other hyps, like type decls are meaningless
 at this stage since jprover assumes a single type.
%
let jprover_hyps p =
    map_to_ilist (\h. (fst (snd (dest_declaration h)))) icons_op
      (filter (\d. let var,T,hidden = dest_declaration d in 
	           is_invisible_var var)
	 (hyps p))
;;

% takes result list and produces ttt for ttt refinement %
let jprover_list_to_ttt jresult =
		  
  jresult_list := jresult;
  letrec jprover_to_nuprl_tactic (name, term1, term2) =
	    
  let nup_tac = jname_map_ttt name term1 term2 in
		  
  let return_tac =
    if tactic_split_p name then 
        let tac1 = jprover_to_nuprl_tactic (pop_jresult_list ()) in
        let tac2 = jprover_to_nuprl_tactic (pop_jresult_list ()) in

	make_icons_term tactic_tree_cons_op
		nup_tac
                (map_to_ilist id tactic_tree_cons_op [tac1; tac2])
     else

  % if at end of proof or at end of a subgoal from a split than don't iterate %
  
     if (null jresult_list) or (name = "Ax") then
     make_icons_term tactic_tree_cons_op
		nup_tac
		(map_to_ilist id tactic_tree_cons_op [])

     else make_icons_term tactic_tree_cons_op
                nup_tac
		(map_to_ilist id tactic_tree_cons_op
		              [jprover_to_nuprl_tactic (pop_jresult_list ())])
    in

  %if jprover introduced new v0 variable, need to do the assert%
  
  if ((name = "Alll") or (name = "Exr")) & (v0_jprover_term_p term2) then
  
     make_icons_term tactic_tree_cons_op
        (itext_cons_term (itext_term " RunT `Assert ` ")
		      (mk_prl_term (mk_exists_term jprover_default_var
						   (get_quantified_type term1) true_term)))
        (make_icons_term tactic_tree_cons_op
		      (itext_term " Try (D 0 THEN Auto) ")
		      (make_icons_term tactic_tree_cons_op (itext_term " D (-1) ") return_tac))

  else % no need to do the assert %
  
  return_tac in

  jprover_to_nuprl_tactic (pop_jresult_list ())

;;
  
let JproverAssert qterm tac =
   Assert (mk_exists_term jprover_default_var (get_quantified_type qterm) true_term)
      THENL [Try (D 0 THEN Auto); D (-1) THEN tac]
;;

let JproverAssertR qterm tac =
   RunT `Assert ` (mk_exists_term jprover_default_var (get_quantified_type qterm) true_term)
      THENL [(Run `Try (D 0 THEN Auto)`); ((Run `D (-1)`) THEN tac)]
;;

% map_f converts the jprover result to a nuprl tactic,
  tree_p true means build the internal proof tree as well%

let jprover_tactic p map_f tree_p =
  
  jresult_list := nil;
  letref assert_p = false in
  let jhyps = map_to_ilist snd icons_op (dest_hyps p) in
  let jresult = dest_jprover_list (jprover jprover_limit (jprover_hyps p) (concl p)) in
  jresult_list := jresult;
  
  letrec build_tactic_f (name, term1, term2) = 
  let nup_tac = map_f name term1 term2 in 
  let return_tac =
    (if tactic_split_p name then 
         let tac1 = build_tactic_f (pop_jresult_list ()) in
         let tac2 = build_tactic_f (pop_jresult_list ()) in
         nup_tac THENL [tac1; tac2]
     else

  % if at end of proof or at end of a subgoal from a split than don't iterate %
     if (name = "Ax") or (null jresult_list) then nup_tac 

     else nup_tac THEN build_tactic_f (pop_jresult_list ())
     ) in

  %if jprover introduced new v0 variable, need to do the assert%
  if ((name = "Alll") or (name = "Exr")) & (v0_jprover_term_p term2) then
      (assert_p := true;
      (if tree_p then JproverAssertR term1 return_tac
       else % not tree_p - no need to call Run to build internals %
       JproverAssert term1 return_tac))

  else  % no need to do the assert %  
  return_tac in

  build_tactic_f (pop_jresult_list ())
;;

let jprover_aux map_f tree_p proof = (jprover_tactic proof map_f tree_p) proof;;

%Top level tactics%

%let JproverD = _aux jname_map_debug false;;%
let Jprover = jprover_aux jname_map false;;
let JproverR = jprover_aux jname_map_tree true;;

%
let JproverTest p =
  PushArgs [(`jprover`, (term_to_arg (jprover jprover_limit (jprover_hyps p) (concl p))))] p  
;;
%

let JproverTestTTT p =
  PushArgs [(`jprover`, (term_to_arg
			 (jprover_list_to_ttt (dest_jprover_list
			       (jprover jprover_limit (jprover_hyps p) (concl p))))))] p  
;;
%produces anno_arg_cons term that can be passed to anno_args_to_ttt function below%
let JproverTest p =
  let jprover_term_list = map_isexpr_to_list icons_op id (jprover jprover_limit (jprover_hyps p) (concl p)) in
  PushArgs (map (\x. (`jprover`, (term_to_arg x))) jprover_term_list) p  
;;
  
let ianno_arg_cons_op = (`!anno_arg_cons`, nil);;
let value_of_anno_arg a =
  let [(),n] = bterms_of_term a in
  let [((),o); ((), p)] = bterms_of_term n in
  let [((),q); ((), r)] = bterms_of_term p in
  let [((),s); ((), u)] = bterms_of_term r in
  let [(),v] = bterms_of_term u in v
;;
       
let anno_args_to_ttt aa =
 let jprover_list = map_isexpr_to_list ianno_arg_cons_op
                        (\t. (dest_jprover_term (value_of_anno_arg t))) aa in
  jprover_list_to_ttt jprover_list
;;

let jprover_asynch_tactic p term =

  jresult_list := nil;
  letref assert_p = false in
  let jresult = dest_jprover_list term in
  jresult_list := jresult;
  
  letrec build_tactic_f (name, term1, term2) = 
  let nup_tac = jname_map name term1 term2 in 
  let return_tac = 
  (if tactic_split_p name then 
  let tac1 = build_tactic_f (pop_jresult_list ()) in
  let tac2 = build_tactic_f (pop_jresult_list ()) in
  nup_tac THENL [tac1; tac2]
  else
  % if at end of proof or at end of a subgoal from a split than don't iterate %
  if (null jresult_list) or (name = "Ax") then nup_tac 
  else nup_tac THEN build_tactic_f (pop_jresult_list ())) in

  if ((name = "Alll") or (name = "Exr")) & (v0_jprover_term_p term2) then

     (assert_p := true;
      JproverAssert term1 return_tac)
  	
  else % no need to do the assert %
  return_tac in

  build_tactic_f (pop_jresult_list ())
;;

%
 Jprover type stuff
%

% The code below transforms a multi-typed sequent to a uni-typed sequent,
then sends this to jprover and transforms the jprover result to match the original
sequent, and builds a nuprl proof from the transformed result.

The sequent transformation is as follows:
  For each distinct type, T, in sequent, convert to common dummy type,
  introduce a new boolean function that returns true if it's arg is in T, 
  For each quantifier term, or type dedeclaration,  convert it to the 
  corresponding propositions as below:

 - Type declaration, i.e. x:T becomes the proposition T'(x) where
    T' is a the new function introduced that is true iff x is in T. If there are
    n types in the nuprl sequent then n functions, named 
    Type_Identifier_Function_1 ... Type_Identifier_Function_n are introduced 
 - All x:T. P(x) becomes all x:Dummy T'(x) => P(x)
 - Exists x:T. P(x) becomes exists x:Dummy T'(x) and P(x)

%

% arbitrary type used in the transformation from multi to uni-type sequent.%
letref dummy_type = enum_universe;;

let new_type_function () =
 let name = concatenate_strings [f_id; int_to_string Function_Index] in
 let val = mk_var_term (string_to_variable name) in
 Function_Index := (1 + Function_Index);
 val
;;

let type_function_apply_term val var =
 mk_apply_term val (mk_var_term var)
;;

% 
  maintain list of props, i.e T'(x) (transformed type declarations)
  that we will assert for jprover sake
  ie - the apply terms that say that
  the boolean type indicator fn of a variable holds true.
%

letref Type_Function_assertions = [] : term list;;

let update_assertions_list m =
  if member_p m Type_Function_assertions alpha_equal_terms then ()
  else (Type_Function_assertions := m.Type_Function_assertions; ())
;;

let convert_typed_term_aux term dest_f mk_f1 mk_f2 =
 let var, type, prop = dest_f term in
 let val = (apply_alist_p Type_Function_alist type alpha_equal_terms) ?
	   (let fun = new_type_function () in
            (Type_Function_alist := update_alist_p Type_Function_alist type fun alpha_equal_terms;
            fun)) in
 let type_term = type_function_apply_term val var in
 update_assertions_list type_term; 
 mk_f1 var dummy_type (mk_f2 type_term prop)
;;

let convert_typed_term term =
 if is_all_term term then
 convert_typed_term_aux term dest_all mk_all_term mk_implies_term

 else if is_exists_term term then
 convert_typed_term_aux term dest_exists mk_exists_term mk_and_term

 else failwith `maybe convert term not`
;;

%transform multi-typed term to single type%
let jprover_map_type term =
  map_term_bottom_up (\t. (is_all_term t) or (is_exists_term t)) convert_typed_term term
;;

let map_type_hyp hyp =
    let var,T,hidden = dest_declaration hyp in
    if is_invisible_var var then T
    else
    let val = (apply_alist_p Type_Function_alist T alpha_equal_terms) ?
	      (let fun = new_type_function () in
              (Type_Function_alist := update_alist_p Type_Function_alist T fun alpha_equal_terms;
              fun)) in
 let type_term = type_function_apply_term val var in
 %update_assertions_list type_term;%
 %LAL don't think I need to update this for hyps? % 
 type_term

;;

let get_jdeclst_term p = map_to_ilist map_type_hyp icons_op (hyps p);;
let get_jdeclst p = map_isexpr_to_list icons_op (\d. d) (get_jdeclst_term p);;

% 
  to convert jresults for the modified sequent into a tactic that can be used on
  the original sequent, skip over the impr rules when the concl had 
  the type_function_test term in it and ignore proof of lhs.  also impl's

  also the ands that were generated by the exist terms
%

let jnamet_map name term1 term2 =

  if name = "Ax" then OnHypOfT term1 hypothesis
  else if name = "Andr" then andR
  else if name = "Andl" then OnHypOfT term1 andL
      
  else if name = "Orr" then fail "orr?" %"TryOnC (D 0)"%  
  else if name = "Orr1" then orR1
  else if name = "Orr2" then orR2
  else if name = "Orl" then OnHypOfT term1 orL
      
  else if name = "Impr" then
    let t1,t2 = dest_implies term1 in
    if is_type_function t1 %term1?% then Id % wip : not really id %
    else impR
  else if name = "Impl" then 
    let t1,t2 = dest_implies term1 in
    if is_type_function t1 then Id % wip : not really id %
    else OnHypOfT term1 impL
      
  else if name = "Negr" then notR
  else if name = "Negl" then OnHypOfT term1 notL

  else if name = "Allr" then RenameFirstIn term1 term2 0 THEN allR
  else if name = "Alll" then OnHypOfT term1 (\t. allL t term2)
 
  else if name = "Exr" then exR term2
  else if name = "Exl" then
   OnHypOfT term1 (\i.RenameFirstIn term1 term2 i THEN exL i)   
      
  else fail (J "jnamet_map: inapproprate jname " name)
;; 

let jnamet_map_ttt name term1 term2 =

  if name = "Ax" then itext_cons_term (itext_term " OnHypOfT ")
    (itext_cons_term (mk_prl_term term1) (itext_term " hypothesis "))
  else if name = "Andr" then itext_cons_term (itext_term " OnHypOfT ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " andR "))
  else if name = "Andl" then itext_cons_term (itext_term " OnHypOfT ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " andL "))
      
  else if name = "Orr" then fail "orr?" %"TryOnC (D 0)"%  
  else if name = "Orr1" then itext_term " orR1 "
  else if name = "Orr2" then itext_term " orR2 "
  else if name = "Orl" then itext_cons_term (itext_term " OnHypOfT ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " orL "))
      
  else if name = "Impr" then
    let t1,t2 = dest_implies term1 in
    if is_type_function t1 %term1?% then itext_term " Id " % wip : not really id %
    else itext_term " impR "
  else if name = "Impl" then 
    let t1,t2 = dest_implies term1 in
    if is_type_function t1 then itext_term " Id " % wip : not really id %
    else  itext_cons_term (itext_term " OnHypOfT ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " impL "))
      
  else if name = "Negr" then itext_term " notR "
  else if name = "Negl" then itext_cons_term (itext_term " OnHypOfT ")
		    (itext_cons_term (mk_prl_term term1) (itext_term " notL "))

  else if name = "Allr" then itext_cons_term
		    (itext_term "RenameFirstIn ")
		    (itext_cons_term (mk_prl_term term1)
				     (itext_cons_term (mk_prl_term term2)
								  (itext_term " 0 THEN allR ")))
  else if name = "Alll" then itext_cons_term (itext_term " OnHypOfT ")
		    (itext_cons_term (mk_prl_term term1)
				     (itext_cons_term (itext_term " (\\t. allL t ")
						      (itext_cons_term (mk_prl_term term2)
								       (itext_term " ) "))))
 
  else if name = "Exr" then itext_cons_term (itext_term " exR ") (mk_prl_term term2)
  else if name = "Exl" then
   itext_cons_term (itext_term " OnHypOfT ")
		   (itext_cons_term
		     (mk_prl_term term1)
		     (itext_cons_term (itext_term " (\\i.RenameFirstIn ")
				      (itext_cons_term (mk_prl_term term1)
						       (itext_cons_term
							(mk_prl_term term2)
							(mk_text_term " i THEN exL i )")))))
            
  else fail (J "jnamet_map: inapproprate jname " name)
;; 

let jprover_ignore_p (name, term1, term2) =
  (if name = "Impr" then
      let t1,t2 = dest_implies term1 in is_type_function t1
   else if name = "Impl" then
      let t1,t2 = dest_implies term1 in is_type_function t1
   else
   if name = "Andr" then 
   let t1,t2 = dest_and term1 in is_type_function t1
   else
   if name = "Andl" then
   let t1,t2 = dest_and term1 in is_type_function t1
   else
   if name = "Ax" then is_type_function term1
   else false)
   ? false
;;

let jprovert_tactic p map_f tree_p =

  jresult_list := nil;
  Type_Function_alist := nil;
  Function_Index := 0;
  Type_Function_assertions := nil;
  letref assert_p = false in  % only need to assert the jp var at most one time %
  let jconcl = jprover_map_type (concl p) in
  let jresult = dest_jprover_list
                 (jprover jprover_limit (map_to_ilist (\d. d)
					icons_op
					(append (get_jdeclst p) Type_Function_assertions)) jconcl) in
			     
  jresult_list := filter (\x. not (jprover_ignore_p x))
                         (map (\(name, t1,t2). (name, (nuprl_map_type t1), t2)) jresult);
  letrec build_tactic_f (name, term1, term2) = 
  let nup_tac = map_f name term1 term2 in 
  let return_tac = if tactic_split_p name then 
   let tac1 = build_tactic_f (pop_jresult_list ()) in
   let tac2 = build_tactic_f (pop_jresult_list ()) in
   nup_tac THENL [tac1; tac2]
   else
  % if at end of proof or at end of a subgoal from a split than don't iterate %
  if (name = "Ax") or (null jresult_list) then nup_tac 
  else nup_tac THEN build_tactic_f (pop_jresult_list ()) in

  if (not assert_p) & ((name = "Alll") or (name = "Exr")) & 
     (v0_jprover_term_p term2) then

     if tree_p then
     (assert_p := true;
     JproverAssertR term1 return_tac)

     else %not tree_p - no need to call Run to build internals%
     (assert_p := true;
     JproverAssert term1 return_tac)
  	
  else % no need to do the assert %
  return_tac in

  build_tactic_f (pop_jresult_list ())
;;      
		 
let JproverT proof = 
  (jprovert_tactic proof jnamet_map false) proof;;


% for debugging %

let jprovert_list_to_ttt jresult =

  jresult_list := nil;
  Type_Function_alist := nil;
  Function_Index := 0;
  Type_Function_assertions := nil;
  letref assert_p = false in  % only need to assert the jp var at most one time %
  			     
  jresult_list := filter (\x. not (jprover_ignore_p x))
                         (map (\(name, t1,t2). (name, (nuprl_map_type t1), t2)) jresult);
  
  letrec build_ttt_f (name, term1, term2) = 
  
  let nup_tac = jnamet_map_ttt name term1 term2 in 

  let return_tac = 
   if tactic_split_p name then 
     let tac1 = build_ttt_f (pop_jresult_list ()) in
     let tac2 = build_ttt_f (pop_jresult_list ()) in
    
     make_icons_term tactic_tree_cons_op
		nup_tac
                (map_to_ilist id tactic_tree_cons_op [tac1; tac2])
   else

  % if at end of proof or at end of a subgoal from a split than don't iterate %
  if (name = "Ax") or (null jresult_list) then 
  make_icons_term tactic_tree_cons_op
		nup_tac
		(map_to_ilist id tactic_tree_cons_op [])
 
  else make_icons_term tactic_tree_cons_op
                nup_tac
		(map_to_ilist id tactic_tree_cons_op
		              [build_ttt_f (pop_jresult_list ())]) in

  if ((name = "Alll") or (name = "Exr")) & (v0_jprover_term_p term2) then

      make_icons_term tactic_tree_cons_op
	 (itext_cons_term (itext_term " JproverAssert ") 
                          (itext_cons_term (mk_prl_term term1) return_tac))
	 (map_to_ilist id tactic_tree_cons_op [])
	
  else % no need to do the assert %
  return_tac in

  build_ttt_f (pop_jresult_list ())
;;      

let JproverTttt p = 
  PushArgs [(`jprover`, (term_to_arg
			 (jprovert_list_to_ttt (dest_jprover_list
			       (jprover jprover_limit
					(map_to_ilist (\d. d)
						      icons_op
						      (append (get_jdeclst p) Type_Function_assertions))
					(jprover_map_type (concl p)))))))] p
;;			       


% call this from the refiner, with g. , after executing Assign on a proof%
let local_jtest p =
  jresult_list := nil;
  Type_Function_alist := nil;
  Function_Index := 0;
  Type_Function_assertions := nil;
  let jconcl = jprover_map_type (concl p) in
  let result = dest_jprover_list
  (jprover jprover_limit
	   (map_to_ilist id icons_op
			 (append (get_jdeclst p) Type_Function_assertions)) jconcl)
  in
  filter (\x. not (jprover_ignore_p x))
         (map (\(name, t1, t2). (name, (nuprl_map_type t1), t2)) result)
;;

%
*** End of type stuff
%


%
below is code used for testing jprover on entire theories, using
strategy of check bot facility
%
%let ref_refine_bottom_up poid =%
  
let ref_refine_jprover_aux failp envoid igoal tttt =
 let reff tgoal ttac contf =
  let failed,iinf_tree = 
       (if failp
           then (with_error_hook (\e. true,(refine_tree_fail envoid tgoal ttac e))
				 (\x. false, if ivoid_term_p ttac then failwith `no tactic`
                                             else (ref_refine tgoal ttac envoid x))
				 nil)
           else (false, (ref_refine tgoal ttac envoid nil))) in

  let subgoals = if failed then nil
                 else subgoals_of_iinf_tree iinf_tree in

    tty_print ("rrt rrvd" J (int_to_string (length subgoals)));

    if null subgoals then icons_term iinf_tree inil_term
    else (rrtbug := hd subgoals;
    let r = contf subgoals in
     if isr r then icons_term iinf_tree (map_to_ilist id icons_op (outr r))
     else 
       ( tty_print "rrt ttt"
       ; if not failp then fail
	  else icons_term iinf_tree (map_to_ilist id icons_op subgoals)
       ))
     
    in

    with_object_id 
      (\(). if failp
               then (apply_ttt_aux_cont reff (tttt_to_ttt tttt) igoal)
               else (apply_ttt_aux_cont reff (tttt_to_ttt tttt) igoal ? ivoid_term))
      (new_dummy_object_id ())  % used to provide an obid for proof cache to use in assoc list. %
;;

let ref_refine_jprover = ref_refine_jprover_aux false
;;
