%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%------------------------------------------------------------------------------+
|                                                                              |
|     File: nuprl5/lib/ml/standard/logic.ml                                    |
|                                                                              |
|     A rule system for first order logic                                      |
|                                                                              |
|     Author: Christoph Kreitz                                                 |
|     Date: 10/ 8/97                                                           |
|     Last Update: 09/29/98                                                    |
|                                                                              |
+------------------------------------------------------------------------------%

 let AXIOM              = Refine `because`  []   ;;

%------------------------------------------------------------------------------+
|    It would be even better to add the rule objects explicitly                |
|    This would give us the proof terms we want                                |
|    We can still use them as defined concepts and reduce to lambda            |
+------------------------------------------------------------------------------%

%------------------------------------------------------------------------------+
|    Predicates for testing the kind of junctor we have                        |
+------------------------------------------------------------------------------%

% these are defined in earlier files
let is_false_term t       = operator_id_of_term t = `false`          ;;
let is_true_term t        = operator_id_of_term t = `true`   	     ;;
let is_not_term t         = operator_id_of_term t = `not`    	     ;;
let is_and_term t         = operator_id_of_term t = `and`    	     ;;
let is_or_term t          = operator_id_of_term t = `or`     	     ;;
let is_iff_term t         = operator_id_of_term t = `iff`    	     ;;
let is_all_term t    	  = operator_id_of_term t = `all`    	     ;;
%
let is_ex_term t          = operator_id_of_term t = `exists`         ;;
let is_imp_term t    	  = operator_id_of_term t = `implies`	     ;;
let is_pmi_term t    	  = operator_id_of_term t = `rev_implies`    ;;

% -----------------------------------------------------------------------------+
| Basic inference rules for each junctor                                       |
+------------------------------------------------------------------------------+
| Decompose the conclusion      |     Decompose assumption i                   |
+-------------------------------+----------------------------------------------+
|                               |                                              |
|                               |        , false, '  T                      |
|                               |           by falseL i                        |
|                               |                                              |
|           A  B             |        , A  B, '  C                      |
|           by andR             |          by andL i                           |
|             A               |           , A, B, '  C                    |
|             B               |                                              |
|                               |                                              |
|           ST               |        , ST, '  C                        |
|           by orR1             |          by orL i                            |
|             S               |          , A, '  C                        |
|                               |          , B, '  C                        |
|           ST               |                                              |
|           by orR2             |                                              |
|             T               |                                              |
|                               |                                              |
|           AB              |        , AB, '  C                       |
|           by impR             |          by impL i                           |
|           , A  B            |          , AB, '  A                     |
|                               |          , ', B  C                        |
|                               |                                              |
|           A                |        , A, '  C                         |
|           by notR             |          by notL i                           |
|           , A  false        |          , A, '  A                       |
|                               |                                              |
|           BA              |        , BA, '  C                       |
|           by pmiR             |          by pmiL i                           |
|           , A  B            |          , BA, '  A                     |
|                               |          , ', B  C                        |
|                               |                                              |
|           BA              |        , BA, '  C                       |
|           by iffR             |          by iffL i                           |
|           , A  B            |          , BA, '  A                     |
|           , B  A            |          , BA, ', B  C                  |
|                               |                                              |
|                               |        , BA, '  C                       |
|                               |          by iffLr i                          |
|                               |          , BA, '  B                     |
|                               |          , BA, ', A  C                  |
|                               |                                              |
|           x:S.A            |        , x:S.A, '  C                     |
|           by allR             |          by allgL i a                         |
|           , z:S             |          , x:S.A, ', A[a/x]   C          |
|                               |                                              |
|           x:S.A            |        , x:S.A, '  C                     |
|            by exR a           |          by exL i                            |
|              A[a/x]         |          , x:S.A, u:S, A[u/x], '  C      |
|                               |                                              |
|                               |        , x:A, '  A                        |
|               		|	   by hypothesis i      	       |
|                               |       				       |
+-------------------------------+----------------------------------------------+
|                                                                              |
|           C                     (proof structuring)                        |
|            by cut A                                                          |
|              A                                                             |
|            ,A  C                                                           |
|                                                                              |
+------------------------------------------------------------------------------+
|                                                                              |
|           A  A                     (classical logic)                     |
|            by magic                                                          |
|                                                                              |
|           C                          (a variation of magic)                |
|            by ClassCases A                                                   |
|            , A  C                                                          |
|            , A  C                                                         |
|                                                                              |
+------------------------------------------------------------------------------%



% ---------------------------------------------------------------------------- %


let andR            = TryOnC (D 0                                   ) is_and_term
and orR1            = TryOnC (OrCD 1 THENW AXIOM               ) is_or_term 
and orR2            = TryOnC (OrCD 2 THENW AXIOM               ) is_or_term 
and impR            = TryOnC (D 0 THENW AXIOM                       ) is_imp_term 
and pmiR            = TryOnC (D 0 THENW AXIOM                       ) is_pmi_term
and iffR            = TryOnC (D 0 THEN D 0 THENW AXIOM              ) is_iff_term
and falseR          = TryOnC (D 0 THENW AXIOM                       ) is_false_term
and notR            = TryOnC (D 0 THENW AXIOM                       ) is_not_term
and allR            = TryOnC (D 0 THENW AXIOM                       ) is_all_term
and exR term        = TryOnC (With term (D 0) THENL [AXIOM;Id;AXIOM]) is_ex_term    
                    
and hypothesis hyp  = Refine `hypothesis`  [mk_int_arg hyp]  
                                                             
and andL   hyp      = TryOnH (D hyp)                              hyp is_and_term 
and orL    hyp      = TryOnH (D hyp)                              hyp is_or_term  
and falseL hyp      = TryOnH (D hyp)                              hyp is_false_term
and exL    hyp      = TryOnH (D hyp)                              hyp is_ex_term
and allL   hyp term = TryOnH (With term (ID hyp) THENW AXIOM)     hyp is_all_term
and impL   hyp      = TryOnH (ID hyp THENL [Id; Thin hyp])        hyp is_imp_term
and notL   hyp      = TryOnH (ID hyp)                             hyp is_not_term
and pmiL   hyp      = TryOnH (      Unfold `rev_implies` hyp 	  
                              THEN  ID hyp			  
                              THENL [ Fold `rev_implies` hyp
                                    ; Thin hyp]
                             )                                    hyp is_pmi_term
and iffL   hyp      = TryOnH (\p.(     Assert (type_of_hyp hyp p) 
       			          THENL [AXIOM; D (-1)] 
                                  THEN  Thin (-1) 		    
                                  THEN  D (-1)    
                             ) p )                                hyp is_iff_term
and iffLr  hyp      = TryOnH (\p.(    Assert (type_of_hyp hyp p)  
			          THENL [AXIOM; D (-1)] 
                                  THEN  Thin (-2) 		    
                                  THEN  D (-1)    
                             ) p )                                hyp is_iff_term
;;		     

% ---------------------------------------------------------------------------- %		     

let cut    term     = Assert term		     

and magic proof     = (let P,notP = dest_or (concl proof)             
                        in                                             
                            if alpha_equal_terms notP (mk_not_term P) then AXIOM proof   
                                                    else fail          
                       )                                               
                       ?  failwith `conclusion does not fit magic rule`
;; 		     
		     
let ClassCases term =  (      Assert (mk_or_term term (mk_not_term term)) 
                         THENL [magic; OnLastHyp orL]                     
                        )


;;		     
    		

% ---------------------------------------------------------------------------+
| Interactively all basic inference rules can be replaced by a uniform call  |
|                                                                            |
|  Decompose: operate on the conclusion                                      |
|  DH i     : decompose assumption (hypothesis) i                            |
|                                                                            |
| Use "Sel 1 Decompose" (or Sel 2) for disjunctions in the conclusion        |
| Use "With <term> Decompose" for existential quantifiers  in the conclusion |
| and "With <term> (DH i)" for universal quantifiers in an assumption        |
|                                                                            |
| The rules for  and  are composed rules and require two simple steps     |
+----------------------------------------------------------------------------%

let Decompose       = D 0   THENW AXIOM
and DH        hyp   = D hyp THENW AXIOM
;;
