%[
**********************************************************************
**********************************************************************
MONITOR.ML
**********************************************************************
**********************************************************************
Tactics for monitoring and reporting on execution of looping tactics
(such as Inclusion, RepEqCD and Auto), and for updating and applying
proof caching tactics.

Reference variables defined: [default]
  
  proof_cache_kind : tok (one of `none`,`basic`, or `thinning`) [`none`]

  tactic_monitor_sel : tok (one of `time`, `time-refine` or `none`) [`time`]
  do_monitoring : bool [`false`]
  call_hist : string list [[]] ;;

When using do_monitoring and interrupting a runaway Auto, the call_hist
needs to be manually reset to [].
]%


%[
**********************************************************************
Repeat with Wrapper tactic and addresses
**********************************************************************
addrRepeat is like Repeat T, except

1. Each call of body is surrounded by wrapper tactic WrapperT.

2. Proof address of repeats is maintained and fed to T via WrapperT.

     WrapperT : string list->(string list -> tactic)-> tactic  

   WrapperT can pass address directly to wrapped body, or is free
   to add something.

   Address in general may also include mnemonics for tactic calls.

3. Repeat is only carried out on descendent proof nodes for which function

      ena: proof -> bool

   returns true. Repeat is always enabled on original proof node AddrRepeat
   is called on.
]%


let AddrRepeat WrapperT ena addr T = 
  letrec Aux addr' p' =
      (WrapperT addr')
      (\ad.Try (LabProgress (T ad) 
           THEN_OnEach (\ps.map (ChildT ad) (number ps))))
      p'
  and ChildT addr' (i,p) = If ena (Aux ((int_to_string i J " ").addr')) Id 
  in  
    Aux addr
;;



%[
**********************************************************************
Call History
**********************************************************************
A call history is a mixture of a proof tree address and names of 
active invocations of tactics.
]%


% fake a dynamic variable %

letref call_hist = [""] ;;

let dyn_bind_call_hist val bodyf = 
  let oldval = call_hist in
  ( call_hist := val ;
    unwind_protect bodyf (\().call_hist := oldval)
  )
;;

let dyn_val_call_hist (():unit) = call_hist ;;


% printing histories %

letref display_cursor_at_left = false ;;

% NAP: This can be easily fixed %
%
let disp_nl (():unit) = 
  display_string (tok_to_string newline) ;
  display_cursor_at_left := true ;
  ()
;;

let disp_hist strs = 
  display_string (concatenate_strings (rev strs)) ;
  display_cursor_at_left := false ;
  ()
;;
%

% if cursor at left, put out inc before str. Then put out str and nl %

%
let disp_hist_inc_and_nl inc strs = 
  if display_cursor_at_left then
  ( disp_hist (inc . strs) ; disp_nl ())
  else
  ( disp_hist [inc] ; disp_nl ())
;;
%


%[
**********************************************************************
Monitoring Function Execution
**********************************************************************

  monitor : (* -> **) -> * -> ( ** + unit) # string

  monitor f a = inr(),str or inl(val),str 

  val is value of applying f to a.
  str is auxiliary information about execution.
  (e.g. time, # prim refinements, # of conses)
]%

let null_monitor f a = (inl (f a) : * + unit), "" ;;

let time_monitor f a = 
  let t1 = get_internal_run_time () in 
  let val = inl (f a) ? inr () in 
  let t2 = get_internal_run_time () in 
  let str = int_to_string ((t2-t1)/1000) J "ms"
  in
    val,str
;;

% counts refinements, both successes and failures 
  NB: #'s of refinements are not too reliable when using ML proof caching,
  since extra Fiats are counted...%
 
let time_refine_monitor f a = 
  let s1 = nrefine_succs in
  let f1 = nrefine_fails in
  let t1 = get_internal_run_time () in 
  let val = inl (f a) ? inr () in 
  let t2 = get_internal_run_time () in 
  let s2 = nrefine_succs in
  let f2 = nrefine_fails in
  let str = int_to_string ((t2-t1)/1000) J "ms "
            J int_to_string (s2-s1) J "s " 
            J int_to_string (f2-f1) J "f "
  in
    val,str
;;

letref tactic_monitor_sel = `time` ;;

let tactic_monitor f = 
  if tactic_monitor_sel = `time` then time_monitor f 
  if tactic_monitor_sel = `time-refine` then time_refine_monitor f
  else null_monitor f
;;
    

%[
**********************************************************************
Monitoring call of tactic.
**********************************************************************
MonitorCall <addr> p (<id>,T) ==> (ps,v)

Monitors call of tactic T on proof p.
Always case that evaluation behaves functionally like that of T p.
i.e.

T p ==> ps,v

If do_monitoring, then has side effect of printing out some information on 
the call of T. Specifically, 1 of:

<addr> <id> (n) <info> <nl>       T produces (n) subgoals
<addr> <id> (NP) <info> <nl>      T makes No Progress
<addr> <id> Fai <info> <nl>       T fails

where <info> is information returned by tactic_monitor function.

MonitorCall always passes a dynamic binding for the address addr to T.
]%


letref do_monitoring = false;;


let MonitorCall addr p (id,T) =
%  if not do_monitoring then% T p %
  else
  let format_val (ps,v) p = 
    let n = length ps in
    if not n = 1 then
      " (" J int_to_string n J ")"
    if equal_sequents p (hd ps) then
      " (NP)"
    else
      " (1)"
  in
  disp_hist (id . addr) ;
  let val,str = dyn_bind_call_hist addr (\().tactic_monitor T p) in
  if isl val then
    %% T succeeds %%
    (disp_hist_inc_and_nl (format_val (outl val) p J str)  addr ; outl val)
  else
    %% T fails %%
    (disp_hist_inc_and_nl ("Fai" J str) addr ; fail) %
;;


% try each T in turn till one succeeds %


let MonitorCalls id_T_prs addr p = 
    first_value (MonitorCall addr p) id_T_prs
;;


%[
**********************************************************************
Caching Using Lisp Proof Cache
**********************************************************************
This never worked properly, so virtually none of functions are used; 
Only function that is used is current_pcache.

Function defined below in ML need to updated if attempt is made to use them.

M> current_pcache;; - : (unit -> tok)
M> update_current_pcache;; - : (proof -> unit)
M> lookup_current_pcache;; - : (proof -> proof)
M> clear_pcache;; - : (tok -> unit)
M> with_pcache;; - : (tok -> (* -> **) -> * -> **)
M> make_pcache_rule;; - : (unit -> rule)
M> proof_of_pcache_rule;; - : (rule -> proof)


The pcache rule will only have a proof if it has been used in a succesful 
refinement.
Ie:
M> proof_of_pcache_rule (make_pcache_rule());;
EVALUATION failed     proof_of_pcache_rule: NoCache

M> pcache_stats ;;
- : (tok -> (int # int) list)

First int is number of hits on proof, second is size of proof.

M> count_proof_nodes ;;
- : (proof -> int list)

Currently, returns six items :

 - total number of proof nodes
 - number of tactic nodes
 - size of largest primitive proof tree
 - length of longest assumption list

Preceding does not look inside cached proofs;

 - number of cache rules
 - sum of size of proofs of cache rules. If the same proof is 
   referenced multiple times it's size is added in multiple times.
]%


%
BEGIN UNUSED CODE

let UpdatingPCache (T:tactic) p = 
  let ps,v = T p in
  if null ps then
  ( update_current_pcache (v ps) ; ps,v)
  else
    ps,v
;;

let ApplyPCache p = 
  let p' = lookup_current_pcache p in
  Graft p' p
;;

let pcache_rule = make_pcache_rule () ;;

let ApplyPCache2 p = refine pcache_rule p ;;
let ApplyPCache3 p = refine (make_pcache_rule ()) p ;;

let WithPCaching addr T p = 
  let concl_str = term_to_print_string (mk_prl_term (concl p)) 0 
  in
    MonitorCall (concl_str . addr) p ("Cach ",ApplyPCache )
    ? UpdatingPCache T p
   
;;
END UNUSED CODE
%

%[
**********************************************************************
Caching using ML Proof Cache 
**********************************************************************
The basic functions

  UpdatingMLPfCache 
  ApplyMLPfCache

are defined in wf-cache.ml
]%

% 
Define dynamic variable for passing cache name to refinement tactics 
Useful when refining from ML rather than proof editor.
%

letref pcache_name = `none` ;;

let dyn_bind_pcache_name val bodyf = 
  let oldval = pcache_name in
  ( pcache_name := val ;
    unwind_protect bodyf (\().pcache_name := oldval)
  )
;;

let dyn_val_pcache_name (():unit) = dummy_object_id ()%"none"% ;;


% NAP: fix this;; LAL proof caching %
%
let dyn_val_pcache_name (():unit) = pcache_name ;;
let thm_being_refined (():unit) = 
  current_pcache () ? dyn_val_pcache_name ()
;;
%

let thm_being_refined (():unit) = 
  current_pcache () ? dyn_val_pcache_name ()
  ;;
  
% modified so that doesn't check if proof already present prior to inserting in cache
  however only apparent method of first attempts to apply cache and thus it should not
  be present if being called.
%
let MLPfCachingWrapper id (addr:string list) T p = 
  UpdatingMLPfCache_aux false false (thm_being_refined ()) id (T addr) p
;;

let ApplyMLPfCache' id p = ApplyMLPfCache false (thm_being_refined ()) id p ;;


let ThinningMLPfCachingWrapper id (addr:string list) T p = 
  let thm = thm_being_refined () in
  let vs,no_prog = essential_vars_of_seq p in
  if no_prog then
    UpdatingMLPfCache true thm id (T addr) p
  else
  ( Complete (ThinIrrevHyps THEN UpdatingMLPfCache true thm id (T ("Th ".addr)))
    ORELSE UpdatingMLPfCache false thm id (T addr)
  ) p
;;

let ThinningApplyMLPfCache id p = 
  let thm = thm_being_refined () in
  let vs,no_prog = essential_vars_of_seq p in
  if no_prog then
    ApplyMLPfCache true thm id p
  else
  ( (ThinToVars vs THEN ApplyMLPfCache true thm id)
    ORELSE ApplyMLPfCache false thm id
  ) p
;;


%[
**********************************************************************
Master Proof Caching Functions
**********************************************************************
]%

% acceptable values are `none`, `basic`, and `thinning`. %
  
letref proof_cache_kind = `basic`;;


let PfCachingWrapper id addr T = 
  if proof_cache_kind = `basic` then
    MLPfCachingWrapper id addr T
  if proof_cache_kind = `thinning` then
    ThinningMLPfCachingWrapper id addr T
  else 
    T addr
;;

let ApplyPfCache id = 
  if proof_cache_kind = `basic` then
     ApplyMLPfCache' id
  if proof_cache_kind = `thinning` then
     ThinningApplyMLPfCache id
  else 
    FailWith `ApplyPfCache`
;;


%[
**********************************************************************
Repeat Function with Monitoring and Caching Options
**********************************************************************
]%

% handles inclusion goals appropriately %


let get_subtype_args p =
  (-1),list_to_pair (dest_simple_term_with_opid `subtype` (concl p))
  ?
  let T,v,() = dest_equal (concl p) in
  let decl_num = get_decl_num (dest_var v) p 
  in
    decl_num, (h decl_num p), T
;;

%
let format_seq p = 
 (let i,A,B = get_subtype_args p in
  if i = -1 then fail else
  ("[" J int_to_string i J "]" J
    term_to_print_string (mk_prl_term (mk_simple_term `subtype` [A;B])) 0 
  )
 ) ? term_to_print_string (mk_prl_term (concl p)) 0 
;;
%

% format_seq' only does formatting work if needed %

let format_seq' p = %if do_monitoring then format_seq p else% ""
;;

 
let AddrRepeatWrapper id addr T = 
    (\p. MonitorCall (format_seq' p . addr) p ("Cach ",ApplyPfCache id))
     ORELSE PfCachingWrapper id addr T
;;


let RepWith ena id id_T_prs p =
  (if do_monitoring then %disp_nl% () else ()) ;
  AddrRepeat 
    (AddrRepeatWrapper id)
    ena 
    ((id J " ") . dyn_val_call_hist ()) 
    (MonitorCalls id_T_prs) 
    p 
;;
