%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

% Decide if a term is a theory_let abstraction %
let is_let_term t =
    is_term `let` t;;

% Guard terms %
let is_nguard_term t =
    is_term `nguard` t;;

let is_xguard_term t =
    is_nguard_term t or is_guard_term t;;

% This tagger is used to guide a form of computation with the following properties:
|      1. Terms inside a guard are not evalutated
|      2. members are not unfolded
|      3. the argument to a "let" is evaluated first (eager)
|      4. the guard of a "decide" must evaluate to an "inl" or "inr"
|         before the branches of the decide are evaluated
|      5. the argument to "spread" must evaluate to a pair before
|         the body of the spread is evaluated
|      6. every other term is open to evaluation
|  The form of evaluation is somewhat eager, somewhat lazy, so it is called "sleepy."
|  The computation uses "ComputeUsing" because the rewrite package to to slow.
%
let tag_non_guard_sleepy let_flag t =
    % Apply function to second component of a pair %
    let pi2apply f (vs, t) = (vs, f t) in

    % Function f returns a pair: (flag, term)
    | Apply this function to a list, and return (flag', tlist)
    | where flag' is the conjunction of all the flags, and
    | tlist is the list of terms returned.
    %
    letrec compute_bterms f bterms =
        (if null bterms then
	     (true, [])
	 else
	     let vs, flag1, tag = pi2apply f (hd bterms) in
	     let flag2, nbterms = compute_bterms f (tl bterms) in
		 (flag1 & flag2, (vs, tag).nbterms))
    in

    % Recursively sweep up the term %
    letrec aux t =
        (if is_xguard_term t then
	     (false, t)
         else if is_function_term t or
                 is_member_term t or
                 is_equal_term t then
	     (let op, bterms = dest_term t in
	      let flag, nbterms = compute_bterms aux bterms in
		  (false, mk_term op nbterms))
	 else if let_flag & is_let_term t then
	     (let op, bterms = dest_term t in
	      let vs, hdterm = (hd bterms) in
		  if is_xguard_term hdterm then
		      (false, t)
		  else
		      (false, mk_term op ((vs, tag_term hdterm).(tl bterms))))
         else if is_beta_redex t or is_ab_term t then
	     (true, tag_term t)
	 else if is_decide_term t then
             (let op, bterms = dest_term t in
	      let vs, hdterm = (hd bterms) in
		  if is_inl_term hdterm or is_inr_term hdterm then
		      (true, tag_term t)
		  else if is_xguard_term hdterm then
		      (false, t)
		  else
		      (false, mk_term op ((vs, tag_term hdterm).(tl bterms))))
	 else if is_spread_term t then
	     (let op, bterms = dest_term t in
	      let vs, hdterm = (hd bterms) in
		  if is_pair_term hdterm then
		      (true, tag_term t)
		  else if is_xguard_term hdterm then
		      (false, t)
		  else
		      (false, mk_term op ((vs, tag_term hdterm).(tl bterms))))
         else
	     (let op, bterms = dest_term t in
	      let flag, nbterms = compute_bterms aux bterms in
		  if flag then
		      (true, tag_term t)
		  else
		      (false, mk_term op nbterms))) in

    % Sweep up the function, and discard the flag %
    let flag, tag = aux t in
	tag
;;

% Compute using the tagger.
|  Computation has two repeating phases:
|      1. Evaluate the arguments of "let"
|      2. Unfold the "let" and return to the first phase.
%
let ComputeGuard i =
    Repeat (Repeat (ComputeUsing (tag_non_guard_sleepy true) i) THEN
            ComputeUsing (tag_non_guard_sleepy false) i);;

%
| Compute using rewrite package
%
let ComputeGuardC =
    RepeatC (TagC (tag_non_guard_sleepy true)
	     ORTHENC TagC (tag_non_guard_sleepy false));;
