%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
*********************************************************************
*********************************************************************
SIMP-TACTICS.ML
*********************************************************************
*********************************************************************
Simplification functions.
]%

%[
*********************************************************************
Redex Strength 
*********************************************************************

Notion of redex `strength' and reduction `force'. 

Seems like a good idea to have different strength's assigned to 
redices. A redex is only contracted when a reduction conversion with sufficient
force is applied to it. 

Q. Should a linear order be sufficient? If so, can use integers.
Otherwise, want to use tokens or strings, and have way of specifying
order. Will usetokens for sake of generality.

Set up spine of numbers, then non-numeric forces used for less common
cases.

Suggested basic strengths:
  (not fully implemented)
  1.beta redices
  2.other primitive redices
  3.abstract redices recursive
  4.abstract redices non-recursive
  5.??
  6.module projection funs with coercion arguments.
    e.g. grp_op(add_grp_of_rng(r)) --> rng_plus(r)
  7.functions creating module elements from concrete parts.
  8.quasi-canonical redices
    (e.g. have discrete set constructors defined in sets_1 which normally
    shouldn't be computed away when they are arguments of set_car terms.)
  9.Never can be reduced.

users can invent new levels between, above and below these if necessary. 
e.g.  `3.5`
max and min should be greatest and least element respectively in
above (below) 
]%

letref red_str_spine = ``1 2 3 4 5 6 7 8 9`` ;;
letref max_red_strength = `9` ;;
letref min_red_strength = `1` ;;

let red_str_geq a b = 
  position a red_str_spine GE position b red_str_spine
  ? 
  failwith (`unrecognized reduction strength pair : ` ^ a ^ ` ` ^ b)
;;

let Force F T p = WithArgs [`force`,tok_to_arg F] T p ;;

% set default at maximum force to avoid existing proofs breaking %
 
let get_red_force p = get_tok_arg `force` p ? max_red_strength;;


%
Associate strengths with terms. Strengths used so far only in 
module-tactics file. This info should eventually annotate abstractions.
However, retrieval of abstraction attributes is currently very slow,
so I take this approach instead.

Use `reduction_strength' not `redex_strength', since these will sometimes
be annotating functions that will be arguments to redex terms, but won't
themselves be redices. 

`reduction_strength' = strength in the face of reduction force
%

letref reduction_strengths = [`variable`,`9`] : (tok # tok) list ;;

let note_reduction_strength opid strength = 
  reduction_strengths := 
    update_alist reduction_strengths opid strength 
  ; ()
;;

% default is to have weak terms to avoid existing proofs breaking  %

let get_reduction_strength opid = 
  apply_alist reduction_strengths opid ? min_red_strength
;;


%[
*********************************************************************
AbReduce for Decomposition
*********************************************************************

AbReduce
~~~~~~~~

Takes care of reducing both primitive redices, and applying user
supplied conversions.

AbReduce is intended to be a fast tactic that can be used frequently, so 
conversions that involve tactic justifications should not be added.

NB: The ForceReduce_alist here replaces the old AbReduce_alist.
]%

%
;;;;	
;;;;	Reference Environment consistency : 
;;;;	
;;;;	  - no cycles. 
;;;;	  foreach refenv, for all visible_statements refenv(stm) <_re refenv
;;;;	  <_re == prereq, or subset of some sort
;;;;	
;;;;	Define inductively : 
;;;;	  assume all prevs consistent then re consistent when :
;;;;	    - all new stms use an re which is member of closure(prev)
;;;;	    - current is not prev.
;;;;	    closure of prevs cand not include current since then prev wouldn't be consistent.
;;;;	    there can be no cycles amount the prevs.
;;;;	
;;;;	Procedures for testing consistency.
;;;;	  - layer res
;;;;	  - verify prf re's are members of prev closure.
;;;;	      * prf re is member of eariler layer may be sufficient?
;;;;	
;;;;	  forall s in stms_of(re_of(stm)).
;;;;		stm not in stms_of(re_of(s))
;;;;		stms_of(re_of(s)) is subset of stms_of(re_of(stm))
;;;;	        * stms_of(re_of(s)) can not be equal to stms_of(re_of(stm)) since
;;;;	          s member of stms_of(re_of(stm)) and s can not be member of stms_of(re_of(s))
;;;;	  
;;;;	
;;;;	Code may evaluate in scope of environment.
;;;;	  - thus code may depend on abs and lemmas referenced during eval.
;;;;	Code fragments may be assigned to reference environments
;;;;	  - thus consistency of reference environments must include re_of code objects too.
;;;;	
;;;;	make_fragment_available 
;;;;	 : oid {ref env} -> ref_state -> oid{code} -> *{fragment} -> ()
;;;;	
;;;;	declare_ref_state_fragment : ref_state -> oid{code} -> * fragment -> ()
;;;;
;;;;	object_id{re} # object_id{code} # refstate # *
;;;;	
;;;;	
;;;;	 at compile time if referenced function is not defined in an "accessible" object
;;;;	   - then fail
;;;;	 accessible : mentioned in ref_environment under code.
;;;;	
;;;;	
;;;;	

%
%
letref ForceReduce_alist = [`lambda`,\f.BetaC] : (tok # (tok -> convn)) list ;;

let add_ForceReduce_conv opid c = 
  ForceReduce_alist := 
    update_alist ForceReduce_alist opid c
  ;
  ()
;;

let add_AbReduce_conv opid c = add_ForceReduce_conv opid (\f.c) ;;
%

letref ForceReduce_alist_ref_state =
  new_list_alist_ref_state `ForceReduce_alist`
    (nil : (tok # ((tok -> convn) list)) list)
;;
update_ref_state_view
 (\(). ForceReduce_alist_ref_state)
 (ref_state_view_list_entry
    (\x. make_term (`ForceReduce_alist`,
		     [make_token_parameter (fst x); make_natural_parameter (length (snd x))]) []))
;;     

let ref_add_ForceReduce_alist_aux index edges items =
 ForceReduce_alist_ref_state
   := declare_ref_state_data_indirect `ForceReduce_alist` ForceReduce_alist_ref_state index items edges
;;

let ref_add_ForceReduce_alist index edges alist =
 declare_ref_state_index index `ForceReduce_alist` index;
 ref_add_ForceReduce_alist_aux index edges (map (\n,c. n,[c]) alist)
;;

let destruct_abreduce_ref_state () =
 let g, upds, name, uf, ug = destruct_ref_state ForceReduce_alist_ref_state
 in upds
;;

let ref_add_ForceReduce_additions items index edges =
 ForceReduce_alist_ref_state
   := ref_state_modify_state_aux (\data gedata. append gedata data)
         ForceReduce_alist_ref_state items index edges
 ; ()
;;

let ForceReduce_add_data oid data =
  ForceReduce_alist_ref_state 
    := ref_state_set_data ForceReduce_alist_ref_state
         [oid,  data]
;;

let AbReduce_adds oid data =
 reset_ref_environment_data oid;
 add_ref_environment_data oid `ForceReduce_alist` ForceReduce_add_data 
   (map (\opid,clist. opid, map (\c. \f.c) clist) data)
;;

let AbReduce_add oid data = AbReduce_adds oid (map (\op,c. op,[c]) data);;
let AbReduce_add_o oid data =
 AbReduce_add oid (map (\oid, c. name_of_abstraction oid, c) data);;

let AbReduce_adds_o oid data =
 AbReduce_adds oid (map (\oid, clist. name_of_abstraction oid, clist) data);;
   
let ForceReduce_add oid data =
 reset_ref_environment_data oid;
 add_ref_environment_data oid `ForceReduce_alist` ForceReduce_add_data (map (\op, c. op, [c]) data)
;;

 let ForceReduce_add_o oid data =
 ForceReduce_add oid (map (\oid, c. name_of_abstraction oid, c) data);;

				
let ForceReduce_additions_do_updates oid edges oids =
  %tty_print "ForceReduce_additions "; view_show_oids edges; %
  ForceReduce_alist_ref_state := ref_state_do_updates ForceReduce_alist_ref_state oid oids edges
; ()
;;

let undeclare_ForceReduce oid =
  (ForceReduce_alist_ref_state := ref_state_remove ForceReduce_alist_ref_state oid; ())
 ? ()
;;



letref find_ForceAbRedexC_list = (\t. fail) : (tok -> ((tok -> convn) list));;

let build_find_ForceAbRedexC_list uoid =
  find_ForceAbRedexC_list := if (isl uoid) then (\t. fail)
                       else apply_alist (ref_state_get ForceReduce_alist_ref_state (outr uoid))
  ; ()
;;

update_ref_environment_cache_hook `ForceReduce_alist` build_find_ForceAbRedexC_list ;;

%-----------
 following for compatability with old objects, calls should be removed.
  maybe change these to simply fail? or warn & no-op
 -----------%
let find_ForceAbRedexC x =
 hd (apply_alist (ref_state_get ForceReduce_alist_ref_state
				(current_ref_environment_index `ForceReduce_alist`))
		 x)
;; 
let ref_add_AbReduce_additions items =
 ref_add_ForceReduce_additions (map (\opid,c. opid, [(\f.c)]) items)
;;
%-----------%

let lookup_ForceReduce_alist () = 
  ref_state_get ForceReduce_alist_ref_state (current_ref_environment_index `ForceReduce_alist`)
;;


% provide ability to extend conversion for an id by combining another conversion using a tactical.
%
% item : tok # (tok -> convn) %
let ref_update_ForceReduce_additions conversional items index edges =
 ref_add_ForceReduce_additions
  (map (\(name, c).
          let cur = (hd (apply_alist (ref_state_get ForceReduce_alist_ref_state
						(ref_state_index_deref `ForceReduce_alist` index))
				 name))
                    ? (tty_print "ref_add_ForceReduce_additions lookup failed"; (\f.FailC)) in
	    name, [(\f. (conversional  (cur f) (c f)))])
       items)
  index edges
;;

%
let AbReduce_extend conversional aoid conv =
  let n = name_of_abstraction aoid in
  let oldc = find_ForceAbRedexC n in
    (\f. conversional conv (oldc f))
;;
let AbReduce_extend_ORELSEC = AbReduce_extend $ORELSEC;;
%
let AbReduce_extend_ORELSEC (aoid:object_id) (c:convn) = (\f. c);;


let ref_update_AbReduce_additions conversional items =
 ref_update_ForceReduce_additions conversional (map (\n,c. n,(\f.c)) items)
;;

let ref_ORELSEC_update_ForceReduce_additions = ref_update_ForceReduce_additions $ORELSEC;;
let ref_ORELSEC_update_AbReduce_additions = ref_update_AbReduce_additions $ORELSEC;;


% needs to be brought into reference environment.
  called by prog_module and prog_enum.
%
let add_AbReduce_conv opid c = fail `add_AbReduce_conv`;;
let add_ForceReduce_conv opid c = fail `add_ForceReduce_conv`;;

% 
1. Key off function name in applications
2. Recognize and unfold infix_aps
3. Recognize `reducible' abstractions.
    The latter are abstractions which hide a term which is reducible.
    Reducible abstractions can be nested.
    An abstraction is denoted `reducible' by adding a `Reducible' attribute
    to its definition.
%

letref reducible_abs = [] : tok list ;;

let add_reducible_ab ab = 
  reducible_abs := insert ab reducible_abs ; ();;

let is_reducible_ab t = 
  member (opid_of_term t) reducible_abs ;;

let ElimReducibleC e t =
  if is_reducible_ab t then
    UnfoldTopAbC e t
  else 
    failwith `elimReducibleC`
;;

letrec elim_reducibles t = 
  if is_reducible_ab t then
   elim_reducibles (unfold_ab t)
  else
    t
;;

let FirstForceAbRedexC strength (farcs : (tok -> convn) list) e t=
 letrec aux farcs =
  if (null farcs) then failwith `NoSuitableForceAbRedex`;
  ( ((hd farcs) strength) e t
  ? aux (tl farcs)) in
 aux farcs
;;
   

let ForceAbRedexC F e t' = 
  letrec get_fun_id t = 
    if is_term `apply` t then get_fun_id (hd (subterms t)) else opid_of_term t

  in let t = elim_reducibles t' 
  in let id = opid_of_term t 
  in
  let id',unfold_infix_ap = 
    if id = `infix_ap` then opid_of_term (hd (subterms t)), true
    if id = `apply` then get_fun_id t,false
    else id,false
  in
  let c = FirstForceAbRedexC F (find_ForceAbRedexC_list id')
          ? failwith `ForceAbRedexC: unrecognized term`
  in
  if unfold_infix_ap then
    (RepeatC ElimReducibleC 
     ANDTHENC UnfoldTopAbC 
     ANDTHENC (c ORELSEC AddrC [1] c)
    ) e t'
  else
    (RepeatC ElimReducibleC ANDTHENC c) e t'
;;

let ForceRedexC F e t = 
 (if red_str_geq F `2` then
    RedexC ORELSEC ForceAbRedexC F
  else
    ForceAbRedexC F
 ) e t
;;

let AbRedexC e t = ForceRedexC max_red_strength e t ;;

let ForceReduceC F = RepeatC (SweepDnC (ForceRedexC F)) ;;

let AbReduceC e t = ForceReduceC max_red_strength e t ;;

let AbReduce i p =
  let F = get_red_force p in
  Rewrite (ForceReduceC F) i p
;;

let FReduceC F = ForceReduceC F ;;

let FReduce F i p =
  Rewrite (ForceReduceC F) i p
;;


let is_ab_redex t = can (apply_conv (AbRedexC)) t ;;


let AbPrimRedexC = ForceRedexC `5` ;;


% Used occasionally, before reduction forces introduced %

let AbReduceIfC en = 
  RepeatC (SweepDnC (IfC en AbPrimRedexC)) ;;

let AbReduceIf en i p = 
  Rewrite (AbReduceIfC en) i p ;;
  

%[
*********************************************************************
AbReduce for Decomposition
*********************************************************************
]%

let AbReduceTopC = RepeatC (RedexC ORELSEC AbRedexC) ;;

let AbReduceTop i p = Rewrite AbReduceTopC i p ;;


let AddCarProperties i p =
  let mk_lemma_name t = 
    let name = opid_of_term (sub_term 1 t) ^ `_car_properties` in
    if is_statement name then name else failwith `AddCarProperties`
  in
    AddPropertiesAux mk_lemma_name i p
;;

let AbRedSetD i p = 
  let i' = get_pos_hyp_num i p in
  ( AddCarProperties i'
    THENM
    ( AbReduceTop i' THEN Repeat (UnfoldTopAb i')
      THEN BasicSetHD i'
      THEN Thin (i'+1)
    )
  ) p
;;

% ordering important here %
let AbRedD = (\i.Progress (AbReduceTop i) THEN PrimD i);;
%
update_D_additions `AbRedSetD` AbRedSetD ;;
update_D_additions `AbRedD` AbRedD ;;
%


%[
*********************************************************************
For Inclusion tactic 
*********************************************************************
]%

let AbReduceTypeForInc i = 
  if i = 0 then RW (AddrC [1] (RedexC ORELSEC AbRedexC)) 0
           else RW (RedexC ORELSEC AbRedexC) i
;;

%
Q. Is it useful to have complete reductions done on both
   types, even when it doesn't result in obviously equal terms?
 
   Note that AbSetSubtype takes care of reduction of outermost 
   structure, even when substructures are different.
%

let SubtypeAbReduce p = 
  let i,A,B = get_subtype_args p in 
  let red t = fst (AbReduceC null_env t) in
  let A' = red A and B' = red B in
  if soft_equal A' B' = `false` then
    failwith `SubtypeAbReduce`
  else
  let cls = if i = (-1) then [0] else [0;i] in
  (
    OnClauses cls AbReduce THEN TrivialSubtype 
  ) p
;;


%[
*********************************************************************
General Direct-Computation Simplification Functions
*********************************************************************
]%

% Scrap Ab prefixes since these are what are used nearly all the time 
  (however, need to retain for old libraries) 
%


let AbEval names i = RepUnfolds names i THEN AbReduce i ;;
let Eval = AbEval ;;

let EvalC names = RepeatC (UnfoldsC names) ANDTHENC AbReduceC ;;
let ForceEvalC str names = RepeatC (UnfoldsC names) ANDTHENC ForceReduceC str;;


% defs renamed and callers milled.
let PrimReduceC = ReduceC ;;
let ReduceC = AbReduceC ;;
%

let ReduceC = AbReduceC ;;

let Reduce = AbReduce ;;

%[
*********************************************************************
Conversion registry
*********************************************************************
If conversions are simply bound to ML identifiers there is no way for
tactics to find out anything about them or select them in some uniform
manner.

By checking conversions into a registry, we provide easy access to 
conversions by the PolyC conversion defined below.

To start, each entry in the registry has format:

  name:tok, kind:tok, LRC:convn, RLC:convn

As time goes on, we can think about recording more information.  

kinds:
 1. atomic  defined by single pattern e.g. by MacroC 
 2. local   Do rewriting at the top of terms they are applied to.
 3. global  Do rewriting everywhere.

The main difference is that it is appropriate to sweep conversions
of kinds 1 and 2, but not kind 3.
]%

letref conv_registry = [] : (tok # tok # convn # convn) list;;

let add_conv_to_reg nam kind C = 
  conv_registry := update_alist conv_registry nam (kind,C,FailC) 
  ; ();;

let add_conv_pr_to_reg nam kind (C,RevC) = 
  conv_registry := update_alist conv_registry nam (kind,C,RevC) 
  ; ();;


let lookup_in_conv_reg nam = 
  apply_alist conv_registry nam ;;


%[
****************************************************************************
Rewriting Using Control String
****************************************************************************
Idea is to use simple syntax for consisely specifying 
enumerated sets of rewrite rules for common cases.

Input string: 

list of tokens separated by spaces:

token = <lemma-name> | <lemma_name>> | <signed-number> | <signed-number><

A "<" to a token indicates that the lemma should be applied in a right-to-left
direction ,rather than left to right.

could extend to take , separated lists.

                                        id          args (tok list) # int list

<int>     rw with hyp <int>            `hyp-lr`         [],[<int>]    
<int><    rw r-to-l with hyp <int>     `hyp-rl`         [],[<int>]    

<name>    rw with lemma <name>         `lemma-lr`       [<name>],[]
<name><   rw r-to-l with lemma <name>  `lemma-rl`       [<name>],[]

r:<id>    reduce <id> only             `reduce-id`      [<id>],[]
r*        reduce all                   `reduce-*`       [],[]
r*<f>     reduce with force <f>        `reduce-force`   [<force>],[]

u:<id> unfold id                       `unfold`         [<id>],[]
f:<id> fold id                         `fold`           [<id>],[]
]%

%
parse_rw_control
~~~~~~~~~~~~~~~~

returns id and args as above.
%

let parse_rw_control_string str = 
  let parse_word word = 
    let chars = string_to_toks word in
    let charsa,rev_p = remove_suffix [`<`] chars,true ? chars,false in
    let chars',cln = 
      if member `.` charsa then
      ( let n = position `.` charsa in
        let cs,(().clnchars) = split (n - 1) charsa in
          cs,string_to_int (toks_to_string clnchars)
      )
      else
        charsa,-1
    in
    let head,rest = split 2 chars' ? chars',[] in
    if head = ``r :`` then
        `reduce-id`,[implode rest],[]
    if head = ``r *`` then
    ( if length chars' > 2 then
        `reduce-force`,[implode rest],[]
      else
        `reduce-*`,[],[]
    )
    if head = ``u :`` then
        `unfold`,[implode rest],[]
    if head = ``f :`` then
        `fold`,[implode rest],[]
    if subset chars' ``- 0 1 2 3 4 5 6 7 8 9`` then
    ( if rev_p then 
        `hyp-rl`,[],[string_to_int (toks_to_string chars');cln]
      else
        `hyp-lr`,[],[string_to_int (toks_to_string chars');cln]
    )
    if rev_p then
        `lemma-rl`,[implode chars'],[cln]
    else
        `lemma-lr`,[implode chars'],[cln]
  in
    map parse_word (string_to_words str)
;;

let AllPurposeC (id,toks,ints) = 
  if id = `hyp-lr`   then GenHypC (second ints) (hd ints)
  if id = `hyp-rl`   then RevGenHypC (second ints) (hd ints)
  if id = `lemma-lr` then GenLemmaC (hd ints) (hd toks)
  if id = `lemma-rl` then RevGenLemmaC (hd ints) (hd toks)

  if id = `reduce-id`    then IfC (\e t.opid_of_term t = hd toks) AbRedexC 
  if id = `reduce-*`     then AbRedexC
  if id = `reduce-force` then ForceRedexC (hd toks)

  if id = `unfold`       then UnfoldTopC (hd toks)
  if id = `fold`         then FoldTopC (hd toks)
  else
    failwith `AllPurposeC`
;;

let AllPurposeC_o id obid = 
  if id = `lemma-lr` then GenLemmaC_o (-1) obid
  if id = `lemma-rl` then RevGenLemmaC_o (-1) obid
  else
    failwith `AllPurposeC_o`
;;

let PolyC str = 
  let instrs = parse_rw_control_string str 
  in    
    FirstC (map AllPurposeC instrs)  
;;

% 
seems like alternating sweeping up and sweeping down will be best in general 
Here, we do progress check after every sweep.

Start with sweep down, so that any HypC's in PolyC
grab environment at top of clause being rewritten.
%

let MultiC str = 
  letref up = false in
  let SweepC c e t = 
    up := up xor true 
    ; if up then SweepDnC c e t else SweepUpC c e t 
  in
    RepeatC (SweepC (Repeat1C (PolyC str)))
;;

let MultiC_aux_o c = 
  letref up = false in
  let SweepC c e t = 
    up := up xor true 
    ; if up then SweepDnC c e t else SweepUpC c e t 
  in
    RepeatC (SweepC (Repeat1C c))
;;


let RewriteWithStr str = Rewrite (MultiC str) ;;

let RWW str = RewriteWithStr str ;;

let RWW_o obid = Rewrite (MultiC_aux_o (AllPurposeC_o `lemma-lr` obid));;
let RevRWW_o obid = Rewrite (MultiC_aux_o (AllPurposeC_o `lemma-rl` obid));;

let RewriteOnceWithStr str = Rewrite (HigherC (PolyC str)) ;;
let RWO str = RewriteOnceWithStr str ;;

let RWO_o obid = Rewrite (HigherC (AllPurposeC_o `lemma-lr` obid)) ;;
let RevRWO_o obid = Rewrite (HigherC (AllPurposeC_o `lemma-rl` obid)) ;;


%[
*****************************************************************************
Shrinking Terms.
*****************************************************************************
Only applies a conversion if it results in reduction of term size.
]%


let ShrinkC (c:convn) (e:env) t =
  let t',r,j = c e t 
  in 
  if term_size t > term_size t' then
     t',r,j
  else
    failwith `ShrinkC: term grew in size`
;;

let SReduceC =
  RepeatC (SweepUpC (ShrinkC RedexC))
;;

