%()
|    SqEq: perform equivalence reasoning for sqequal.
|    First make a graph that describes the relation,
|    then if the goal has the form "t1 ~ t2", find the shortest
|    path from t1 to t2.   If there is one, carry out the tactic,
|    otherwise fail.
]%

%
|  The graph is a set of terms, which represent the nodes
|  in the graph, and a list of edges, which are derived from the pairs in the
|  relation.
%

% Empty graph %
let SqEq_EmptyGraph = ([], []);;

% Add a node to the graph (set union) %
let SqEq_AddNode node (nodes, edges) =
    if member_p node nodes alpha_equal_terms then
	(nodes, edges)
    else
	(node.nodes, edges);;

% Add an edge to the graph %
let SqEq_AddEdge edge (nodes, edges) =
    (nodes, edge.edges);;

%
| An edge has a hyp number and a leading and following term.
| Scan the hyps for edges.
%
let SqEq_ScanHyps graph p =
    let i = num_hyps p in
    letrec f j =
	if j > i then
	    graph
	else
	    let next = f (j + 1) in
	    let hyp = h j p in
		if is_sqequal_term hyp then
		    let t1, t2 = dest_sqequal hyp in
			(SqEq_AddEdge (t1, t2, j)
			 (SqEq_AddNode t1
			  (SqEq_AddNode t2 next)))
		else
		    next
    in
	f 1;;

%
| Find the index of a term in the list of nodes.
%
let SqEq_NodeIndex nodes term =
    letrec f nodes i =
        if null nodes then
	    -1
	else if alpha_equal_terms term (hd nodes) then
	    i
	else
	    f (tl nodes) (i + 1)
    in
	f nodes 0;;

%
| Convert the graph so that the edges use natural numbers,
| rather than terms, for efficiency.
%
letrec SqEq_Convert (nodes, edges) =
    if null edges then
	(nodes, [])
    else
	let (t1, t2, i) = hd edges in
        let (n, e) = SqEq_Convert (nodes, tl edges) in
	    (n, (SqEq_NodeIndex n t1, SqEq_NodeIndex n t2, i).e);;

%
| Given a starting point, and an ending point, find the shortest path in the graph.
| The graph is naturally directed.  However, we can use symmetry to product a path
| taking an edge backwards.
|
| Use a breadth-first-search.  We maintain a set of nodes
| that we have paths to.  As long as we can make progress, we choose an edge
| between a node in the set, and a node outside the set.  We add the edge,
| then update paths accordingly.  The *path* to a node is a list of edges,
| where we maintain only the hypothesis number, and whether the
| edge is taken forward (true), or backward (false).
|
| The frontier is a list of tuples: each contains a
|    1. node
|    2. distance to the node
|    3. Path to the node, as a of hyp numbers and directions.
%
let SqEq_ShortestPath t1 t2 (nodes, edges) =
    let VoidEdge = (0, 0, -1) in
    let start = SqEq_NodeIndex nodes t1 in
    let finish = SqEq_NodeIndex nodes t2 in
    letrec within frontier node =
	% See if a node is in the frontier %
	if null frontier then
	    false
	else
	    let (n, distance, path) = hd frontier in
		if node = n then
		    true
		else
		    within (tl frontier) node
    in
    letrec get_path frontier node =
	if null frontier then
	    failwith `get_path: no such node in frontier`
	else
	    let (n, distance, path) = hd frontier in
		if node = n then
		    (distance, path)
		else
		    get_path (tl frontier) node
    in
    letrec search frontier =
        % Find the best edge crossing the frontier %
	letrec fedge edges best newedge =
	    if null edges then
		(best, newedge)
	    else
		let (t1, t2, i) = hd edges in
		let w = within frontier in
		    if (w t1) & not (w t2) then
			let distance, path = get_path frontier t1 in
			    if best = (-1) or distance < best then
				fedge (tl edges) distance (t2, distance + 1, (i, true).path)
			    else
				fedge (tl edges) best newedge
		    else if (w t2) & not (w t1) then
			let distance, path = get_path frontier t2 in
			    if best = (-1) or distance < best then
				fedge (tl edges) distance (t1, distance + 1, (i, false).path)
			    else
				fedge (tl edges) best newedge
		    else
			fedge (tl edges) best newedge
	in
	let distance, newedge = fedge edges (-1) (0, 0, []) in
	    if distance = (-1) then
		frontier
	    else
		search (newedge.frontier)
    in
	let frontier = search [(start, 0, [])] in
	    if within frontier finish then
		let distance, path = get_path frontier finish in
		    path
	    else
		[];;


%
| Prove a reflexive goal.
%
let SqEq_Reflexive =
    Refine `sqequalReflexivity` [];;

%
| Refine a goal by symmetry.
%
let SqEq_SymmetryCD p =
    let v = new_proof_var `z' p in
    let goal = concl p in
    let a, b = dest_sqequal goal in
	(Refine `sqequalSubstitution`
	     [mk_term_arg (mk_sqequal_term b a);
	      mk_var_arg v;
	      mk_term_arg (mk_sqequal_term a (mk_var_term v))]
         THEN_OnLast SqEq_Reflexive) p;;
%
| Given a sqequal hypothesis, reverse it using symmetry.
%
let SqEq_SymmetryHD i p =
    let v = new_proof_var `z' p in
    let hyp = h i p in
    let t1, t2 = dest_sqequal hyp in
        (AssertAtHyp i (mk_sqequal_term t2 t1)
         THENL [Refine `sqequalSubstitution` [mk_term_arg hyp;
                                              mk_var_arg v;
                                              mk_term_arg (mk_sqequal_term t2 (mk_var_term v))]
                THENL [NthHyp i; SqEq_Reflexive];
                Thin (i + 1)]) p;;

%
| Refine by symmetry.
%
let SqEq_Symmetry i =
    if i = 0 then
	SqEq_SymmetryCD
    else
	SqEq_SymmetryHD i;;
%
| Here we are given a goal of the form 'a ~ b',
| and a path from a to b.  Peform the refinement to
| prove 'a ~ b' successive substitution.
%
letrec SqEq_Trans path p =
    if null path then
	SqEq_Reflexive p
    else
	let v = new_proof_var `z' p in
        let goal = concl p in
        let a, b = dest_sqequal goal in
        let i, dir = hd path in
        let hyp = h i p in
        let t1, t2 = dest_sqequal hyp in
	    (Refine `sqequalSubstitution`
	         [mk_term_arg (if dir then
				   mk_sqequal_term t2 t1
			       else
				   hyp);
		  mk_var_arg v;
		  mk_term_arg (mk_sqequal_term a (mk_var_term v))]
             THENL [if dir then
			Refine `sqequalSubstitution`
			    [mk_term_arg hyp;
			     mk_var_arg v;
			     mk_term_arg (mk_sqequal_term t2 (mk_var_term v))]
                        THENL [NthHyp i; SqEq_Reflexive]
		    else
			NthHyp i;
		    SqEq_Trans (tl path)]) p;;

%
| Perform the full-blown sqequal equivalence reasoning.
| Construct the graph, and then apply a path if one exists.
%
let SqEq p =
    % Conclusion must be sqequal term %
    let goal = concl p in
    let t1, t2 = dest_sqequal goal in
    let graph = SqEq_Convert ((SqEq_ScanHyps
			       (SqEq_AddNode t1
				(SqEq_AddNode t2
				 SqEq_EmptyGraph))) p)
    in
    let path = SqEq_ShortestPath t1 t2 graph in
	(if null path & not (alpha_equal_terms t1 t2) then
	     FailWith `SqEq: not provable by equivalence reasoning`
	 else
	     SqEq_Trans path) p;;

%
| Define tools for assisting with equality reasoning.
| These convert between sqequal and equal,
| and decompose goals.
%

% Given a goal of the form 'a ~ b', replace it with a goal 'a = b in Z' %
let SqEqualInt =
    Refine `sqequalInt` [];;

let SqEqualAtom =
    Refine `sqequalAtom` [];;

let SqEqualEqual eq p =
    let le_arg, WFTac = mk_le_arg_and_wf_tac p eq in
	(Refine `sqequalEquality` [le_arg; mk_term_arg eq]
	 THENL [WFTac; Id]) p;;

let SqEqualUnit p =
    let zero = mk_natural_number_term 0 in
	(SqEqualEqual (mk_equal_term int_term zero zero)
	 THENL [Fold `member` 0 THEN Fold `unit` 0;
		FoldAtAddr `member` [1] 0 THEN FoldAtAddr `unit` [1] 0]) p;;

%
| Generalize this to any type that has canonical forms.
| If a type is canonical, then there should be a theorem in
| the library with the name "type_canon(_?)".  This theorem
| should state that the type has canonical forms; that is, it
| should be the lemma:
|    forall x1, ..., forall xn. Canonical(type[x1, ..., xn])
| We can backchain thorugh this lemma to reduce the sqequal
| to a real equality.
%
let unit_term = mk_simple_term `unit` [];;

% Get the names of the canonical lemmas %
let get_sq_lemma_names opid =
   current_visible_statements_with_prefix (concatenate_strings [(tok_to_string opid);  "_sq"])
%  opid_alist_names opid `sq`
%;;

%
| If no special case applies, look for a lemma.
%
let SqEqual p =
    let T = get_term_arg `t1` p ?
	let a, b = dest_sqequal (concl p) in
	    get_type p a ? get_type p b
    in
	(if alpha_equal_terms T int_term then
	     SqEqualInt
         else if alpha_equal_terms T atom_term then
	     SqEqualAtom
         else if alpha_equal_terms T unit_term then
	     SqEqualUnit
         else if is_equal_term T then
	     SqEqualEqual T
         else 
	     % Look for the canonical lemma %
	     let opid = opid_of_term T in
	     let names = get_sq_lemma_names opid in
	     let a, b = dest_sqequal (concl p) in
	     letrec TryEach names =
	         if null names then
		     Id
		 else
		     (InstLemma (hd names) [] THEN Trivial)
		     ORELSE TryEach (tl names)
             in
		 Assert (mk_equal_term T a b)
		 THENL [AddHiddenLabel `main`;
			Assert (mk_simple_term `sq_type` [T])
			THENL [TryEach names THEN AddHiddenLabel `aux`;
			       BackThruHyp (-1) THEN Trivial]]) p;;

%
| It is possible to prove 'a ~ b' if a = b, or by decompoing
| a and b into their consitutent parts and then proving
| the subgoals.
|
| We add this line of reasoning into EqCD.
%
let SqEqCD p =
    let goal = concl p in
    let a, b = dest_sqequal goal in
	if alpha_equal_terms a b then
	    Refine `sqequalReflexivity` [] p
	else
	    Refine `sqequal` [] p;;

%
| Now we provide substitution using sqequal. 
| This will be an addition to the normal Subst tactic.
| Normally, Subst is used as follows:
|     Subst 'x = y in T' i
| and three subgoals are generated:
|    The equality goal: 'x = y in T'
|    The new goal by the subtitution performed
|    A functionality subgoal, to show that the
|    ther being substiuted is functional over the type
|    of substitution.
|
| In sqequal substitution, the functionality subgoal is not generated.
| In the Subst tactic, we key off of the type of term being substituted
| and call distinct subtactics for 'equal' and 'sqequal'.
%

%
| Basic substitution performed in the goal.
|
| ... |- C[t]
|
| By BasicSqSubstC t ~ t'  z.C[z]
|
| `equality` ... |- t ~ t'
| `main`     ... |- C[t']
%
let BasicSqSubstC A_sim_B z Cz p =
    let z' = maybe_new_proof_var z p in
    let Cz' = z = z' => Cz | subst [z, mvt z'] Cz in
        (Refine `sqequalSubstitution`
	     [mk_term_arg A_sim_B;
	      mk_bterm_arg [z'] Cz']
         THENL [AddHiddenLabel `equality`; Id]) p;;

%
| Basic substitution performed in a Hyp.
|
| H, #i: x: H[t], J |- C
|
| By BasicSqSubstH i t ~ t' z.C[z]
|
| `equality` H, #i: x: H[t], J |- t ~ t'
| `main`     H, #i: x: H[t'], J |- C
%
let BasicSqSubstH i A_sim_B z Cz p =
    let z' = maybe_new_proof_var z p in
    let Cz' = z = z' => Cz | subst [z, mvt z'] Cz in
        (Refine `sqequalHypSubstitution`
	     [mk_int_arg i;
	      mk_term_arg A_sim_B;
	      mk_bterm_arg [z'] Cz']
         THENL [AddHiddenLabel `equality`; Id]) p;;

%
| Substitute part of the conclusion.
|
| >> C[t1] 
|
|   BY SqSubstInConcl 't1 ~ t2'
|
| `equality` >> t1 ~ t2
| `main`     >> C[t2]
%
let SqSubstInConcl t1_sim_t2 p =
    let C_of_t1 = concl p in
    let t1, t2 = dest_sqequal t1_sim_t2 in
    let z = maybe_new_proof_var `z' p in
    let C_of_z = replace_subterm t1 (mk_var_term z) C_of_t1 
    in
	BasicSqSubstC t1_sim_t2 z C_of_z p;;

%
| Substitute part of a hypothesis.
| This is a little more complicated because the refinement
| works only on the conclusion.
|
|  ... #i: x:H[t1] ... >> C 
|
|  BY SqSubstInHyp 't1 ~ t2' i
|
| `equality` ... #i: x:H[t1] ... >> t1 ~ t2
| `main`     ... #i: x:H[t2] ... >> C 
|
| Will fail if variable declared in hyp i is used in later hyps or in concl.
%
let SqSubstInHyp t1_sim_t2 i p =
    % ... i:H[t1] ... >> C %
    let i' = get_pos_hyp_num i p in
    let H_of_t1 = h i' p in
    let t1, t2 = dest_sqequal t1_sim_t2 in
    let z = maybe_new_proof_var `z' p in
    let H_of_z = replace_subterm t1 (mk_var_term z) H_of_t1
    in
	BasicSqSubstH i' t1_sim_t2 z H_of_z p;;

%
| Generally, do substitution of t1 for t2 in clause i.
%
let SqSubst t1_sim_t2 i =
    if i = 0 then
	SqSubstInConcl t1_sim_t2
    else
	SqSubstInHyp t1_sim_t2 i;;

%
| Substitute part of the conclusion that is at a particular address.
|
| >> C[t1] 
|
|   BY SqSubstInConclAtAddr addr 't2'
|
| `equality` >> t1 ~ t2
| `main`     >> C[t2]
%
let SqSubstInConclAtAddr addr t2 p =
    let C_of_t1 = concl p in
    let t1 = get_addressed_subterm addr C_of_t1 in
    let t1_sim_t2 = mk_sqequal_term t1 t2 in
    let z = maybe_new_var `z' (all_vars C_of_t1) in
    let C_of_z = apply_to_addressed_subterm (\x. mk_var_term z) addr C_of_t1 in
	BasicSqSubstC t1_sim_t2 z C_of_z p;;

%
| Substitute part of a hypothesis.
|
|  ... #i: x:H[t1] ... >> C 
|
|  BY SqSubstInHypAtAddr addr 't2' i
|
| `equality` ... #i: x:H[t1] ... >> t1 ~ t2
| `main`     ... #i: x:H[t2] ... >> C 
|
| Will fail if variable declared in hyp i is used in later hyps or in concl.
%
let SqSubstInHypAtAddr addr t2 i p =
    % ... i:H[t1] ... >> C %
    let i' = get_pos_hyp_num i p in
    let H_of_t1 = h i' p in
    let t1 = get_addressed_subterm addr H_of_t1 in
    let t1_sim_t2 = mk_sqequal_term t1 t2 in
    let z = maybe_new_var `z' (all_vars H_of_t1) in
    let H_of_z = apply_to_addressed_subterm (\x. mk_var_term z) addr H_of_t1 in
	BasicSqSubstH i' t1_sim_t2 z H_of_z p;;

%
| General substitution at an address.
%
let SqSubstAtAddr addr t2 i =
    if i = 0 then
	SqSubstInConclAtAddr addr t2
    else
	SqSubstInHypAtAddr addr t2 i;;

%
| Execute a sqequal operation with a tagged term.
| The tagged term without tags is the term we want to rewrite
| to.  The tags identify places where substitution occurs.
| This tactic is declared (and used) in compute-tactics.ml.
%
let SqSubstWithTaggedTerm tagged_term i =
    % Make a tactic to perform the substitution %
    letrec inspect_term addr term tac =
        if is_tag_term term then
	    let (), substterm = dest_tag term
	    in
		tac THEN_OnLast SqSubstAtAddr addr substterm i
	else
	    let (), bterms = dest_term term in
		inspect_bterms addr bterms tac 1
    and inspect_bterms addr bterms tac j =
	if null bterms then
	    tac
	else
	    let ((), hd). tl = bterms in
            let tac' = inspect_term (addr @ [j]) hd tac in
		inspect_bterms addr tl tac' (j + 1)
    in
	inspect_term [] tagged_term Id;;

%
| Delayed binding.
%
Ref_SqEqSymmetry := SqEq_SymmetryCD;;

Ref_SqSubstWithTaggedTerm := SqSubstWithTaggedTerm;;
