%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%
***************************************************************************
***************************************************************************
TACTICALS.ML
***************************************************************************
***************************************************************************
The type of tactics is given by:

lettype tactic = (proof -> ((proof list) # ((proof list) -> proof)));;

This definition can be found in type-inc.ml
%

%
***************************************************************************
Simple Tacticals 
***************************************************************************
CAPITALS are used to designate infix operators.

%

% type of tactics defined in sequent.ml %



let (Id : tactic) p =  [p],hd;;
let (Fail: tactic) p = failwith `Fail` ;;
let (Abort: string -> tactic) m p = abort m; fail;;

let If (predicate: proof -> bool) (T1:tactic) (T2:tactic) p =
  if predicate p then T1 p else T2 p
;;

ml_curried_infix `ORELSE` ;;
let $ORELSE (T1:tactic) (T2:tactic) p = T1 p ? T2 p ;;
 


let Try T = T ORELSE Id ;;


let Complete (T:tactic) p =
  let ps,v = T p in
  if null ps then ps,v else failwith `Complete`
;;

let Progress (tac:tactic) (p : proof) =
  let ps,v = tac p in
    if length ps = 1 
      then
	let p' = (hd ps) in
          if equal_sequents p' p
	  then failwith `Progress`
          else ps,v
    else ps,v
;;


let Collapse T p =
 let pl,v = T p in
 let pp = v pl in
 let pp', pl', m = frontier_and_map pp in
  if null m
     then frontier pp'
     else refine (make_permute_proof_rule pp' m) p
;;


%
***************************************************************************
Sequencing Tacticals 
***************************************************************************
%


% 
In future might want to rename these  "ANDTHEN..." tactics, to contrast
with the ORTHEN tactic.
%


ml_curried_infix `THEN_OnEach` ;;
ml_curried_infix `THENL` ;;
ml_curried_infix `THEN` ;;
ml_curried_infix `CollapseTHEN` ;;
ml_curried_infix `THEN_OnFirstL` ;;
ml_curried_infix `THEN_OnFirst` ;;
ml_curried_infix `THEN_OnLastL` ;;
ml_curried_infix `THEN_OnLast` ;;
ml_curried_infix `THEN_OnSameConcl` ;;

% Put the validation juggling into one tactical %

letrec mapshape nl fl l =  
  if null nl then nil
    else 
      (let m,l = split (hd nl) l in 
        (hd fl)m . mapshape(tl nl)(tl fl)l) 
;;

let $THEN_OnEach (T: tactic) (F : proof list -> tactic list) (p:proof) =

  let pl,v = T p  in
  let Ts = F pl in
  if not length Ts = length pl then
    failwith `THEN_OnEach: Wrong number of tactics`
  else
  let pll,vl = unzip (map2 ap Ts pl)
  in
    flatten pll,  (v o mapshape (map length pll) vl)
;;

let $THEN T1 T2 = T1 THEN_OnEach (\ps.replicate T2 (length ps)) ;;
let $THENL T TList = T THEN_OnEach (\ps.TList) ;;

let THEN_f (f: int -> int -> *) T1 (T2: * -> tactic)  = 
   T1 THEN_OnEach (\ps. let l = length ps in
   			 map (\index. T2 (f l index)) (upto 1 (length ps)));;


let $CollapseTHEN T1 T2 p =
  let pl,v = Collapse T1 p in
  let pll,vl = unzip (map T2 pl)
  in
    flatten pll,  (v o mapshape (map length pll) vl)
;;

% 
Like THEN_OnEach except hides subgoals p  for which f p = false. i.e. f 
selects out the active subgoals.
%

let ThenOnEachIf 
  (T:tactic) (f:proof->bool) (F :proof list -> tactic list) (p:proof) =

  let F' ps =
    let active_ps = filter f ps in
    let active_Ts = F active_ps  
    in
    ( map_with_repair
        (\p.if f p then fail else Id) 
        active_Ts
        ps 
      ?
      failwith `ThenOnEachIf: wrong number of tactics`
    )
  in
    (T THEN_OnEach F') p
;;


% Select out single or groups of subgoals by position. %

let $THEN_OnFirstL (T:tactic) (Ts:tactic list) =
    T 
    THEN_OnEach 
    (\ps. Ts @ replicate Id (length ps - length Ts))
;;

let $THEN_OnFirst T1 T2 =  T1 THEN_OnFirstL [T2]
;;

let $THEN_OnLastL (T:tactic) (Ts:tactic list) =
    T 
    THEN_OnEach 
    (\ps. replicate Id (length ps - length Ts) @ Ts)
;;

let $THEN_OnLast T1 T2 =  T1 THEN_OnLastL [T2]
;;

let $THEN_OnSameConcl T1 T2 p = 
  let c = concl p in
  T1 THEN If (\p'.c = concl p') T2 Id 
;;

% old definition of FilterThen. If we want to revive this, it
should be rewritten with THEN_OnEach %
%
let FilterThen (T: tactic) (P: proof->bool) (Tlist:tactic list) (p:proof) =
 letref l = Tlist in
  let pl,v = T p  in
  let pll,vl = 
    split (map (\p. if not P p then Id p
                    if null l then failwith `FilterThen` 
                    else (let T = hd l in l := tl l; T p))
               pl) in
  if not null l then failwith `FilterThen`
  else flat pll,  (v o mapshape (map length pll) vl)
;;
%

% prevent Repeat from infinite looping %
letref repeat_limit = 256;;

let RepeatForAtMost_auxlog n T T2 p =
 letrec power_of_2 i = if i = 0 then 1 else 2 * (power_of_2 (i-1)) in
 let f n l index = if l = 1 then n-1 else ((n - 1) / (power_of_2 index)) in
 letrec aux n p =
  if n = 0 then T p
  else 
  ((THEN_f (f n) (Progress T2) aux) ORELSE Id) p in
  aux n p
;;

let RepeatForAtMost_aux msg n T T2 p =
 if n < 0 then failwith `RepeatForAtMost limit must be greater than zero`;
 letref limit = n in
 letrec aux p =
  if limit = 0 then (tty_print ( "Repeat limit exceeded["
			       J (int_to_string n) J "] :" J msg);
		    T p)
  else 
  (limit := limit - 1;
  (((Progress T2) THEN aux) ORELSE Id) p) in
  aux p
;;

let RepeatForAtMost m n = RepeatForAtMost_aux m n Id;;
let RepeatForAtMostFail m n = RepeatForAtMost_aux m n Fail;;
let RepeatForAtMostAbort msg n =
  RepeatForAtMost_aux msg n
   (Abort ("Repeat limit exceeded[" J (int_to_string n) J "] : " J msg));;
let Repeat = RepeatForAtMostAbort "Repeat" repeat_limit;;

letrec RepeatFor n T p =
  if n = 0 then Id p
  else
  (T THEN RepeatFor (n-1) T) p
;;

ml_curried_infix `ORTHEN` ;;
let $ORTHEN T1 T2 = (T1 THEN Try T2) ORELSE T2 ;;

let First (Ts:tactic list) p =
  first_fun Ts p
;;

%
***************************************************************************
Conditional Tacticals
***************************************************************************
%

let IfOnConcl pred (T1:tactic) T2 p =
  if pred (concl p) then
    T1 p
  else 
   T2 p
;;

let IfOnHyp pred (T1:int->tactic) T2 i p =
  if pred (h i p) then
    T1 i p
  else 
   T2 i p
;;

let IfOnClause = IfOnHyp ;;

let IfThen pred T =
  If pred T Id ;;

let IfThenOnConcl pred T = 
  IfOnConcl pred T Fail
;;

let IfThenOnHyp pred T i = 
  IfOnHyp pred T (\i.Fail) i
;;

let IfThenOnClause = IfThenOnHyp 
;;

let RepeatIf pred T =
  letrec Rep p =
    Try (If pred (Progress T THEN Rep) Id) p
  in
    Rep
;;

let While = RepeatIf ;;

let RepeatUntil pred T =
  RepeatIf (\p.not pred p) T
;;

let SeqOnSameConcl T_list p =

  let c = concl p   in

  letrec Aux T_list p =
    ( if not null T_list & (alpha_equal_terms (concl p) c)
      then (hd T_list THEN Aux (tl T_list)) p
      else Id p
    )           in

  Aux T_list p
;;

%
***************************************************************************
Label tactics
***************************************************************************
Or Death To THENL!

See sequent.ml for basic functions to set and examine labels.

During label matching, 
`main` is a wildcard matching any label in main_labels.
`aux`  is a wildcard matching any label not in main_labels.
`predicate` is a wildcard matching any label in predicate_labels.

All other labels must match exactly.

The adjective "Hidden" is no longer relevant and should be replaced at 
some stage.
 
The predicate class is thought of as a subdivision of the `aux` class,
so its intersection with the main class should be null.
%

letref main_labels = 
  ``main upcase downcase basecase truecase falsecase subterm`` ;;

letref predicate_labels = 
  `set predicate`
  .`rewrite subgoal`
  .``assertion antecedent ``
;;

%
Stop labels are used on subgoals which tactics (esp Auto) should stop on. 
For example, failed inclusion subgoals.
%

letref stop_labels = ([]: tok list) ;;

let update_stop_labels lab = 
  (stop_labels := insert lab stop_labels) ; ()
;;

let label_match pat inst =
	%mlbreak `match`;%
  if pat = `main` then member inst main_labels
  if pat = `predicate` then member inst predicate_labels
  if pat = `aux` then not member inst main_labels
  if pat = `any` then true
  else
    pat = inst
;;

  

let AddHiddenLabel label p =
   Id (add_label_to_proof label (clear_number_of_proof p))
;;

let RemoveHiddenLabel p =
  AddHiddenLabel `main` p
;;

let AddHiddenLabelAndNumber label n p =
  Id (add_number_to_proof n (add_label_to_proof label p))
;;


let KeepingLabel T p = 
  (T THEN 
   if is_numbered_proof p then
     AddHiddenLabelAndNumber
       (label_of_proof p) (number_of_proof p)
   else
     AddHiddenLabel (label_of_proof p)
  ) p 
;;


let IfLabL tok_tactic_list p =
 (let label = label_of_proof p in
    snd (find (\lab,().label_match lab label) tok_tactic_list)
    ? 
    Id
 ) p
;;

let IfLab label (T1:tactic) T2 p =
  let proof_label = label_of_proof p in
  if 
    label_match label proof_label 
  then 
    T1 p 
  else 
    T2 p
;;

% 
Could generalize this easily to work on arbitrary predicates on goals.
Do a separate version: would be less efficient...

Normally the ith tactic for a given class is applied to the ith subgoal
of that class in left to right order. However if a subgoal has a number label,
say n, then we run the nth tactic of the appropriate class on that subgoal.

If we supply too few tactics of a given class, then tactics are run
on leftmost subgoals of that class. If too many, then we don't run excess
tactics. 

We assume that all or no subgoals of a given class have numbers.
If subgoals of a given class have numbers then there either 
should be enough tactics supplied or no tactics supplied.
%

% put most specific first, so match below 
always finds most relevant alist entry
%

% most specific -> most general %
letref proof_label_order = ``predicate aux main any`` ;;

let reorder_proof_label_alist alist = 
  let fetch_entry lab = lab,apply_alist alist lab in
  let ordered_alist_part = mapfilter fetch_entry proof_label_order in

    remove_alist_entries alist proof_label_order 
    @ ordered_alist_part
;;




ml_curried_infix `THENLL` ;;

 
let $THENLL T lab_tac_list_pair_list =

 letrec aux alist ps =
   if null ps then [] 
   else
   let p = hd ps in
   let label = label_of_proof p in
   let label',Tlist = 
   ( find (\lab,().label_match lab label) alist
     ? `dummy`,[]
   ) in
   let T,new_alist = 
   ( if null Tlist then
       Id, alist
     if is_numbered_proof p then
      ( nth (number_of_proof p) Tlist, alist 
        ? failwith 
           `THENLL: missing tactic for numbered subgoal of type: ` ^ label
      )
     else
       hd Tlist, update_alist alist label' (tl Tlist)
   ) 
   in
     T . aux new_alist (tl ps)
 in
   T THEN_OnEach (aux (reorder_proof_label_alist lab_tac_list_pair_list))
;;


ml_curried_infix `THENML` ;;
ml_curried_infix `THENM` ;;
ml_curried_infix `THENAL` ;;
ml_curried_infix `THENA` ;;
ml_curried_infix `THENW` ;;
ml_curried_infix `THENw` ;;
ml_curried_infix `THENP` ;;

% These only apply to main (unlabelled) subgoals %

let $THENM T1 T2 = 
  T1 THEN IfLab `main` T2 Id 
;;

let $THENML T TList = 
  T THENLL [`main`,TList]
;;

% These only apply to auxiliary (labelled) subgoals %

let $THENA T1 T2 = 
  T1 THEN IfLab `aux` T2 Id 
;;

let $THENW T1 T2 = 
  T1 THEN IfLab `wf` T2 Id 
;;

let $THENw T1 T2 = 
  T1 THEN IfLab `wf` Id T2 
;;
  
let $THENAL T TList = 
  T THENLL [`aux`,TList]
;;

let $THENP T1 T2 = 
  T1 THEN IfLab `predicate` T2 Id 
;;


letrec RepeatM T p =
  (((Progress T) THENM RepeatM T) ORELSE Id) p ;;

letrec RepeatMFor n T p =
  if n = 0 then Id p
  else
  (T THENM RepeatMFor (n-1) T) p
;;

% sequence tactics on main subgoals... Abbreviates T1 THENM ... THENM Tn
  without thinking about associativity of THENM...
%

letrec SeqOnM Ts p =
  if null Ts then 
    Id p
  else 
  ( (hd Ts) THENM SeqOnM (tl Ts) ) p
;;

letrec Seq Ts p =
  if null Ts then 
    Id p
  else 
  ( (hd Ts) THEN Seq (tl Ts) ) p
;;

let CompleteM (T:tactic) p = 
  let ps,v = T p in
  if exists (label_match `main` o label_of_proof) ps then
    failwith `CompleteM`
  else 
    ps,v
;;

% notes change of label as well as sequent as progress %

let LabProgress (tac:tactic) (p : proof) =
  let ps,v = tac p in
    if length ps = 1 
      then
	let p' = (hd ps) in
          if equal_sequents p' p &
             label_of_proof p' = label_of_proof p
	  then failwith `LabProgress`
          else ps,v
    else ps,v
;;

%
***************************************************************************
     Tacticals for use with Hypothesis and Clausal Tactics
***************************************************************************
%

%
Clausal tactics are ones which work on both hypotheses and the conclusion.
Concl = Clause 0
Hyp i = Clause i    i > 0

Hypothesis tactics should leave one subgoal with the same conclusion as 
before their application. ?? Show principle subgoal with tag??

~~~
For convenience, we use negative hypothesis indices 
to number hyps from end of hyp list.

%

let OnClause i (T: int->tactic) p = 
  T (get_pos_hyp_num i p) p;;

let OnConcl (T: int->tactic) =
  T 0 
;;

let OnHyp = OnClause
;;

% repeat tactic on all subgoals %

let OnClauses clauses (T: int->tactic) =
  letrec Aux cs p =
    if null cs then Id p
    else
    ( OnClause (hd cs) T THEN (Aux (tl cs)) ) p
  in
    Aux clauses
;;
let OnHyps = OnClauses;;
let OnCls = OnClauses ;;

% repeat tactic on main subgoals. %

let OnMClauses clauses (T: int->tactic) =
  letrec Aux cs p =
    if null cs then Id p
    else
    ( OnClause (hd cs) T THENM Aux (tl cs) ) p
  in
    AddHiddenLabel `main` THEN Aux clauses
;;

let OnMHyps = OnMClauses
;;
let OnMCls = OnMClauses ;;

let On = OnMClauses ;;

% 
Hit all hyps or all clauses. Work from end of hyp list towards start.
%

let OnAllHyps T p = OnHyps (rev (upto 1 (num_hyps p))) T p ;;
let OnAllClauses T = OnAllHyps T THEN OnConcl T ;;
let OnAll = OnAllClauses ;;

let TryOnAllHyps T = OnAllHyps (\i.Try (T i)) ;;
let TryOnAllClauses T = OnAllClauses (\i.Try (T i)) ;;
let TryOnAll = TryOnAllClauses ;;

let OnAllMHyps T p = OnMHyps (rev (upto 1 (num_hyps p))) T p ;; 
let OnAllMClauses T = OnAllMHyps T THENM OnConcl T ;;

let TryOnAllMHyps T = OnAllMHyps (\i.Try (T i)) ;;
let TryOnAllMClauses T = OnAllMClauses (\i.Try (T i)) ;;

% abbreviations for commonest tacticals in this family: %

let All T = TryOnAllMClauses T ;;
let AllHyps T = TryOnAllMHyps T ;;


let OnLastHyp = OnHyp (-1) ;;

% work from end of hyp list %

let OnSomeHyp T p =
  letrec Aux i p' =
    if i = 0 then failwith `OnSomeHyp`
    else 
    (OnHyp i T ORELSE Aux (i-1)) p'
  in
    Aux (num_hyps p) p
;;


let OnVar v (T:int->tactic) p = T (get_decl_num v p) p ;;

   
%
***************************************************************************
Tactic argument tactics.
***************************************************************************
These tactics are for decorating proof terms with arguments for tactics.

See sequent.ml for basic definitions.
%

let PushArgs arg_alist p =
  let p' = push_args arg_alist p in
    Id p'
;;

let PopArgs p =
  let p' = pop_arg_stack p in
    Id p'
;;

let WithArgSet args T =
  PushArgs args THEN T THEN PopArgs
;;

let WithArgs args T p = 
  let old_args = get_arg_stack_top p ? []
  in
    WithArgSet (multi_update_alist old_args args) T p
;;


% Some common WithArgs ... %

% For args to DTerm, and types of principle args of non-canonical terms in 
  EqCD %

let With tm = WithArgs [`t1`,term_to_arg tm]  ;;

% For supplying new variable names %

let New vs = WithArgs (zip (get_indexed_id_set `v` (length vs))
                             (map var_to_arg vs)
                        )
;;

% For augmenting matches in forward and backward chaining. %

let Using sub = WithArgs [`sub`,sub_to_arg sub] ;;

% For universe levels required by many primitive rules. `At' takes a term
rather than level expression as argument, allowing use of both type universes
and `propositional' universes. (The `prop` abstraction.)
%

let At U = WithArgs [`universe`,term_to_arg U] ;;

% For selecting subterms in DNth and selecting simple formula components
when chaining with general formulae.
%

let Sel n = WithArgs [`n`,int_to_arg n] ;;

let Simple = Sel (-1) ;;

% 
For controlling thinning of hypotheses by tactics.
NB: Some tactics default behaviour will be to thin, others default behaviour
will be to not thin.
%


let NotThinning = WithArgs [`thinning`,tok_to_arg `no`] ;;
let Thinning = WithArgs [`thinning`,tok_to_arg `yes`] ;;






%
***************************************************************************
Utility Tactics.
***************************************************************************
%

letref enable_messages = false 
;;

let Message message_string p =
  if enable_messages then 
    display_message  message_string 
  else 
    ()
  ; Id p
;;
