%
*************************************************************************
*                                                                       *
*    Copyrighted Cornell University 2000                                *
*                                                                       *
*                                                                       *
*                Nuprl Proof Development System                         *
*                ------------------------------                         *
*                                                                       *
*   Developed by the Nuprl group, Department of Computer Science,       *
*   Cornell University, Ithaca NY.  See the release notes for a list    *
*   of the members of the group.                                        *
*                                                                       *
*   Permission is granted to use and modify Nuprl provided this notice  *
*   is retained in derived works.                                       *
*                                                                       *
*                                                                       *
*************************************************************************
%

%[
****************************************************************************
****************************************************************************
TYPE-INC.ML
****************************************************************************
****************************************************************************
Intro to functions for reasoning about type inclusion. 
Used by matching functions and inclusion tactics.

See inclusion-tactics and inc-tactics-3.ml for where supertype_funs
list is initialized.
]%

% 
supertype_funs
~~~~~~~~~~~~~~

list with elements of form id,f where

  id is an identifying token for the function. Used only for debugging 
     purposes to tell what is in the list.

  f t p --> t',T,k

     if f can find a supertype t' of t, perhaps looking at information in 
     the proof context p.
     T should be a tactic which when applied to clause i opens up the
     type t to t'. (In a hyp, it is expected to be the whole clause, in 
     the concl it is expected to be the type arg of a (membership or ?)
     equality term.

     k is a token: either `equal` or `maybe proper` indicating the
     inclusion relationship between t and t'.

     f should fail otherwise.
%

letref supertype_funs = [] 
  : (tok # (term -> proof -> (term # (int -> tactic) # tok))) list ;;


% Keep order of entries the same as that in which they might initially
  be added, since probably more frequently used ones will be defined
  earlier
%

let update_supertype_funs id fun = 
  supertype_funs := update_alist supertype_funs id fun ; ()
;;

let find_next_supertype p t = 
  first_value (\id,f.f p t) supertype_funs
  ? failwith `find_next_supertype`
;;


%
find_supertypes p t returns list 

  [t1,T1,k1; ... tn-1,Tn-1 ; tn,Fail,kn]

where t1 = t, ti are successive supertypes of t 
  Ti are tactics for getting ti+1 from ti.
  ki indicate kind of inclusion from ti to ti+1.
     should be one of `equal`, `maybe proper` or `none`
%

letrec find_supertypes p t =
 letrec aux l t =
 (let t',T,k = find_next_supertype t p in
    if member_p t' l lex_equal_terms then fail;
    (t,T,k) . aux (t'. l) t'
 ) ?
 [t,(\i p.failwith `Fail`),`none`]

 in aux [] t
;;


%
find_supertype_terms
~~~~~~~~~~~~~~~~~~~~
on input A, returns list of terms

[t1;...;tn]

Which are successive supertypes, starting with t1 = A
%

let find_supertype_terms tm = 
  map fst (find_supertypes (make_proof_node [] void_term) tm)
;;

%[
**********************************************************************
Finding Smallest Common Supertypes
**********************************************************************
Functions here used in type matching. Also used by inclusion tactics
in inc-tactics-3.ml
]%
  
% 
PreC: as & bs have some common suffix of length > 0.
Action: prune suffix
PostC: last pair are non-set terms & same,
   and are first such pair. 


This function had a previous obselete use. Now, its only caller
is find_common_supertype_depths so it could be simplified.
%


let prune_set_inc_info_list_suffixes (as,bs) = 

  % format: (rev els on left, els on right) 
    distinguished el is last on left
  %

  let el (c.cs , ds) = c in
  let left (c.cs , ds) = cs, c. ds in
  let right (cs, d. ds) = d.cs, ds in
  let at_start (cs,ds) = null cs in

  letrec goto_diffs as bs = 
   if at_start as or at_start bs or not el as = el bs then 
     as,bs % right lists are maximal common suffixes %
   else 
     goto_diffs (left as) (left bs) 
  in 
  letrec skip_sets as bs = 
    if el as = `set` then skip_sets (right as) (right bs) else as,bs
  in
  let rev_as = rev as in
  let rev_bs = rev bs in
  if not hd rev_as = hd rev_bs then
    failwith `prune_set_inc_info_suffices: no common suffix`
  else
  let as',bs' = goto_diffs (rev_as,[]) (rev_bs,[]) in
  let as'',bs'' = skip_sets (right as') (right bs') in
    rev (fst as''), rev (fst bs'')
;;

let find_common_supertype_depths (as : term list) (bs : term list) = 
  let as',bs' = 
    prune_set_inc_info_list_suffixes 
      (map opid_of_term as, map opid_of_term bs)
  in
    length as',length bs'
;;



% takes two types and finds the smallest supertypes with common
  outermost constructors
  Fail if no super type, or no progress.

%

let find_common_super_types (A,B) = 
  let ainfo = find_supertype_terms A
  in let binfo = find_supertype_terms B
  in
  if null (tl ainfo) & null (tl binfo) then
    failwith `find_common_super_types: no progress possible`
  else
  let na,nb = find_common_supertype_depths ainfo binfo 
  in
    nth na ainfo, nth nb binfo
;;

let get_hard_and_supertype_alts pat_inst_pr = 
  get_hardened_pr pat_inst_pr @ ([find_common_super_types pat_inst_pr] ? [])
;;

let type_inc_match = 
  full_match_with_retry 
   get_hard_and_supertype_alts 0 [] ;;



% 
Quick test. If this returns false, then S is (definitely?) not subtype of T 

Need to check policy on soft abstractions here.
%

let maybe_subtype S T = 
  member (opid_of_term (unfold_soft_abs T)) 
    (map opid_of_term (find_supertype_terms S))
;;


%[
**********************************************************************
Caches for set inclusion info
**********************************************************************

set_inc_info contains pairs ida,idb where 

ida is the opid of a type that has a supertype and either:

1. if idb is non-numeric, it is the opid of the immediate supertype 

2. if idb is a natural number, it indicates that the nth subterm is the 
   supertype.

We make the plausible assumption that subtypes of type 1 are never nested inside 
subtypes of type 2. 

Supertypes specified in 1. should always be those that are under any
soft abstractions, so that soft type abstractions are treated as being
transparent.  
]%

 % this stuff is defined in inc-tactics-3 as well so removed from here 2/4/2000.

letref set_inc_info = [] : (tok # tok) list
;;

let add_set_inc_info ida idb = 
  set_inc_info := update_alist set_inc_info ida idb
  ;
  ()
;;


letref set_inc_tactics = [] : (tok # (int->tactic) # tactic) list
;;

let add_set_inclusion_info ida idb DTac PTac = 
  set_inc_tactics:=
    update_alist set_inc_tactics ida (DTac,PTac)
  ;
  set_inc_info :=
    update_alist set_inc_info ida idb
  ;
  ()
;;
%
  
