/***************************************************************************
                          print.h  -  description
                             -------------------
    begin                : Mon Aug 18 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


/*! \file print.h
 *  Contains Definitions of CObjectCounter and all kinds of other
 *  debugging information
 */


#ifndef _PRINT_H_
#define _PRINT_H_

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include <iostream>
#include <iomanip>

/***************************************************************************
 *                                                                         *
 * DEFINES                                                                 *
 *                                                                         *
 ***************************************************************************/
//! Debug mode.
/*! Only enabled in DEBUG versions of nomore++ for DEBUG informations
 *  on stdout. Enable pointer tests.
 */
//#define _DEBUG

//! Debug output mode.
/*! Output of error informations not important for tests! Disable if
 * create release!
 */
//#define _DEBUG_OUTPUT

//! Enable all libunit++ tests.
//#define ALLTESTS

//! Enable answer set output on stdout.
/*! Disable this makro for better libunit++ test output.
 */
#define ASOUTPUT

//! Shows the alphabetical ordering on stdout.
/*! \attention Enable the alphabetical heuristic for right ordering (prolog).
 */
//#define SHOW_ALPHABETICAL_SORT

//! Enable single libunit++ tests.
//#define TESTAGGREGATION
//#define TESTCCOLORNODE
//#define TESTCCOLORSTRUCTURE
//#define TESTCGRAPH
//#define TESTCHEURISTIC
//#define TESTCNODE
//#define TESTCOBJECTCOUNTER
//#define TESTCPROGRAM
//#define TESTCRULE
//#define TESTCSTATUSINFORMATION
//#define TESTCSET
//#define TESTHELPER
//#define TESTMISC
//#define TESTOPERATORS

//! Print counter informations.
#define PRINTCALLCOUNTER

/***************************************************************************
 *                                                                         *
 * _DEBUG                                                                  *
 *                                                                         *
 ***************************************************************************/
//! Checks the pointer for null pointer and prints errors.
#ifdef _DEBUG

  //! Prints errors.
  #define PRINT_ERROR(x) x

  //! Checks pointers.
  #define CHECK_POINTER(text, x) if(x == NULL) {                                \
                                   PRINT_ERROR(std::cerr << text <<             \
                                               " - encountered NULL pointer" << \
                                               std::endl);                      \
                                   exit(1);                                     \
                                 }

  //! Using for object counter.
  #define CREATEOBJECT(STR) _object_counter.CreateObject(STR);

  //! Using for object counter.
  #define DELETEOBJECT(STR) _object_counter.DeleteObject(STR);

#else

  //! Prints errors.
  #define PRINT_ERROR(x)

  //! Checks pointers.
  #define CHECK_POINTER(text, x)

  //! Using for object counter.
  #define CREATEOBJECT(STR)

  //! Using for object counter.
  #define DELETEOBJECT(STR)

#endif

/***************************************************************************
 *                                                                         *
 * _DEBUG_OUTPUT                                                           *
 *                                                                         *
 ***************************************************************************/
//! Prints debug informations.
#ifdef _DEBUG_OUTPUT

  //! Prints debug information.
  #define PRINT_DEBUG(x) x

#else

  //! Prints debug information.
  #define PRINT_DEBUG(x)

#endif

/***************************************************************************
 *                                                                         *
 * ALLTESTS                                                                *
 *                                                                         *
 ***************************************************************************/
//! Enable all tests.
#ifdef ALLTESTS

  #define TESTHELPER
  #define TESTAGGREGATION
  #define TESTCCOLORNODE
  #define TESTCCOLORSTRUCTURE
  #define TESTCGRAPH
  #define TESTCHEURISTIC
  #define TESTCNODE
  #define TESTCOBJECTCOUNTER			    
  #define TESTCPROGRAM
  #define TESTCRULE
  #define TESTCSTATUSINFORMATION
  #define TESTCSET
  #define TESTMISC
  #define TESTOPERATORS
  
#endif

/***************************************************************************
 *                                                                         *
 * PRINTCALLCOUNTER                                                        *
 *                                                                         *
 ***************************************************************************/
//! Enable call counter output.
#ifdef PRINTCALLCOUNTER

  /*************************************************************************
   * Counters for counters :)                                              *
   *************************************************************************/
  //! Counter for the supported method.
  extern long methode_supported;

  //! Counter for the weak supported method.
  extern long methode_weak_supported;

  //! Counter for the unsupported method.
  extern long methode_unsupported;

  //! Counter for the blocked method.
  extern long methode_blocked;

  //! Counter for the unblocked method.
  extern long methode_unblocked;

  //! Counter for the supported_by_set method.
  extern long methode_supported_by_set;

  //! Counter for the total colored method.
  extern long methode_totalcoloring;

  //! Counter for the allsuccscolored (ignore) method.
  extern long methode_all_succs_colored;

  /*************************************************************************
   * Counters for deterministic operators                                  *
   *************************************************************************/
  //! Counter for the operator none call.
  extern long methode_operator_none_call;

  //! Counter for the operator p call.
  extern long methode_operator_p_call;

  //! Counter for the operator ps call.
  extern long methode_operator_ps_call;

  //! Counter for the operator ps call.
  extern long methode_operator_labh_call;

  //! Counter for the operator bp call.
  extern long methode_operator_bp_call;

  //! Counter for the operator u call.
  extern long methode_operator_u_call;

  //! Counter for the operator ms call.
  extern long methode_operator_ms_call;

  //! Counter for the operator v call.
  extern long methode_operator_v_call;

  //! Counter for the operator v2 call.
  extern long methode_operator_v2_call;

  //! Counter for the operator v3 call.
  extern long methode_operator_v3_call;

  //! Counter for the operator v4 call.
  extern long methode_operator_v4_call;

  //! Counter for the operator v5 call.
  extern long methode_operator_v5_call;

  //! Counter for the operator n call.
  extern long methode_operator_n_call;

  //! Counter for the operator nsecure call.
  extern long methode_operator_nsecure_call;

  //! Counter for the operator pre call.
  extern long methode_operator_pre_call;

  /*************************************************************************
   * Counters for non-deterministic operators                              *
   *************************************************************************/
  //! Counter for the operator c call.
  extern long methode_operator_c_call;

  //! Counter for the operator cq call.
  extern long methode_operator_cq_call;

  //! Counter for the operator d call.
  extern long methode_operator_d_call;

  //! Counter for the operator dh call.
  extern long methode_operator_dh_call;

  //! Counter for the operator ch call.
  extern long methode_operator_dq_call;

  /*************************************************************************
   * Counters for queueing                                                 *
   *************************************************************************/
  //! Counter putting node into todo queue.
  extern long queue_todo_call;

  //! Counter putting node into bp queue.
  extern long queue_jumping_call;

  //! Counter putting node into cp queue.
  extern long queue_cp_call;

  //! Counter for the ignore count method.
  extern long count_ignore;

  /*************************************************************************
   * Counters for heuristics                                               *
   *************************************************************************/
  //! Counter for heuristics s1 calls.
  extern long heuristics_s1_call;

  //! Counter for heuristics s2 calls.
  extern long heuristics_s2_call;

  //! Counter for heuristics sr calls.
  extern long heuristics_sr_call;

  //! Counter for heuristics d1 calls.
  extern long heuristics_d1_call;

  //! Counter for heuristics dr calls.
  extern long heuristics_dr_call;

  //! Counter for heuristics alpha calls.
  extern long heuristics_alpha_call;

  /*************************************************************************
   * More                                                                  *
   *************************************************************************/  
  //! Counter for the colornode method.
  extern long methode_colornode;

  //! Counter for choice points.
  extern long choice_points;


  //! Using for incrementing a counter.
  #define INC_COUNT(x) x++

  //! Print a counter to stdout.
  #define PRINT_COUNTER(text, x) if(x>0) std::cout << "Call to "        \
                                                   << text << ": " << x \
                                                   << std::endl;

  //! Print a counter to stdout.
  #define PRINT_COUNTERE(text, x) std::cout << "Call to " << text \
                                            << ": " << x << std::endl;

  //! Prints a horizonal line.
  #define PRINT_HR(x) std::cout << "--" << std::setw(30) << std::setfill('-') \
                                << x  << std::setfill('-') << std::endl;

  //! Print all counters to stdout.  
  #define PRINT_ALL_COUNTER PRINT_HR("counters            ");                                  \
                            PRINT_COUNTER("supported       ", methode_supported);              \
                            PRINT_COUNTER("weak_supported  ", methode_weak_supported);         \
                            PRINT_COUNTER("supported_by_set", methode_supported_by_set);       \
                            PRINT_COUNTER("unsupported     ", methode_unsupported);            \
                            PRINT_COUNTER("blocked         ", methode_blocked);                \
                            PRINT_COUNTER("unblocked       ", methode_unblocked);              \
                            PRINT_COUNTER("allsuccscolored ", methode_all_succs_colored);      \
                            PRINT_HR("det. operators      ");                                  \
                            PRINT_COUNTER("operator-P      ", methode_operator_p_call);        \
                            PRINT_COUNTER("operator-P*     ", methode_operator_ps_call);       \
                            PRINT_COUNTER("operator-BP     ", methode_operator_bp_call);       \
                            PRINT_COUNTER("operator-U      ", methode_operator_u_call);        \
                            PRINT_COUNTER("operator-MS     ", methode_operator_ms_call);       \
                            PRINT_COUNTER("operator-V      ", methode_operator_v_call);        \
                            PRINT_COUNTER("operator-V2     ", methode_operator_v2_call);       \
                            PRINT_COUNTER("operator-V3     ", methode_operator_v3_call);       \
                            PRINT_COUNTER("operator-V4     ", methode_operator_v4_call);       \
                            PRINT_COUNTER("operator-V5     ", methode_operator_v5_call);       \
                            PRINT_COUNTER("operator-N      ", methode_operator_n_call);        \
                            PRINT_COUNTER("operator-Nsecure", methode_operator_nsecure_call);  \
                            PRINT_COUNTER("operator-Pre    ", methode_operator_pre_call);      \
                            PRINT_HR("non-det. operators  ");                                  \
                            PRINT_COUNTER("operator-C      ", methode_operator_c_call);        \
                            PRINT_COUNTER("operator-CQ     ", methode_operator_cq_call);       \
                            PRINT_COUNTER("operator-D      ", methode_operator_d_call);        \
                            PRINT_COUNTER("operator-DQ     ", methode_operator_dq_call);       \
                            PRINT_COUNTER("operator-DH     ", methode_operator_dh_call);       \
                            PRINT_HR("heuristics          ");                                  \
                            PRINT_COUNTER("heuristics-S1   ", heuristics_s1_call);             \
                            PRINT_COUNTER("heuristics-S2   ", heuristics_s2_call);             \
                            PRINT_COUNTER("heuristics-SR   ", heuristics_sr_call);             \
                            PRINT_COUNTER("heuristics-D1   ", heuristics_d1_call);             \
                            PRINT_COUNTER("heuristics-DR   ", heuristics_dr_call);             \
                            PRINT_COUNTER("heuristics-Alpha", heuristics_alpha_call);          \
                            PRINT_HR("queueing            ");                                  \
                            PRINT_COUNTER("todo            ", queue_todo_call);                \
                            PRINT_COUNTER("jumping         ", queue_jumping_call);             \
                            PRINT_COUNTER("cp              ", queue_cp_call);                  \
                            PRINT_COUNTER("ignore          ", count_ignore);                   \
                            PRINT_HR("more                ");                                  \
                            PRINT_COUNTERE("colornode       ", methode_colornode);             \
                            PRINT_COUNTERE("total coloring  ", methode_totalcoloring);         \
                            PRINT_COUNTERE("choice points   ", choice_points);

#else

  //! Using for incrementing a counter.
  #define INC_COUNT(x)

  //! Print a counter to stdout.
  #define PRINT_COUNTER(text, x)

  //! Print a counter to stdout.
  #define PRINT_COUNTERE(text, x)
  
  //! Print all counters to stdout.
  #define PRINT_ALL_COUNTER

#endif


/***************************************************************************
 *                                                                         *
 * only in _DEBUG mode                                                     *
 *                                                                         *
 ***************************************************************************/
#ifdef _DEBUG

/***************************************************************************
 *                                                                         *
 * includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include <map>
#include <string>

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE{

/***************************************************************************
 *                                                                         *
 * typedefinitions                                                         *
 *                                                                         *
 ***************************************************************************/
/*! \struct TCounter
 *  \brief Counts the created and destroyed objects of one object type.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Aug 18 2003
 */
typedef struct{

  //! Counts the created instances of any type.
  long Created;

  //! Counts the deleted instances of any type.
  long Deleted; 

} TCounter;

/***************************************************************************
 *                                                                         *
 * class CObjectCounter                                                    *
 *                                                                         *
 ***************************************************************************/
/*! \class CObjectCounter
 *  \brief Counts the created and destroyed objects.
 *  \author Andre Neumann, Christian Anger
 *  \date Mon Aug 18 2003
 */
class CObjectCounter {

  //! Output operator.
  friend std::ostream& operator<<(std::ostream& os, CObjectCounter& obj);
    
private:
  //! structure for counting the objects.
  std::map<std::string, TCounter>  _count_list; 

public:
  //! Constructor.
	CObjectCounter();

  //! Destructor.
 	~CObjectCounter();

  //! Increase the Created counter for the type.
  void CreateObject(std::string type);

  //! Increase the Deleted counter for the type.
  void DeleteObject(std::string type);

  //! Returns the count of object which are not deleted yet of specified type.
  long ObjectCount(std::string type);

  //! Returns true if all objects are deleted.
  bool Clear();

  //! Clears the map.
  void DeleteAll();

};

//! single instance
extern CObjectCounter _object_counter;

} // NS_NOMORE

#endif

#endif
