/***************************************************************************
                          caggregation.h  -  description
                             -------------------
    begin                : Tue Jul 29 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file caggregation.h
 *  Declaration of class CAggregation, CAggregationStar and
 *  CAggregationSequence.
*/

#ifndef CAGGREGATION_H
#define CAGGREGATION_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "coperator.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class CAggregation                                                      *
 *                                                                         *
 ***************************************************************************/
/*! \class CAggregation caggregation.h
 *  \brief Abstract class for defining aggregations like * or sequence.
 *  \author Christian Anger, Andre Neumann
 *  \date Tue Jul 29 2003
 *
 *  CAggregation is the base class for all aggregations. If you want to
 *  implement new aggregations your class must be inherit by CAggregation
 *  and must be overwrite the methode CDetOperator::Call().
 */
class CAggregation : public CDetOperator {

public:
  //! Constructor.
	CAggregation();
	
  //! Destructor.
	virtual ~CAggregation();

};

/***************************************************************************
 *                                                                         *
 * class CAggregationStar                                                  *
 *                                                                         *
 ***************************************************************************/
/*! \class CAggregationStar caggregation.h
 *  \brief Aggregation for computing the closure operator.
 *  \author Christian Anger, Andre Neumann
 *  \date Tue Jul 29 2003
 *
 *  The Call() methode executes the first operator (or first aggregation)
 *  until the first operator (or aggregation) returns return_unchanged or
 *  return_colorerror.
 */
class CAggregationStar : public CAggregation {

private:
	//! Pointer to the operator in star aggregation.
  CDetOperator*  _operator;

public:
  //! Constructor.
  CAggregationStar();

  //! Destructor.
  virtual ~CAggregationStar();

  //! Call the first operator as loop.
  /*! \pre  The aggregation must be initialized by SetOperator().
   *  \post No.
   *  \param graph  Graph on which the computation started.
   *  \param stack  Stack to which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If nothing has been changed the methode return
   *          return_unchanged, else return_changed. If an error occured
   *          the computation the methode returns return_colorerror.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);
  
  //! Returns an operator specific id which is used for testing for equality.
  inline virtual TType GetType() const {
    
    return type_caggregation_star;
    
  }

  //! Returns the operator id like GetType.
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   *  \pre  The aggregation must be initialized with SetOperator.
   *  \post No.
   */
  virtual TTypeEnum GetOperatorID();
  
  //! Equality operator.
  /*! Checks whether the first operator (or aggregation) is equal.
   */
  virtual bool operator==(COperator& op);

  //! Inequality operator.
  /*! Checks whether the first operator (or aggregation) is not equal.
   */
  virtual bool operator!=(COperator& op);

  //! Sets the operator for the star aggregation.
  /*! \pre  No.
   *  \post No.
   *  \param op Operator or aggregation used for star-operator. If pointer
   *            is NULL the program terminated.
   */
  void SetOperator(CDetOperator *op);

  //! Returns the pointer to the operator or aggregation.
  /*! Use GetType() for more information about type of operator.
   */
  inline CDetOperator* GetOperator(){
    
    return _operator;
    
  }
  
};

/***************************************************************************
 *                                                                         *
 * class CAggregationSequence                                              *
 *                                                                         *
 ***************************************************************************/
/*! \class CAggregationSequence caggregation.h
 *  \brief Aggregation for computing the sequences of operators or other
 *         aggregations.
 *  \author Christian Anger, Andre Neumann
 *  \date Tue Jul 29 2003
 *
 *  The Call() methode of CAggregationSequence calls the first operator
 *  (or aggregation) and after successful termination the second operator
 *  (or aggregation).
 */
class CAggregationSequence : public CAggregation {

private:
	//! Pointer to the first operator in sequence.
  CDetOperator*  _first_operator;

  //! Pointer to the second operator in sequence.
  CDetOperator*  _second_operator;

public:
  //! Constructor.
  CAggregationSequence();

  //! Destructor.
  virtual ~CAggregationSequence();

  //! Call the first operator followed by the second operator.
  /*! \pre  Both operators must be set.
   *  \post No.
   *  \param graph  Graph on which the computation started.
   *  \param stack  Stack to which the changes are stored.
   *  \param todo   Contains information about all todo lists.
   *  \param status Contains information about how many nodes are colored.
   *  \return If nothing has been changed the methode return
   *          return_unchanged, else return_changed. If an error occured
   *          the computation the methode returns return_colorerror.
   */
  virtual TRetOperator Call(CGraph* graph,
                            CStack* stack,
                            CTodoLists* todo,
                            CStatusInformation* status = NULL);

  //! Returns an operator specific id which is used for testing of equality.
  inline virtual TType GetType() const {
    
    return type_caggregation_sequence;
    
  }

  //! Returns the operator id like GetType. 
  /*! All deterministic operators return its own identifier and the
   *  aggregations return a bit code of operators.
   *  \pre  Both operators must be set.
   *  \post No.
   */
  virtual TTypeEnum GetOperatorID();

  //! Equality operator.
  /*! Checks whether the first and the second operators (or aggregations)
   *  are equal.
   */
  virtual bool operator==(COperator& agg);

  //! Inequality operator.
  /*! Checks whether the first or the second operators (or aggregations)
   *  are not equal.
   */
  virtual bool operator!=(COperator& agg);

  //! Sets the first operator for the aggregation.
  /*! \param op Operator or aggregation used for star-operator or for the
   *            first element of the sequence aggregation. If pointer is
   *            NULL the program terminated.
   */
  void SetFirst(CDetOperator *op);

  //! Sets the second operator for the aggregation.
  /*! \param op Operator or aggregation used for the second element of the
   *            sequence aggregation. If pointer is NULL the program
   *            terminated.
   */
  void SetSecond(CDetOperator *op);

  //! Returns the pointer to the first operator or aggregation.
  /*! Use GetType() for more information about type of first operator.
   */
  inline CDetOperator* GetFirst() {
    
    return _first_operator;
    
  }

  //! Returns the pointer to the second operator or aggregation.
  /*! Use GetType() for more information about type of second operator.
   */
  inline CDetOperator* GetSecond() {
    
    return _second_operator;
    
  }
  
};

}

#endif
