/***************************************************************************
                          cnode.h  -  description
                             -------------------
    begin                : Thu Jun 19 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

/*! \file cnode.h
 *  Declaration of class CNode, CHeadNode and CBodyNode.
 */


#ifndef CNODE_H
#define CNODE_H

/***************************************************************************
 *                                                                         *
 * Includes                                                                *
 *                                                                         *
 ***************************************************************************/
#include "misc.h"
#include "typedefinitions.h"
#include "cheuristic.h"

/***************************************************************************
 *                                                                         *
 * Namespace NS_NOMORE                                                     *
 *                                                                         *
 ***************************************************************************/
/*! \namespace NS_NOMORE
 *  \brief Contains everything related to nomore++.
*/
namespace NS_NOMORE {

/***************************************************************************
 *                                                                         *
 * class CNode                                                             *
 *                                                                         *
 ***************************************************************************/
/*! \class CNode cnode.h
 *  \brief Defines the abstract class for a node in a graph.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  CNode is the base class for all nodes. It defines the basic behaviour
 *  that will be common to all future node classes, such as a color and
 *  functionality for interaction with a stack for backtracking purposes.
 */
class CNode {

protected:
  //! Contains the color of the node.
  TColor _color;

private:
  CNode(const CNode& n) {}
  CNode& operator=(const CNode& n) { return *this; }

  //! Contains the id of the node. Used to identify node within the TNodeSet.
  TId _id;

  //! Static or dynamic heuristic for the node.
  CHeuristic* _pheuristic; 

  //! Is the node in the propagation queue (TODO)
  bool _is_in_propagation_queue;

  //! Is the node in the backward propagation queue (BP)
  bool _is_in_backward_propagation_queue;

  //! Is the node in the possible choice point queue (CP)
  bool _is_in_possible_choices_queue;

  //! Is the node in the C possible choice point queue (CP)
  bool _is_in_c_possible_choices_queue;

  //! Is the node in the minus queue.
  bool _is_in_minus_queue;

  //! Minimum heuristics value for the node.
  long _min;

  //! Maximum heuristics value for the node.
  long _max;
  
  //! The node can be propagated by any other node (important for lookahead).
  bool _can_be_propagated;
  
  bool _is_in_v4_set;

public:
  //! Constructor.
  CNode();

  //! Destrcutor.
  virtual ~CNode();

  //! Equality operator.
  virtual bool operator==(const CNode &rhs) = 0;

  //! Inequality operator.
  virtual bool operator!=(const CNode &rhs) = 0;

  //! Recovers dynamic state attributes during backtracking.
  /*! \param stack_element Contains information about node for loading.
   *  \param stack         Stack for saving the stack element.
   *  \return If the state is successful loaded the methode returns true
   *          else false.
   */
  virtual bool LoadState(TStackElement *stack_element, CStack *stack) = 0;

  //! Colors a node with a new color and save the old state into the stack.
  /*! \param color        The new color of the node.
   *  \param stack        Stack for saving the old state.
   *  \param choice_point Is this node a choice point?
   *  \param choice_color Some additional information of the choice point.
   *  \return If the node is successful recolored the methode returns true
   *          else false.
   */
  bool ColorNode(const TColor color, CStack *stack, const bool choice_point,
                 TColor choice_color);

  //! Returns the node identifier.
  inline TId GetId() const { 
    return _id;
  }

  //! Sets the identifier.
  inline void SetId(const TId id) {
    _id = id;
  }

  //! Returns the heuristic.
  inline CHeuristic* GetHeuristic() {
    return _pheuristic;
  }

  //! Sets the heuristic.
  inline void SetHeuristic(CHeuristic* heu) {
    CHECK_POINTER("CNode::SetHeuristic(heu)", heu);

    if(_pheuristic != NULL)
      delete _pheuristic;
    _pheuristic = heu;  
  }

  //! Returns the color.
  inline TColor GetColor() const {
    return _color;
  }

  //! Returns type of instance.
  virtual TNodeType GetType() const = 0;

  //! Returns the heuristic value of the node.
  /*! \pre The heuristics must be set.
   */
  inline int HeuristicValue() {
    CHECK_POINTER("CNode::HeuristicValue() -> _pheuristic?", _pheuristic);
    return _pheuristic->GetHeuristicValue();
  }
  
  //! Is the node in the propagation queue (TODO)
  inline bool IsInPropagationQueue() const {
    return _is_in_propagation_queue;
  }

  //! Is the node in the backward propagation queue (BP)
  inline bool IsInBackwardPropagationQueue() const {
    return _is_in_backward_propagation_queue;
  }

  //! Is the node in the possible choice point queue (CP)
  inline bool IsInPossibleChoicesQueue() const {
    return _is_in_possible_choices_queue;
  }

  //! Is the node in the c possible choice point queue (CP)
  inline bool IsInCPossibleChoicesQueue() const {
    return _is_in_c_possible_choices_queue;
  }

  //! Is the node in the minus queue.
  inline bool IsInMinusQueue() const {
    return _is_in_minus_queue;
  }

  //! Is the node in the propagation queue (TODO)
  inline void SwitchIsInPropagationQueue(bool val) {
    _is_in_propagation_queue = val;
  }

  //! Is the node in the backward propagation queue (BP)
  inline void SwitchIsInBackwardPropagationQueue(bool val) {
    _is_in_backward_propagation_queue = val;
  }

  //! Is the node in the possible choice point queue (CP)
  inline void SwitchIsInPossibleChoicesQueue(bool val) {
    _is_in_possible_choices_queue = val;
  }

  //! Is the node in the C possible choice point queue (CP)
  inline void SwitchIsInCPossibleChoicesQueue(bool val) {
    _is_in_c_possible_choices_queue = val;
  }

  //! Is the node in the minus queue.
  inline void SwitchIsInMinusQueue(bool val) {
    _is_in_minus_queue = val;
  }

  //! Returns the maximum heuristic value (e.g. used for smodels heuristics and lookahead).
  long GetMaxHeuristics() const {
    return _max;
  }

  //! Returns the minimum heuristic value (e.g. used for smodels heuristics and lookahead).
  long GetMinHeuristics() const {
    return _min;
  }

  //! Sets the maximum heuristic value (e.g. used for smodels heuristics and lookahead).
  void SetMaxHeuristics(long max) {
    _max = max;
  }

  //! Sets the minimum heuristic value (e.g. used for smodels heuristics and lookahead).
  void SetMinHeuristics(long min) {
    _min = min;
  }
  
  //! Sets true if the node can be propagated by any other node (important for lookahead).
  void SetCanBePropagated(bool value) {
    _can_be_propagated = value;
  }

  //! Returns true if the node can be propagated by any other node (important for lookahead).
  bool GetCanBePropagated() const {
    return _can_be_propagated;
  }

  bool IsInV4Set() const {
    return _is_in_v4_set;  
  }
  
  void SetIsInV4Set(bool value) {
    _is_in_v4_set = value;  
  }
};

/***************************************************************************
 *                                                                         *
 * class CBodyNode                                                         *
 *                                                                         *
 ***************************************************************************/
/*! \class CBodyNode
 *  \brief The node corresponds to a body of a rule.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  The body node consists of a list of successors, 0- and 1-predecessors.
 *  Additionally the class holds counters for faster computation.
 */
class CBodyNode : public CNode {

  //! Output operator.
  friend std::ostream& operator<<(std::ostream& os, CBodyNode& node);

private:
  CBodyNode(const CBodyNode& n) {}
  CBodyNode& operator=(const CBodyNode& n) { return *this; }
	
        //! Pointer to the body that is represented.
  CBody     *_pbody; 

  //! Contains all 0-preds.
  THeadNodeSet  _zero_predecessors;

  //! Contains all 1-preds.
  THeadNodeSet  _one_predecessors;

  //! Contains all unlabeled succs.
  THeadNodeSet  _successors;

  //! Supported counter.
  /*! Supported counter counts the number of positive (zero) predecessors
   *  not colored plus. If all zero predecessors colored plus the value
   *  is 0. Initially the counter is set to the number of zero predecessors.
   */
  long _supported;

  //! Weak-Supported counter.
  /*! Weak-Supported counter counts the number of positive (zero)
   *  predecessors not colored plus or weak_plus. If all zero predecessors
   *  colored plus or weak-plus the value is 0. Initially the counter is
   *  set to the number of zero predecessors.
   */
  long _weak_supported;

  //! Unsupported counter.
  /*! Unsupported counter counts the number of positive (zero) predecessors
   *  colored minus. If one zero predecessor becomes minus the value would
   *  be decremented. Initially the counter is set to 0.
   */
  long _unsupported;

  //! Blocked counter.
  /*! Blocked counter counts the number of negative (one) predecessors
   *  colored plus. If one one-predecessor becomes plus the value would be
   *  decremented. Initially the counter is set to 0.
   */
  long _blocked;

  //! Unblocked counter.
  /*! Unblocked counter counts the number of negative (one) predecessors
   *  not colored minus. If all one predecessors colored minus the value
   *  is 0. Initially the counter is set to the number of one predecessors.
   */
  long _unblocked;

  //! Unblocked-Successor counter.
  /*! Unblocked-Successor counter counts the number of successors not
   *  colored minus or plus. If all successors colored minus or plus the
   *  value is 0. Initially the counter is set to the number of successors.
   */
  long _uncolored_succs;

public:
  //! Constructor. Initalizes Node with an id and a body.
  CBodyNode(CIdGenerator *ig, CBody *body = NULL);

  //! Constructor. Initalizes Node with an id and a body. 
  CBodyNode(const TId sid, CBody *body = NULL);

  //! Destructor.
  virtual ~CBodyNode();

  //! Equality operator.
  /*! Checks for equal bodies.
   */
  virtual bool operator==(const CNode &rhs);

  //! Inequality operator.
  /*! Checks the inequality of the bodies.
   */
  virtual bool operator!=(const CNode &rhs);

  //! Returns type of instance.
  inline virtual TNodeType GetType() const {
    return type_cbodynode;
  }

  //! Returns 0-Preds.
  inline THeadNodeSet &GetZeroPredecessors() {
    return _zero_predecessors;
  }

  //! Returns 1-Preds.
  inline THeadNodeSet &GetOnePredecessors() {
    return _one_predecessors;
  }

  //! Returns unlabeled Succs.
  inline THeadNodeSet &GetSuccessors() {
    return _successors;
  }

  //! Inserts a node as 0-Pred.
  bool InsertZeroPredecessor(CHeadNode*);

  //! Inserts a node as 1-Pred.
  bool InsertOnePredecessor(CHeadNode*);

  //! Inserts a node as unlabeled Successor and calls CHeadNode::InsertPredecessor().
  bool InsertSuccessor(CHeadNode*);

  //! Sets the corresponding body.
  inline void SetBody(CBody *body) {
    CHECK_POINTER("CBodyNode::SetBody(body)", body);
    _pbody = body;
  }

  //! Returns the corresponding body.
  inline CBody* GetBody() {
    return _pbody;
  }

  //! Returns true if the node is supported.
  /*! A node is supported if the supported counter of the node is zero.
   *  That means that all zero predecessors are colored to plus.
   */
  inline bool Supported() const {
    INC_COUNT(methode_supported);
    return 0 >= SupportedCounter();
  }

  //! Returns the supported counter.
  inline long SupportedCounter() const {
    return _supported;
  }

  //! Modifies the supported counter with input number.
  inline void ModifySupportedCounter(const long value) {
    _supported += value;
  }

  //! Returns true if all successors of the node are colored to plus or minus.
  /*! If all successors of a node are colored (plus or minus) this node is
   * not important. This node can be ignored. (used for ignore)
   */
  inline bool AllSuccsColored() const {
    INC_COUNT(methode_all_succs_colored);
    return _uncolored_succs == 0;
  }

  //! Returns the counter for all successor colored.
  /*! The return value is equivalente to the number of successors which are
   *  not colored to plus or minus.
   */
  inline long AllSuccsColoredCounter() const {
    return _uncolored_succs;
  }

  //! Modifies the supported counter with input number (used for ignore).
  inline void ModifyAllSuccsColoredCounter(const long value) {
    _uncolored_succs+=value;
  }
  
  //! Returns true if the node is weak supported.
  /*! A node is weak supported if the weak supported counter of the node is
   *  zero. That means that all zero predecessors are colored to plus or
   *  weak-plus.
   */
  inline bool WeakSupported() const {
    INC_COUNT(methode_weak_supported);
    return 0 >= WeakSupportedCounter();
  }

  //! Modifies the weak supported counter with input number.
  inline void ModifyWeakSupportedCounter(const long value) {
    _weak_supported += value;
  }

  //! Returns the weak-supported counter.
  inline long WeakSupportedCounter() const {
    return _weak_supported;
  }

  //! Returns true if the node is supported by the input set of nodes.
  bool Supported(TNodeSet* plus_colored);

  //! Returns true if the node is unsupported.
  /*! A node is unsupported if the unsupported counter of the node is less
   *  than zero. That means that one or more zero predecessor(s) are
   *  colored to minus.
   */
  inline bool Unsupported() const {
    INC_COUNT(methode_unsupported);
    return 0 > UnsupportedCounter();
  }

  //! Returns the unsupported counter.
  inline long UnsupportedCounter() const {
    return _unsupported;
  }

  //! Modifies the unsupported counter with input number.
  inline void ModifyUnsupportedCounter(const long value) {
    _unsupported += value;
  }

  //! Returns true if the node is blocked.
  /*! A node is blocked if the blocked counter of the node is less than
   *  zero. That means that one or more one predecessor(s) are colored
   *  to plus.
   */
  inline bool Blocked() const {
    INC_COUNT(methode_blocked);
    return 0 > BlockedCounter();
  }

  //! Returns the counter for blocked.
  inline long BlockedCounter() const {
    return _blocked;
  }

  //! Modifies the blocked counter with input number.
  inline void ModifyBlockedCounter(const long value) {
    _blocked += value;
  }

  //! Returns true if the node is unblocked.
  /*! A node is unblocked if the unblocked counter of the node is zero.
   *  That means that all one predecessors are colored to minus.
   */
  inline bool Unblocked() const {
    INC_COUNT(methode_unblocked);
    return 0 >= UnblockedCounter();
  }

  //! Returns the counter for unblocked.
  inline long UnblockedCounter() const {
    return _unblocked;
  }

  //! Modifies the unblocked counter with input number.
  inline void ModifyUnblockedCounter(const long value) {
    _unblocked += value;
  }

  //! Returns true if the node is a selfblocker.
  /*! Selfblocker means that one successor of this node has as 1-successor
   *  this node.
   */
  bool SelfBlocker();

  //! Recovers dynamic state attributes during backtracking.
  /*! \param stack_element Contains information about node for loading.
   *  \param stack         Stack for saving the stack element.
   *  \return If the state is successful loaded the methode returns true
   *          else false.
   */
  virtual bool LoadState(TStackElement *stack_element, CStack *stack);
};

/***************************************************************************
 *                                                                         *
 * class CHeadNode                                                         *
 *                                                                         *
 ***************************************************************************/
/*! \class CHeadNode
 *  \brief The node corresponds to one head of a rule.
 *  \author Andre Neumann, Christian Anger
 *  \date Thu Jun 19 2003
 *
 *  This node holds a set of predecessors, 0- and 1-successors and some
 *  counters.
 */
class CHeadNode : public CNode {

  //! Output operator.
  friend std::ostream& operator<<(std::ostream& os, CHeadNode& node);

private:
  CHeadNode(const CHeadNode& n) {}
  CHeadNode& operator=(const CHeadNode& n) { return *this; }
  
  //! Pointer to the atom that is represented.
  CAtom     *_patom; 

  //! Contains all 0-succs.
  TBodyNodeSet  _zero_successors;

  //! Contains all 1-succs.
  TBodyNodeSet  _one_successors;

  //! Contains all unlabeled preds.
  TBodyNodeSet  _predecessors;

  //! Unsupported counter.
  /*! Unsupported counter counts the number of (unlabeled) predecessors
   *  colored minus. If one zero predecessor becomes minus the value would
   *  be decremented. Initially the counter is set to the number of
   *  predecessors.
   */
  long _unsupported;

public:
  //! Constructor. Initalizes Node with an id and an atom.
  CHeadNode(CIdGenerator *ig, CAtom* atom = NULL);

  //! Constructor. Initalizes Node with an id and an atom.
  CHeadNode(const TId sid, CAtom* atom = NULL);

  //! Destructor.
  virtual ~CHeadNode();

  //! Equality operator.
  /*! Checks for equal heads.
   */
  virtual bool operator==(const CNode &rhs);

  //! Inequality operator.
  /*! Checks the inequality of the heads.
   */
  virtual bool operator!=(const CNode &rhs);

  //! Returns type of instance.
  inline virtual TNodeType GetType() const {
    return type_cheadnode;
  }

  //! Returns 0-Succs.
  inline TBodyNodeSet &GetZeroSuccessors() {
    return _zero_successors;
  }

  //! Returns 1-Succs.
  inline TBodyNodeSet &GetOneSuccessors() {
    return _one_successors;
  }

  //! Returns unlabeled Preds.
  inline TBodyNodeSet &GetPredecessors() {
    return _predecessors;
  }

  //! Inserts a node as 0-Pred and calls CBodyNode::InsertZeroPredecessor().
  bool InsertZeroSuccessor(CBodyNode*);

  //! Inserts a node as 1-Pred and calls CBodyNode::InsertOnePredecessor().
  bool InsertOneSuccessor(CBodyNode*);

  //! Inserts a node as unlabeled Pred.
  bool InsertPredecessor(CBodyNode*);

  //! Sets the corresponding head.
  inline void SetHead(CAtom *atom) {
    CHECK_POINTER("CHeadNode::SetHead(atom)", atom);
    _patom = atom;
  }

  //! Returns the corresponding head.
  inline CAtom* GetHead() {
    return _patom;
  }

  //! Returns true if the node is unsupported.
  /*! Unsupported for a head node means that all predecessors are colored
   *  minus.
   */
  inline bool Unsupported() const {
    return 0 == UnsupportedCounter();
  }

  //! Returns the unsupported counter for a head node.
  inline long UnsupportedCounter() const {
    return _unsupported;
  }

  //! Modifies the unsupported counter with input number.
  inline void ModifyUnsupportedCounter(const long value) {
    _unsupported += value;
  }  

  //! Recovers dynamic state attributes during backtracking.
  /*! \param stack_element Contains information about node for loading.
   *  \param stack         Stack for saving the stack element.
   *  \return If the state is successful loaded the methode returns true
   *          else false.
   */
  virtual bool LoadState(TStackElement *stack_element, CStack *stack);
};

} // end of namespace NS_NOMORE

#endif
