/***************************************************************************
                          cprogram.cpp  -  description
                             -------------------
    begin                : Mon Jul 21 2003
    copyright            : (C) 2003 by nomore-dg
    email                : nomore-dg@cs.uni-potsdam.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <algorithm>
#include <iostream>
#include "cprogram.h"
#include "misc.h"
#include "print.h"

namespace NS_NOMORE {

/***************************************************************************
  class CProgram
****************************************************************************/
CProgram::CProgram() : _body_counter(0) {
  CREATEOBJECT("CProgram");
}

CProgram::~CProgram() {

  for_each(this->_bodies.begin(), this->_bodies.end(), DerefDelete());
  for_each(this->_rules.begin(), this->_rules.end(), DerefDelete());
  for_each(this->_atoms.begin(), this->_atoms.end(), DerefDelete());

  this->_atoms.clear();
  this->_bodies.clear();
  this->_heads.clear();
  this->_rules.clear();  
  
  DELETEOBJECT("CProgram");  
}

bool CProgram::ReadBody(std::istream &f, long size, long size_neg, CRule* rule) {

  CHECK_POINTER("CProgram::ReadBody(rule)", rule);
  
  TId id2 = ++_body_counter;
  CBody *body = new CBody(id2, this);
  TId id = 0;
  
  for (long i = 0; i < size_neg; i++) {
    f >> id;
    if (!f.good() || id < 1) {
      PRINT_ERROR(cerr << "atom out of bounds" << endl);
      return false;
    }

    CAtom *atom = InsertAtom(id);
    body->InsertNegAtom(id);
  }

  long size_pos = size - size_neg;
  for (long i = 0; i < size_pos; i++) {
    f >> id;
    if (!f.good() || id < 1) {
      PRINT_ERROR(cerr << "atom out of bounds" << endl);
      return false;
    }
    
    CAtom *atom = InsertAtom(id);
    body->InsertPosAtom(id);
  }
  
  CBody* b = InsertBody(body);
  rule->SetBody(b);
  b->AddRule(rule);
  
  return true;
}

bool CProgram::AddBasicRule(std::istream &f) {
  TId head_id;
  f >> head_id;
  
  if (!f.good() || head_id < 1) {
    PRINT_ERROR(cerr << "head atom out of bounds" << endl);
    return false;
  } // if
  
  // handle head of rule
  CAtom *head = InsertHead(head_id);

  // generate new crule
  CRule *rule = new CRule(CIdGenerator::GetId());
  rule->SetHead(head);

  long body_size;
  f >> body_size;
  if (!f.good() || body_size < 0) {
    PRINT_ERROR(cerr << "body_size out of bounds" << endl);
    return false;
  } // if

  long neg_body_size;
  f >> neg_body_size;
  if (!f.good() || neg_body_size < 0 || neg_body_size > body_size) {
    PRINT_ERROR(cerr << "neg_body_size out of bounds" << endl);
    return false;
  } // if

  if (!ReadBody(f,body_size,neg_body_size,rule))
    return false;
  else
    InsertRule(rule);
    
  return true; 
}

bool CProgram::Read(std::istream &f) {

  if(!f) {
    PRINT_ERROR(cerr << "could not read from stream" << endl);
    return false;
  }

  // read rules
  int type;
  bool stop = false;
  while (!stop) {
    f >> type;
    switch (type) {
      case ENDRULE :
        stop = true;
        break;
      case BASICRULE:
        if (!AddBasicRule(f))
          return false;
        break;
      default:
        return false;
    } // switch
  } // while	

  // read atom names
  const int len = 1024;
  char name[len], space;
  TId id;
  f.getline(name, len);  // Get newline
  if (!f.good()) {
    PRINT_ERROR(cerr << "expected atom names to begin on new line" << endl);
    return false;
  } // if
  
  f >> id;
  f.get(space); // get rid of useless " "
  f.getline(name, len);
  while (id) {
    if (!f.good()) {
      PRINT_ERROR(cerr << "atom name too long or end of file" << endl);
      return false;
    } // if

    CAtom* atom = GetAtom(id);
    if (atom == NULL) {
      PRINT_ERROR(cerr << "found unused atom " << id << endl);
      return false;
    }
    
    atom->SetName(name);
    
    f >> id;
    f.get(space); // get rid of useless " "
    f.getline (name, len);
  } // while
      
  char s[len];
  // read B+
  if (!f.good () || strcmp (name, "B+")) {
    cerr << "B+ expected" << endl;
    return false;
  } // if

  f >> id;
  while (id) {
    if (!f.good () || id < 1) {
      cerr << "B+ atom out of bounds" << endl;
      return false;
    } // if

    CAtom* atom = GetAtom(id);
    if (atom != NULL) { 
      atom->SetColor(color_plus);
    } else {
      //PRINT_ERROR(cerr << "found unused atom" << id << endl);
      //return false;
    }

    f >> id;
  } // while

  f.getline (s, len);  // Get newline.
  // read B-
  f.getline (s, len);
  if (!f.good () || strcmp (s, "B-")) {
    cerr << "B- expected, line " << endl;
    return 1;
  } // if

  f >> id;
  while (id) {
    if (!f.good () || id < 1) {
      cerr << "B- atom out of bounds" << endl;
      return false;
    } // if

    CAtom* atom = GetAtom(id);
    if (atom != NULL) {
      atom->SetColor(color_minus);
    } else {
      //PRINT_ERROR(cerr << "found unused atom" << id << endl);
      //return false;
    }
    
    f >> id;
  } // while
  
  return true;
}

bool CProgram::operator!=(const CProgram& prg) const {
  return !((*this) == prg);
}

bool CProgram::operator==(const CProgram& prg) const {
  for(TRuleVectorIterator ite=_rules.begin();ite!=_rules.end();++ite) {
    TRuleVectorIterator beg = prg._rules.begin();
    TRuleVectorIterator end = prg._rules.end();
    for(;beg!=end;++beg) {
      if(*(*beg) == *(*ite))
        break;
    }
    
    if(beg==end)
      return false;
  }  
}

CAtom* CProgram::InsertAtom(TId id) {
  id--;
  if(id >= this->_atoms.size()) {
    _atoms.resize(id+1, static_cast<CAtom*>(NULL));
    _atoms[id] = new CAtom(id+1);
  } else 
  if(_atoms[id] == NULL) {
    _atoms[id] = new CAtom(id+1);
  }

  return _atoms[id];  
}

/*CRule* CProgram::GetRule(const TId id) {
  
  TRuleSet::iterator ite = _prules->find(id);

  if(ite != _prules->end())
    return (*ite).second;
  else
    return NULL;
    
}*/

CAtom* CProgram::GetAtom(const TId id) {
  TId i = id-1;
  if(i >= this->_atoms.size())
    return NULL;
  
  return _atoms[i];
}
  
CRule* CProgram::InsertRule(CRule* rule) {
  CHECK_POINTER("CProgram::InsertRule(rule)", rule);
    
  TId id = rule->GetId()-1;
  if(id >= this->_rules.size()) {
    _rules.resize(id+1, static_cast<CRule*>(NULL));
    _rules[id] = rule;
  } else 
  if(_rules[id] == NULL) {
    _rules[id] = rule;
  }
  
  return _rules[id];  
}

CAtom* CProgram::InsertHead(TId id) {
  TIdSetIterator it = _heads.find(id);
  if(it == _heads.end()) {
    CAtom *atom = GetAtom(id);
    if(atom == NULL) {
      atom = InsertAtom(id);
    }
    _heads.insert(it, id);
    return atom;
  }
  
  return GetAtom(id);
}

CBody* CProgram::InsertBody(CBody* body) {
  CHECK_POINTER("CProgram::InsertBody(body)", body);
  TnBodySetIterator it = _bodies.find(body);
  if(it == _bodies.end()) {
    _bodies.insert(it, body);
    return body;
  }
  
  delete body;
  return *it;
}
  
} //end of NS_NOMORE
