/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_GRAPH_H
#define LIBNOMORE_GRAPH_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif


#include <color.h>
#include <body_node.h>
#include <head_node.h>
#include <stack.h>
#include <event/channel_manager.h>
#include <util/hash_set.h>
#include <node.h>
#include <vector>
#include <cassert>
#include <iterator>
#include <limits>

namespace NS_NOMORE {

//! This class implements nomore's body-head graph. 
/*! The main responsibility of a graph is to store the set of head- and body nodes
 *	of a logic program (edges between the nodes are stored in the nodes not in 
 *  the graph). In addition a graph manages a 
 *  \ref BacktrackStack "backtracking stack" to be used to restore and recolor 
 *  choice points.  
 *	Furthermore a graph is the central place where event handlers register for
 *	specific events.
 *	\see _event
 *
 *	\note The Graph should be used as a static graph, i.e. you construct the graph
 *	in a seperate construction step and once this step is over
 *	you call the Graph's \ref Graph::finishConstruction "finishConstruction" 
 *	method to signal that the graph's structure is now immutable. */
class Graph {
  //! Struct for creating a hash of a head node.
  struct HashHeadNode {
    long operator()(const Node* n) const {return n->getId();}
  };

  //! Struct for creating a hash of a body node.
  struct HashBodyNode {
    long operator()(const Node* n) const {return std::numeric_limits<long>::max() - n->getId();}
  };

  //! Struct for comparing two nodes.
  struct EqualNode {
    long operator()(const Node* n1, const Node* n2) const {return n1->getId() == n2->getId();}
  };
  typedef bk_lib::hash_set<HeadNode*, HashHeadNode, EqualNode> HeadIndex;
  typedef bk_lib::hash_set<BodyNode*, HashBodyNode, EqualNode> BodyIndex;

public:
  //! Set of head nodes.
  typedef std::vector<HeadNode*> HeadNodeSet;
  typedef HeadNodeSet::size_type size_type;

  //! iterator type for iterating over the set of uncolored head nodes.
  typedef HeadIndex::const_iterator UncoloredHeadNodeIterator;
  
  //! iterator type for iterating over the set of uncolored body nodes.
  typedef BodyIndex::const_iterator UncoloredBodyNodeIterator;
  
  //! Iterator type that can be used to iterate over the graph's set of head nodes.
	/*! HeadNodeIterator models the bidirectional-iterator concept. 
   */
  class HeadNodeIterator : public std::iterator<std::bidirectional_iterator_tag, HeadNode*> {
		typedef std::vector<HeadNode*>::const_iterator const_iterator;
	
  public:
    typedef HeadNode* const& reference;

    //! Initializes the iterator by a begin and end iterator.
    /*! \param it The begin iterator of the HeadNodeSet.
     *  \param end The end iterator of the HeadNodeSet.
     */
    HeadNodeIterator(const_iterator it, const_iterator end) 
      : iterator_(it), endIterator_(end) {
      skipNullEntries();
    }

    
    //! returns the node pointed to by this iterator
    reference operator*() const {
      assert(*iterator_ != NULL);
      return *iterator_;
    }

    //! returns a pointer to the node pointed to by this iterator.
    HeadNode*const* operator->() const {
      assert(*iterator_ != NULL);
      return &*iterator_;
    }

    HeadNodeIterator& operator++() {
      ++iterator_;
      skipNullEntries();
      return *this;
    }

    HeadNodeIterator operator++(int) {
      HeadNodeIterator tmp = *this;
      ++*this;
      return tmp;
    }

    HeadNodeIterator& operator--() {
      while(!*--iterator_);
      return *this;
    }

    HeadNodeIterator operator--(int) {
      HeadNodeIterator tmp = *this;
      --*this;
      return tmp;
    }

    //! returns true if the two iterators refer to the same head node.
    bool operator==(const HeadNodeIterator& it) const {
      return it.iterator_ == iterator_;
    }

    bool operator!=(const HeadNodeIterator& it) const {
      return it.iterator_ != iterator_;
    }

  private:
    void skipNullEntries() {
      while (iterator_ != endIterator_ && !*iterator_)
        ++iterator_;
    }
    const_iterator iterator_;
    const_iterator beginIterator_;
    const_iterator endIterator_;
  };

  //! Set of body nodes.
  typedef std::vector<BodyNode*> BodyNodeSet;
  
  //! iterator type for iterating over the set of body nodes.
  typedef BodyNodeSet::const_iterator BodyNodeIterator;

  //! Creates an empty graph and starts its construction phase.
  Graph();

  //! Destroys the graph and all contained nodes.
  ~Graph();

  //! Colors a node.
  /*! Calls the method Node::setColor() of the node and fires a color event if 
   *  the node's color was changed.
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \return Returns true if the color has been changed and false otherwise.
	 *	\see Node::setColor
   *  \exception ColorError Throws this exception if a color error occured because 
   *                        of an invalid coloring.
	 *  \post A corresponding stack item was pushed on the backtrack-stack. */
  bool color(Node& n, Color::ColorValue c);

  //! Colors a node as choice point.
  /*! Calls the method Node::setColor() of the node and fires a color choice point 
   *  event if the node's color was changed.
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \param btc The choice-point's alternative color that is used when the choice-point
   *  is recolored 
   *  \return returns true if the color has been changed and false otherwise.
	 *	\see Node::setColor
   *  \see recolorChoicePoint
   *  \exception ColorError Throws this exception if a color error occured because 
   *                        of an invalid coloring.
	 *  \post A new branch was pushed on the backtrack-stack.
   */
  bool colorChoicePoint(Node& n, Color::ColorValue c, Color::ColorValue btc);

  //! Colors a node as choice point that can not be recolored.
  /*! Calls the method Node::setColor() of the node and fires a color choice point 
   *  event if the node's color was changed.
   *  \param n The node to be colored.
   *  \param c The new color for the node.
   *  \return returns true if the color has been changed and false otherwise.
	 *	\see Node::setColor
   *  \exception ColorError Throws this exception if a color error occured because 
   *                        of an invalid coloring.
	 *  \post A corresponding stack item was pushed on the backtrack-stack.
   */
  bool colorChoicePointStable(Node& n, Color::ColorValue c);

  //! Restores a node.
  /*! Calls the method Node::restoreColor() of the node and fires a restore event.
   *  \param n The node to be restored.
   *  \param c The new color of the node. Valid parameters are Color::ignore, 
   *           Color::none and Color::weak_plus.
	 *	\see Node::restoreColor 
   */
  void restore(Node& n, Color::ColorValue c);

  //! Creates and inserts a new body node.
  /*! \pre isConstructionFinished() returns false.
	 *  \return Returns a pointer to the newly created body node.
	 *	\attention The lifetime of the created body node is managed by the graph. */
  BodyNode* insertBodyNode();

  //! Creates and inserts a head node with the given id if necessary
  /*! \pre isConstructionFinished() returns false.
	 *  \return Returns a pointer to a head node with the given id.
	 *	\attention The lifetime of the created head node is managed by the graph. */
  HeadNode* insertHeadNode(long id);

  //! Returns the body node for the identifier.
  /*! \param id Identifier of the node to find.
   *  \return The node corresponding to the identifier. If the node does not exists 
   *          in the graph the method returns a null pointer. 
	 *	\complexity Constant time. */
  BodyNode* getBodyNode(long id) const;

  //! Checks if their exists a body node with a corresponding identifier in the body node set.
	/*! \complexity Constant time. */
  bool hasBodyNode(long id) const {
	  return getBodyNode(id) != 0;
  }

  //! Returns the head node for the identifier.
  /*! \param id Identifier of the node to find.
   *  \return The node corresponding to the identifier. If the node does not exists 
   *          in the graph the method returns a null pointer. 
	 *	\complexity Constant time. */
  HeadNode* getHeadNode(long id) const;

  //! Checks if their exists a head node with a corresponding identifier in the head node set.
	/*! \complexity Constant time. */
  bool hasHeadNode(long id) const {
	  return getHeadNode(id) != 0;
  }

	//! Checks if the graph is total colored.
  /*! That means all nodes (head and body nodes) are colored \ref Color::plus "plus",
   *  \ref Color::minus "minus" or \ref Color::ignore "ignore".
   *  \return True if the graph is total colored else false. */
  bool totalColoring() const {
    // is greater than zero by data type size_t
    // assert(countUncoloredNodes() >= 0);
    return countUncoloredNodes() == 0;
  }

  //! Deletes all nodes of this graph, clears the backtracking stack and starts a new construction phase.
  /*! \note Clearing a graph does not change the graph's
	 *  \ref event::ChannelManager "ChannelManager", i.e. connected handlers stay
	 *	connected. */
  void clear();

  //! Returns the number of nodes this graph contains.
	size_type countNodes() const {
	  return sizeHeadNodes_ + bodyNodes_.size();
  }

	//! Returns the number of uncolored nodes this graph contains.
  /*! \note A node counts as uncolored if its color is either Color::none or 
   *        Color::weak_plus. */
	size_type countUncoloredNodes() const {
	  return uncoloredHeads_.size() + uncoloredBodies_.size() + weakColoredNodes_;
  }

	//! Returns the number of head nodes this graph contains.
	size_type countHeadNodes() const {
		return sizeHeadNodes_;
	}
	
	//! Returns the number of body nodes this graph contains.
	size_type countBodyNodes() const {
		return bodyNodes_.size();
	}

  //! Returns the number of weak_plus colored nodes of the graph.
  size_type countWeakColoredNodes() const {
    return weakColoredNodes_;
  }

  //! restores the graph to the state it had just \b after the last choice point was colored.
	/*!
   *  I.e. Restores all nodes that were colored after the last choice point was colored but not the choice point itself.
   * \pre At least one choice point was previously colored.
	 *	\note For every node that is restored a corresponding Restore-Event is fired.
   *  \see BacktrackStack::restore() 
   */
  void restoreToChoicePoint();

  //!	restores the graph to the state it had just \b before the last choice point was colored.
	/*! 
   *  I.e. restores all nodes that were colored after the last choice point was colored
   *  as well as the last choice point itself.
   *  \note The branch that stored the consequences of the last choice point is discarded.
   *  \return True if the stack is restored successfully, false if the stack is empty. 
   */
  bool restoreChoicePoint();

  
  //!	first calls restoreLastChoicePoint then recolors the last choice point to its alternative color.	
	/*! Fires a color choice point event. 
   *  If recoloring the last choice points leads to an invalid coloring 
   *  the second last choice point is restored and recolored. This process is repeated
   *  until either the recoloring of a choice point is sucessful or there are
   *  no more choice points left. 
   *  
   *  \return True if the choice point is successfully recolored, false no choice point
   *  could be restored.
   */
  bool recolorChoicePoint();

  
  //! Returns an iterator that points to the first head node of this graph.
	/*! If this graph does not have any head nodes the end-iterator is returned.
	 *  \note Traversing the sequence of head nodes yields the nodes in ascending
	 *	      id-order. */
	HeadNodeIterator headNodesBegin() const {
    return HeadNodeIterator(headNodes_.begin(), headNodes_.end());
  }

  //! Returns an iterator that points just beyond the end of head nodes of this graph.
	HeadNodeIterator headNodesEnd() const {
    return HeadNodeIterator(headNodes_.end(), headNodes_.end());
  }

  //! Returns an iterator that points to the first head node of this graph.
	/*! If this graph does not have any head nodes the end-iterator is returned. */
	BodyNodeIterator bodyNodesBegin() const {
    return bodyNodes_.begin();
  }

  //! Returns an iterator that points just beyond the end of head nodes of this graph.
	BodyNodeIterator bodyNodesEnd() const {
    return bodyNodes_.end();
  }

  //! Returns the begin iterator of the uncolored body node set.
  /*!
   * \note if isUncoloredNodesUpdatesEnabled() returns true changing the color
   * of an uncolored node invalidates uncolored node iterators pointing to that 
   * node!.
   */
  UncoloredBodyNodeIterator uncoloredBodyNodesBegin() const {
    return uncoloredBodies_.begin();
  }
  
  //! Returns the end iterator of the uncolored body node set.
  UncoloredBodyNodeIterator uncoloredBodyNodesEnd() const {
    return uncoloredBodies_.end();
  }
  
  //! Returns the begin iterator of the uncolored head node set.
  /*! \note if isUncoloredNodesUpdatesEnabled() returns true changing the color
   * of an uncolored node invalidates uncolored node iterators pointing to that 
   * node!.
   */
  UncoloredHeadNodeIterator uncoloredHeadNodesBegin() const {
    return uncoloredHeads_.begin();
  }
  
  //! Returns the end iterator of the uncolored head node set.
  UncoloredHeadNodeIterator uncoloredHeadNodesEnd() const {
    return uncoloredHeads_.end();
  }

  //! Returns the first uncolored head node if exists else a null pointer.
  HeadNode* getFirstUncoloredHeadNode() const {
    return !uncoloredHeads_.empty() ? *uncoloredHeads_.begin() : static_cast<HeadNode*>(0);
  }
  
  //! Returns the first uncolored body node if exists else a null pointer.
  BodyNode* getFirstUncoloredBodyNode() const {
    return !uncoloredBodies_.empty() ? *uncoloredBodies_.begin() : static_cast<BodyNode*>(0);
  }

  //! Returns the \ref event::ChannelManager "ChannelManager" of this graph.
	/*! Interested Event handlers need to register with this 
	 *  \ref event::ChannelManager "ChannelManager" in order to receive events
	 *  concerning this graph. */
	event::ChannelManager& getEventManager() {
    return eventManager_;
  }

	//! Ends the construction phase and fires a GraphConstructionFinished event.
	/*! After calling this method the graph's structure is immutable and
	 *	calls to the insert-methods are prohibited. */
	void finishConstruction();

	//! Returns true, if the graph's construction is completed.
	/*! If this method returns true, the graph's structure can't change anymore.
	 *	If this method returns false, the graph's structure is still volatile. */
	bool isConstructionFinished() const;

  //! Enables the uncolored nodes update mode.
  /*!
   * In this mode the graph upates its uncolored nodes set automatically whenever 
   * a node is colored.
   */
  void enableUncoloredNodesUpdates();

  //! Disables the uncolored nodes update mode.
  /*!
   * In this mode the graph does not update its uncolored node set. This mode
   * is useful if an operation temporarily needs to color uncolored nodes (restoring
   * their color to Color::none upon completion)
   * but needs valid uncolored node iterators at the same time.
   * See the lookahead-operation for an example.
   *
   */
  void disableUncoloredNodesUpdates();

  //! Returns true if the graph currently updates its uncolored node set automatically.
  bool isUncoloredNodesUpdatesEnabled() const;

  //! totalizes the \ref NS_NOMORE::Graph "graph".
  /*! 
   *  totalize colors all uncolored nodes to Color::minus. 
   *
   *  \note 
   *  This operation is reasonable only if no operator is able to color the 
   *  uncolored nodes.
   *  For example the \ref ForwardPropagator "forward propagator" and
   *  \ref BackwardPropagator "backward propagator" can not always decide whether a 
   *  positve (zero) loop must be colored \ref Color::plus "plus" or 
   *  \ref Color::minus "minus".
   *
   *  \note totalize was formerly implemented as the Post-Operator N.
   *
   *	\note This operator does not verify if the resulting coloring is valid.
   *  This needs to be done in a seperate propagation step (for example using
   *	the \ref ForwardPropagator "forward propagator")
   *
   *  \throw ColorError A ColorError-exception is thrown if the graph contains
   *  Color::weak_plus colored nodes.
   */
  bool totalize();

private:
  // graphs currently are not copyable.
  Graph(const Graph& g);
  Graph& operator=(const Graph& g);
  
  void colorChanged(Node& n, Color::ColorValue old);
  inline void eraseUncolored(Node&);
  inline void insertUncolored(Node&);
  
  //! state of the construction phase.
	bool constructionFinished_;

  //! Set of body nodes.
  BodyNodeSet bodyNodes_;

  //! Set of head nodes.
  HeadNodeSet headNodes_;

	//! Number of Head Nodes in the HeadNodeSet
	size_type sizeHeadNodes_;

  long nextBodyId_;

  bool modifyUncoloredListEnabled_;

  //! Backtracks stack used to restore nodes of this graph.
  BacktrackStack backtrackingStack_;

	//! ChannelManager that manages all event channels needed by this graph
	//! and its nodes.
	event::ChannelManager eventManager_;
  
  HeadIndex uncoloredHeads_;
  BodyIndex uncoloredBodies_;
  size_type weakColoredNodes_;
  
};

//! Output operator for printing the graph to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const Graph& grp);

//! Output operator for printing simple head node to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const HeadNode& head);

//! Output operator for printing simple body node to stdout or to a file.
std::ostream& operator<<(std::ostream& os, const BodyNode& body);

//! Event type fired to signal that the construction of a graph is completed.
/*!  A graph fires this event when it receives the \ref Graph::finishConstruction 
 *   "finishConstruction" message to signal that the graph's structure is
 *   now stable and won't change anymore. */
struct GraphConstructionFinished {
	explicit GraphConstructionFinished(Graph& theGraph)
		: theGraph_(&theGraph) {
	}
	Graph* theGraph_;
};

} // NS_NOMORE

#endif
