/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_OBJECT_FACTORY_H
#define LIBNOMORE_OBJECT_FACTORY_H

#include <string>
#include <stdexcept>
#include <map>
#include <vector>
#include <memory>

namespace NS_NOMORE {

class Operator;
class Graph;
class Heuristic;
class OperatorDecoratorFactory;

//! Factory that maps logical operator (and heuristic) names to their C++ classes.
/*!
 * This Factory can be used to create operators and heuristic by only specifying
 * their logical name.
 * A DecoratorFactory can be installed to decorate each constructed operator.
 *
 */
class ObjectFactory {
public:
  //! creates a new factory object
  ObjectFactory();
  
  /*!
   * creates a new object that decorates requested operators using the given
   * DecoratorFactory
   * \pre df points to a dynamically allocated object
   * \post The factory owns the DecoratorFactory
   */
  ObjectFactory(OperatorDecoratorFactory* df);
  
  /*!
   * creates a new object that decorates requested operators using the given
   * DecoratorFactory
   * \note the factory only stores a reference to the DecoratorFactory but
   * does not take over ownership. 
   */
  ObjectFactory(OperatorDecoratorFactory& df);
  ~ObjectFactory();
  
  //! object types this factory can create
  enum ObjectType {CHOICE_OP, DET_OP, LOOKAHEAD_OP, HEURISTIC};
  
  //! type used as result type for the factory's createOperators operation
  struct Operators {
    ~Operators();
    Operators()
    {}
    Operators(const Operators& other)
      : detOp_(other.detOp_.release())
      , choiceOp_(other.choiceOp_.release())
    {}
    
    mutable std::auto_ptr<Operator>       detOp_;
    mutable std::auto_ptr<Operator>       choiceOp_;
  };
  
  //! creates the operators described in objStr to be used with the given Graph g.
  /*!
   * \param g The graph the operators should be applied on.
   * \param objStr a string containing the names of the to be created operators.
   * objStr must have the following format: ChoiceOp:DetOp:LOp[DetOp]:Heuristic
   * where:
   *  - ChoiceOp is the name of the requested Choice operator, e.g. D
   *  - DetOp is the requested set of propagation operators
   *  - LOp[DetOp] is the name of the requested Lookahead Operator followed
	 *  by the set of propagation operators the Lookahead Operator should use
	 *  enclosed in square brackets
   * - Heuristic is the name of the requested heuristic the Choice operator should use
   * .
   *
   * \param validate if true, then this method checks whether the created operators can
   * be used together.
   *
   * \return An object of type Operators that contains the requested operators.
   *
   * \note 
   * - ChoiceOp and DetOp are required.
	 * - All other parts may be set to "None" if not needed.
   * - DetOp must be specified in a format that is understood by the \ref OperatorStringParser "OperatorStringParser class".
   *
   * \throw FormatError if objStr does not have a valid format.
   * \throw UnknownObject if objStr contains an unknown name
   * \throw InvalidOperatorCombination if validate is true and objStr contains
   * an invalid operator combination.
   * \throw OperatorStringParser::SyntaxError if the DetOp part of objStr is invalid.
   * 
   */
  Operators createOperators(Graph& g, const std::string& objStr, bool validate = true);
  
  //! creates the operator with the name name.
  /*!
   * \throw UnknownObject if name is not a known operator name
   */
  std::auto_ptr<Operator> createOperator(Graph& g, const std::string& name) const;
  
  //! creates the choice operator with the name name using the heuristic with the name heu
  /*!
   * \throw UnknownObject if name is not a known choice operator or heu is not a known heuristic
   */
  std::auto_ptr<Operator> createChoiceOperator(Graph& g, const std::string& name, const std::string& heu) const;

  //! creates the lookahead operator with the name name.
  /*!
   * \throw UnknownObject if name is not a known operator name
   */
  std::auto_ptr<Operator> createLookaheadOperator(Graph& g, const std::string& name) const;
  
  //! creates the heuristic with the name name
  /*!
   * \throw UnknownObject if name is not a known heuristic
   */
  std::auto_ptr<Heuristic> createHeuristic(Graph& g, const std::string& name) const;

  //! returns the names of the objects of type ot known to this factory.
  static std::vector<std::string> getKnownObjects(ObjectType ot);
private:
  std::auto_ptr<Operator> addLookahead(Graph& g,
                                       std::auto_ptr<Operator> detOp, 
                                       const std::string& lopStr,
                                       const std::string& detOpStr);
  ObjectFactory(const ObjectFactory&);
  ObjectFactory& operator=(const ObjectFactory&);
  
  OperatorDecoratorFactory* opDecorator_;
  bool owner_;
  typedef std::map<std::string, std::auto_ptr<Operator> (*)(Graph&)> OpMap;
  typedef std::map<std::string, std::auto_ptr<Operator> (*)(Graph&, Heuristic* h)> ChoiceOpMap;
  typedef std::map<std::string, std::auto_ptr<Heuristic> (*)()> HeuristicsMap;

  static OpMap initOpMap();
  static OpMap initLookaheadOpMap();
  static ChoiceOpMap initChoiceOpMap();  
  static HeuristicsMap initHeuristicsMap();
  
  static OpMap operators_s;
  static OpMap lookaheadOps_s;
  static ChoiceOpMap choiceOps_s;
  static HeuristicsMap heuristics_s;
};

//! Exception type thrown if the factory is requested to create an unknown object.
class UnknownObject : public std::runtime_error {
public:
  UnknownObject(const std::string& name, ObjectFactory::ObjectType t)
    : std::runtime_error(name)
    , ot_(t) {

  }
  //! returns the type of the unknown object
  /*!
   * \see ObjectFactory::ObjectType
   */
  ObjectFactory::ObjectType getObjectType() const {
    return ot_;
  }
private:
  ObjectFactory::ObjectType ot_;
};

//! Exception type thrown when a string has an invalid/unexpected format
class FormatError : public std::runtime_error {
public:
  FormatError(const std::string& desc)
    : std::runtime_error(desc) {

  }
};

//! Exception type thrown when an operator string contains an invalid operator combination
class InvalidOperatorCombination : public std::runtime_error {
public:
  InvalidOperatorCombination() 
    : std::runtime_error("invalid operator combination") {
  }
};

}

#endif
