/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#define for if (0); else for
#endif

#include <graph.h>
#include <operators/lookahead.h>
#include "detail/functors.h"
#include <algorithm>

namespace NS_NOMORE {

Graph::Graph()
  : constructionFinished_(false)
	, sizeHeadNodes_(0)
  , nextBodyId_(std::numeric_limits<long>::max()) 
  , modifyUncoloredListEnabled_(true)
  , weakColoredNodes_(0) {
}

Graph::~Graph() {
  clear();
}

void Graph::finishConstruction() {
	if (!constructionFinished_) {
		constructionFinished_ = true;
		GraphConstructionFinished e(*this);
		eventManager_.dispatch(e);
	}
}

bool Graph::isConstructionFinished() const {
	return constructionFinished_;
}

BodyNode* Graph::insertBodyNode() {
  assert(!isConstructionFinished());
  BodyNode* nb = new BodyNode(nextBodyId_--);
  bodyNodes_.push_back(nb);
  uncoloredBodies_.insert(nb);
  return nb;
}

HeadNode* Graph::insertHeadNode(long id) {
  assert(!isConstructionFinished());
	assert(id >= 0);
  assert(id < nextBodyId_);
  HeadNode* node = getHeadNode(id);
  if (!node) {
    node = new HeadNode(id);
    if ((size_type)id >= headNodes_.size())
      headNodes_.resize(id + 1, static_cast<HeadNode*>(0));
      
    headNodes_[id] = node;
    ++sizeHeadNodes_;
    uncoloredHeads_.insert(node);
  }
  return node;
}

BodyNode* Graph::getBodyNode(long id) const {
  unsigned index = std::numeric_limits<long>::max() - id;
  if (index < bodyNodes_.size())
    return bodyNodes_[index];
  return 0;
}

HeadNode* Graph::getHeadNode(long id) const {
  assert(id >= 0);
  return (size_type)id < headNodes_.size() ? headNodes_[id] : 0;
}

void Graph::clear() {
  std::for_each(bodyNodes_.begin(), bodyNodes_.end(), DETAIL::DeleteObject());
  std::for_each(headNodes_.begin(), headNodes_.end(), DETAIL::DeleteObject());
  headNodes_.clear();
  bodyNodes_.clear();
  nextBodyId_ = std::numeric_limits<long>::max();
	sizeHeadNodes_ = 0;
	constructionFinished_ = false;
  uncoloredBodies_.clear();
  uncoloredHeads_.clear();
  weakColoredNodes_ = 0;
  modifyUncoloredListEnabled_ = true;
}

void Graph::restoreToChoicePoint() {
  backtrackingStack_.restore(*this);
}


void Graph::restore(Node& n, Color::ColorValue c) {
  Color::ColorValue old = n.getColor();
  n.restoreColor(c);
  colorChanged(n, old);
  n.fireRestoreEvent(eventManager_, old);  
}

bool Graph::color(Node& n, Color::ColorValue c) {
  Color::ColorValue oldColor = n.getColor();
  if (n.setColor(c)) {
    colorChanged(n, oldColor);
    backtrackingStack_.pushItem(BacktrackStack::Item(n, oldColor));
    n.fireColorEvent(eventManager_, oldColor);
    return true;
  }
  return false;
}

bool Graph::colorChoicePoint(Node& n, Color::ColorValue c, Color::ColorValue btc) {
  assert(n.getColor() == Color::none);
  Color::ColorValue oldColor = n.getColor();
  if (n.setColor(c)) {
    colorChanged(n, oldColor);
    backtrackingStack_.pushBranch(BacktrackStack::Item(n, oldColor), btc);
    n.fireColorChoicePointEvent(eventManager_, oldColor, false);
    return true;
  }
  return false;
}

bool Graph::colorChoicePointStable(Node& n, Color::ColorValue c) {
  assert(n.getColor() == Color::none);
  Color::ColorValue oldColor = n.getColor();
  if (n.setColor(c)) {
    colorChanged(n, oldColor);
    backtrackingStack_.pushItem(BacktrackStack::Item(n, oldColor));
    n.fireColorChoicePointEvent(eventManager_, oldColor, true);
    return true;
  }
  return false;
}

bool Graph::recolorChoicePoint() {

  while(!backtrackingStack_.isEmpty()) {
  
    BacktrackStack::Item cp = backtrackingStack_.restoreAndPop(*this);
    try {
      return colorChoicePointStable(*cp.node_, cp.color_);
    } catch(...) {
    }
  }
  return false;
}

bool Graph::restoreChoicePoint() {

  if (!backtrackingStack_.isEmpty()) {
    backtrackingStack_.restoreAndPop(*this);
    return true;
  }
  return false;
}

std::ostream& operator<<(std::ostream& os, const Graph& grp) {
  os << "HeadNodes [Count=" << (unsigned int)grp.countHeadNodes() << "]" << std::endl;
  for(Graph::HeadNodeIterator it = grp.headNodesBegin(); it != grp.headNodesEnd(); ++it) {
    HeadNode *head = *it;
    os << "  " << *head << " [headid="<<head->getId()<<"]" <<std::endl;

    os << "    Predecessors" << std::endl;
    for(HeadNode::BodyNodeIterator itBody = head->predecessorsBegin(); 
      itBody != head->predecessorsEnd(); ++itBody) {
        
      os << "      " << *(*itBody) << std::endl;
    }

    os << "    0-Successors" << std::endl;
    for(HeadNode::BodyNodeIterator itBody = head->zeroSuccessorsBegin(); 
      itBody != head->zeroSuccessorsEnd(); ++itBody) {
        
      os << "      " << *(*itBody) << std::endl;
    }

    os << "    1-Successors" << std::endl;
    for(HeadNode::BodyNodeIterator itBody = head->oneSuccessorsBegin(); 
      itBody != head->oneSuccessorsEnd(); ++itBody) {
        
      os << "      " << *(*itBody) << std::endl;
    }
  }

  os << "BodyNodes [Count=" << (unsigned int)grp.countBodyNodes() << "]" << std::endl;
  for(Graph::BodyNodeIterator it = grp.bodyNodesBegin(); it != grp.bodyNodesEnd(); ++it) {
    BodyNode *body = *it;
    os << "  " << *body << " [bodyid="<<body->getId()<<"]" <<std::endl;

    os << "    Successors" << std::endl;
    for(BodyNode::HeadNodeIterator itHead = body->successorsBegin(); 
      itHead != body->successorsEnd(); ++itHead) {
        
      os << "      " << *(*itHead) << std::endl;
    }

    os << "    0-Predecessors" << std::endl;
    for(BodyNode::HeadNodeIterator itHead = body->zeroPredecessorsBegin(); 
      itHead != body->zeroPredecessorsEnd(); ++itHead) {
        
      os << "      " << *(*itHead) << std::endl;
    }

    os << "    1-Predecessors" << std::endl;
    for(BodyNode::HeadNodeIterator itHead = body->onePredecessorsBegin(); 
      itHead != body->onePredecessorsEnd(); ++itHead) {
        
      os << "      " << *(*itHead) << std::endl;
    }
  }

  os << "corresponding Rules " << std::endl;
  for(Graph::BodyNodeIterator it = grp.bodyNodesBegin(); it != grp.bodyNodesEnd(); ++it) {
    BodyNode *body = *it;
        
    for(BodyNode::HeadNodeIterator itHead = body->successorsBegin(); 
      itHead != body->successorsEnd(); ++itHead) {
        
      os << "  " << *(*itHead) << " :- " << *body << "." << std::endl;
    }
  }

  return os;
}

void Graph::enableUncoloredNodesUpdates() {
  modifyUncoloredListEnabled_ = true;
}

bool Graph::isUncoloredNodesUpdatesEnabled() const {
  return modifyUncoloredListEnabled_;
}

void Graph::disableUncoloredNodesUpdates() {
  modifyUncoloredListEnabled_ = false;
}

void Graph::colorChanged(Node& n, Color::ColorValue oldColor) {
  Color::ColorValue newColor = n.getColor();
  assert(oldColor != newColor);
  weakColoredNodes_ += (newColor == Color::weak_plus) - (oldColor == Color::weak_plus);
  if (modifyUncoloredListEnabled_) {
    if (oldColor == Color::none)
      eraseUncolored(n);
    else if (newColor == Color::none)
      insertUncolored(n);
  }
}


void Graph::eraseUncolored(Node& n) { 
  if (n.getId() < nextBodyId_) {
    uncoloredHeads_.erase(static_cast<HeadNode*>(&n));
  }
  else {
    uncoloredBodies_.erase(static_cast<BodyNode*>(&n));
  }
}

void Graph::insertUncolored(Node& n) {
  if (n.getId() < nextBodyId_) {
    uncoloredHeads_.insert(static_cast<HeadNode*>(&n));
  }
  else {
    uncoloredBodies_.insert(static_cast<BodyNode*>(&n));
  }
}

std::ostream& operator<<(std::ostream& os, const HeadNode& head) {
  os << head.getAtom().getName();  
  return os;
}

std::ostream& operator<<(std::ostream& os, const BodyNode& body) {  

  int count = 0;
  for(BodyNode::HeadNodeIterator it = body.zeroPredecessorsBegin();
    it != body.zeroPredecessorsEnd(); ++it) {

    os << *(*it) << ", ";
    count++;
  }

  for(BodyNode::HeadNodeIterator it = body.onePredecessorsBegin();
    it != body.onePredecessorsEnd(); ++it) {

    os << "not " << *(*it) << ", ";
    count++;
  }

  // only for bodies without preconditions
  if(count == 0) 
    os << "{}";

  return os;
}

bool Graph::totalize() {
  if (weakColoredNodes_) {
    throw ColorError();
  } 
  disableUncoloredNodesUpdates();
  struct Local {
    Graph* g;
    ~Local() {
      g->enableUncoloredNodesUpdates();
    }
  } l = {this};
  (void) l;
  if (countUncoloredNodes()) {
    BodyIndex::const_iterator end = uncoloredBodies_.end();
    for (BodyIndex::const_iterator it = uncoloredBodies_.begin(); it != end; ++it) {
      color(**it, Color::minus);
    }
    HeadIndex::const_iterator hEnd = uncoloredHeads_.end();
    for (HeadIndex::const_iterator it = uncoloredHeads_.begin(); it != hEnd; ++it) {
      color(**it, Color::minus);
    }
    uncoloredBodies_.clear();
    uncoloredHeads_.clear();
    return true;
  }
  return false;
}

} // NS_NOMORE
