/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <operators/unfounded_set.h>
#include <graph.h>
#if defined (_MSC_VER) && _MSC_VER <= 1300
#define for if(0); else for
#endif

namespace NS_NOMORE {

UnfoundedSetOperator::UnfoundedSetOperator(Graph& grp) 
  : GraphOperator(UnfoundedSetOperator::getOpName(), grp) {
  event::ChannelManager &em = grp.getEventManager();

  em.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
  em.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);
  em.getChannelFor(event::Event<BodyNodeColoredChoicePoint>()).connect(*this);
  em.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).connect(*this);
}

///////////////////////////////////////////////////////////////////////////////
// common interface
///////////////////////////////////////////////////////////////////////////////
bool UnfoundedSetOperator::execute() {
  addReachableNodes();
  removePossiblySupportedNodes();
  return colorCandidates();
}


void UnfoundedSetOperator::postExecute() {
  bodyNodes_.clear();
  headNodes_.clear();
  candidateSet_.clear();
  
}

bool UnfoundedSetOperator::validate(const OpSet&) const {
  return true;
}

///////////////////////////////////////////////////////////////////////////////
// event handlers
///////////////////////////////////////////////////////////////////////////////
void UnfoundedSetOperator::handle(const HeadNodeColored& e) {
  if(e.node_->getColor() == Color::minus)
    headNodes_.push_back(e.node_);
}

void UnfoundedSetOperator::handle(const BodyNodeColored& e) {
  if(e.node_->getColor() == Color::minus)
    bodyNodes_.push_back(e.node_);
}

void UnfoundedSetOperator::handle(const HeadNodeColoredChoicePoint& e) {
  if(e.node_->getColor() == Color::minus)
    headNodes_.push_back(e.node_);
}

void UnfoundedSetOperator::handle(const BodyNodeColoredChoicePoint& e) {
  if(e.node_->getColor() == Color::minus)
    bodyNodes_.push_back(e.node_);
}

///////////////////////////////////////////////////////////////////////////////
// private helpers
///////////////////////////////////////////////////////////////////////////////
// step 1: add all nodes reachable from one of the minus colored nodes
void UnfoundedSetOperator::addReachableNodes() {
  HeadNodes headsToCheck;
  BodyNodes bodiesToCheck;
  do {
    addReachableNodes(headNodes_, headsToCheck);
    addReachableNodes(bodyNodes_, bodiesToCheck);
  } while (!headNodes_.empty());

  headsToCheck.swap(headNodes_);
  bodiesToCheck.swap(bodyNodes_);
}

// step 2: remove those nodes that are possibly supported.
void UnfoundedSetOperator::removePossiblySupportedNodes() {
  do {
    removePossiblySupportedNodes(headNodes_);
    removePossiblySupportedNodes(bodyNodes_);
  } while (!headNodes_.empty());
}

// step 3: color the remaining nodes to minus
bool UnfoundedSetOperator::colorCandidates() {
  bool changed = false;
  NodeSet::const_iterator end = candidateSet_.end();
  Graph& g = getGraph();
  for (NodeSet::const_iterator it = candidateSet_.begin(); it != end; ++it) {
    changed |= g.color(**it, Color::minus);
  }
  return changed;
}


void UnfoundedSetOperator::addSuccessors(const HeadNode& n) {
	std::copy(n.zeroSuccessorsBegin(), n.zeroSuccessorsEnd(),
    std::back_inserter(bodyNodes_));
}

void UnfoundedSetOperator::addSuccessors(const BodyNode& n) {
	std::copy(n.successorsBegin(), n.successorsEnd(),
    std::back_inserter(headNodes_));
}

bool UnfoundedSetOperator::isPossiblySupported(HeadNode* node) const {
  if (node->getColor() == Color::minus || candidateSet_.find(node) == candidateSet_.end())
    return false;

  HeadNode::BodyNodeIterator end = node->predecessorsEnd();
  for(HeadNode::BodyNodeIterator it = node->predecessorsBegin(); it != end; ++it) {
    // is this pred in the candidates or minus colored?
    if((*it)->getColor() != Color::minus && candidateSet_.find(*it) == candidateSet_.end()) {
      return true;
    }
  } // for
  return false;
}

bool UnfoundedSetOperator::isPossiblySupported(BodyNode* node) const {
  if (node->getColor() == Color::minus || candidateSet_.find(node) == candidateSet_.end())
    return false;
  
  BodyNode::HeadNodeIterator end = node->zeroPredecessorsEnd();
  for(BodyNode::HeadNodeIterator it = node->zeroPredecessorsBegin(); it != end; ++it) {
    
    // is pred in candidates?
    if(candidateSet_.find(*it) != candidateSet_.end()) 
      return false;
  } 

  // if no pred is in head_set!
  return true;
}

}
