#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/forward_prop.h>
#include <graph.h>
#include <head_node.h>
#include <body_node.h>
#include "test_common.h"

using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestForwardProp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestForwardProp);
  CPPUNIT_TEST(testBodyPropPlus);
  CPPUNIT_TEST(testBodyPropWeakPlus);
  CPPUNIT_TEST(testBodyPropMinus);
  CPPUNIT_TEST(testHeadPropPlusSupported);
  CPPUNIT_TEST(testHeadPropWeakPlusSupported);
  CPPUNIT_TEST(testHeadPropPlusBlocked);
  CPPUNIT_TEST(testHeadPropWeakPlusBlocked);
  CPPUNIT_TEST(testHeadPropMinusUnsupported);
  CPPUNIT_TEST(testHeadPropMinusUnblocked);
	CPPUNIT_TEST_SUITE_END();

public:
  void setUp() {
    head1 = new HeadNode(1);
    head2 = new HeadNode(2);
    body = new BodyNode(3);
    body->insertSuccessor(*head1);
    body->insertSuccessor(*head2);
    
    grp = new Graph();
    p = new ForwardPropagator(*grp);
  }

  void tearDown() {
    delete head1;
    delete head2;
    delete body;
    delete grp;
    delete p;
  }

	void testBodyPropPlus() {
    grp->color(*body, Color::plus);
    //p->handle(BodyNodeColored(body, Color::plus));
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::plus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::plus, head2->getColor() );
	}

	void testBodyPropWeakPlus() {
    grp->color(*body, Color::weak_plus);
    //p->handle(BodyNodeColored(body, Color::weak_plus));
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, head2->getColor() );
	}

	void testBodyPropMinus() {
    BodyNode body2(4);
    body2.insertSuccessor(*head2);
    
    grp->color(*body, Color::minus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::minus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );

    grp->color(body2, Color::minus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::minus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head2->getColor() );
	}

  void testHeadPropPlusSupported() {
    head1->insertZeroSuccessor(*body);
    head2->insertZeroSuccessor(*body);

    grp->color(*head1, Color::plus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::none, body->getColor() );

    grp->color(*head2, Color::plus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::plus, body->getColor() );
  }

  void testHeadPropWeakPlusSupported() {
    head1->insertZeroSuccessor(*body);
    head2->insertZeroSuccessor(*body);

    grp->color(*head1, Color::weak_plus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::none, body->getColor() );

    grp->color(*head2, Color::weak_plus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::weak_plus, body->getColor() );
  }

  void testHeadPropWeakPlusBlocked() {
    BodyNode body2(5);
    head1->insertOneSuccessor(body2);

    grp->color(*head1, Color::weak_plus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2.getColor() );
  }

  void testHeadPropPlusBlocked() {
    BodyNode body2(5);
    head1->insertOneSuccessor(body2);

    grp->color(*head1, Color::plus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2.getColor() );
  }

  void testHeadPropMinusUnsupported() {
    head1->insertZeroSuccessor(*body);
    head2->insertZeroSuccessor(*body);

    grp->color(*head1, Color::minus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::minus, body->getColor() );
  }

  void testHeadPropMinusUnblocked() {
    BodyNode body2(100);
    head1->insertOneSuccessor(body2);
    head2->insertOneSuccessor(body2);

    grp->color(*head1, Color::minus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::none, body2.getColor() );

    grp->color(*head2, Color::minus);
    (*p)();
    CPPUNIT_ASSERT_EQUAL( Color::plus, body2.getColor() );
  }

private:
  HeadNode* head1;
  HeadNode* head2;
  BodyNode* body;
  Graph* grp;
  ForwardPropagator* p;
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestForwardProp);

}	// end namespace NS_NOMORE_TESTS
