#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include "test_common.h"
#include <head_node.h>
#include <body_node.h>
#include <event/channel_manager.h>
using namespace std;
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

// Unsere Testklasse ist selbst eine HeadNode
class TestHeadNode : public CppUnit::TestFixture, public HeadNode {
public:
  TestHeadNode()
    : HeadNode(1) {
  }

  CPPUNIT_TEST_SUITE(TestHeadNode);

  CPPUNIT_TEST(testInit);

  CPPUNIT_TEST(testEmptyIsUnsupported);
  CPPUNIT_TEST(testIsUnsupportedByOne);
  CPPUNIT_TEST(testIsUnsupported);

  CPPUNIT_TEST(testInsertPredecessor);
  CPPUNIT_TEST(testInsertZeroSuccessor);
  CPPUNIT_TEST(testInsertOneSuccessor);
  CPPUNIT_TEST(testDoSetColorNotifiesPreds);
  CPPUNIT_TEST(testDoSetColorNotifiesZeroSuccs);
  CPPUNIT_TEST(testDoSetColorNotifiesOneSuccs);
  CPPUNIT_TEST(testDoRestoreColorNotifiesPreds);
  CPPUNIT_TEST(testDoRestoreColorNotifiesZeroSuccs);
  CPPUNIT_TEST(testDoRestoreColorNotifiesOneSuccs);
	
	CPPUNIT_TEST(testFireColorEvent);
	CPPUNIT_TEST(testFireRestoreEvent);
  CPPUNIT_TEST_SUITE_END();

  void testInit() {
    CPPUNIT_ASSERT(predecessorsBegin() == predecessorsEnd());
    CPPUNIT_ASSERT(zeroSuccessorsBegin() == zeroSuccessorsEnd());
    CPPUNIT_ASSERT(oneSuccessorsBegin() == oneSuccessorsEnd());
  }

  
  void testEmptyIsUnsupported() {
    CPPUNIT_ASSERT_EQUAL(true, isUnsupported());
  }

  void testIsUnsupportedByOne() {
    BodyNode b(1);
    insertPredecessor(b);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());

    predecessorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(true, isUnsupported());

    predecessorRestored(Color::minus, Color::none);

    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());
  }

  void testIsUnsupported() {
    BodyNode b1(1), b2(2);
    insertPredecessor(b1);
    insertPredecessor(b2);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());

    predecessorColored(Color::none, Color::minus);
    predecessorColored(Color::none, Color::plus);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());

    predecessorRestored(Color::plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());

    predecessorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(true, isUnsupported());

    predecessorRestored(Color::minus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());
  }

  
  void testInsertPredecessor() {
    FakeBodyNode b(1);
    insertPredecessor(b);
    CPPUNIT_ASSERT(findNode(predecessorsBegin(), predecessorsEnd(), b));

    FakeBodyNode b2(2);
    insertPredecessor(b2);
    CPPUNIT_ASSERT(findNode(predecessorsBegin(), predecessorsEnd(), b));
  }

  void testInsertZeroSuccessor() {
    FakeBodyNode b1(1);
    insertZeroSuccessor(b1);
    CPPUNIT_ASSERT(findNode(zeroSuccessorsBegin(), zeroSuccessorsEnd(), b1));
    CPPUNIT_ASSERT(findNode(b1.zeroPredecessorsBegin(), b1.zeroPredecessorsEnd(), *this));
    FakeBodyNode b2(2);
    insertZeroSuccessor(b2);
    CPPUNIT_ASSERT(findNode(zeroSuccessorsBegin(), zeroSuccessorsEnd(), b2));
    CPPUNIT_ASSERT(findNode(b2.zeroPredecessorsBegin(), b2.zeroPredecessorsEnd(), *this));
  }

  void testInsertOneSuccessor() {
    FakeBodyNode b1(1);
    insertOneSuccessor(b1);
    CPPUNIT_ASSERT(findNode(oneSuccessorsBegin(), oneSuccessorsEnd(), b1));
    CPPUNIT_ASSERT(findNode(b1.onePredecessorsBegin(), b1.onePredecessorsEnd(), *this));
    FakeBodyNode b2(2);
    insertOneSuccessor(b2);
    CPPUNIT_ASSERT(findNode(oneSuccessorsBegin(), oneSuccessorsEnd(), b2));
    CPPUNIT_ASSERT(findNode(b2.onePredecessorsBegin(), b2.onePredecessorsEnd(), *this));
  }

  void testDoSetColorNotifiesPreds() {
    FakeBodyNode b1(1), b2(2);
    insertPredecessor(b1);
    insertPredecessor(b2);
    callDoSetColor(Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, b1.successorColoredCalled(Color::none, Color::plus));
    CPPUNIT_ASSERT_EQUAL(true, b2.successorColoredCalled(Color::none, Color::plus));
  }

  void testDoSetColorNotifiesZeroSuccs() {
    FakeBodyNode b1(1), b2(2);
    insertZeroSuccessor(b1);
    insertZeroSuccessor(b2);
    callDoSetColor(Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, b1.zeroPredColoredCalled(Color::none, Color::plus));
    CPPUNIT_ASSERT_EQUAL(true, b2.zeroPredColoredCalled(Color::none, Color::plus));
  }

  void testDoSetColorNotifiesOneSuccs() {
    FakeBodyNode b1(1), b2(2);
    insertOneSuccessor(b1);
    insertOneSuccessor(b2);
    callDoSetColor(Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, b1.onePredColoredCalled(Color::none, Color::plus));
    CPPUNIT_ASSERT_EQUAL(true, b2.onePredColoredCalled(Color::none, Color::plus));
  }

  void testDoRestoreColorNotifiesPreds() {
    callDoSetColor(Color::plus);
    FakeBodyNode b1(2), b2(3);
    insertPredecessor(b1);
    insertPredecessor(b2);
    callDoRestoreColor(Color::none);
    CPPUNIT_ASSERT_EQUAL(true, b1.successorRestoredCalled(Color::plus, Color::none));
    CPPUNIT_ASSERT_EQUAL(true, b2.successorRestoredCalled(Color::plus, Color::none));
  }

  void testDoRestoreColorNotifiesZeroSuccs() {
    callDoSetColor(Color::plus);
    FakeBodyNode b1(2), b2(3);
    insertZeroSuccessor(b1);
    insertZeroSuccessor(b2);
    callDoRestoreColor(Color::none);
    CPPUNIT_ASSERT_EQUAL(true, b1.zeroPredRestoredCalled(Color::plus, Color::none));
    CPPUNIT_ASSERT_EQUAL(true, b2.zeroPredRestoredCalled(Color::plus, Color::none));
  }

  void testDoRestoreColorNotifiesOneSuccs() {
    callDoSetColor(Color::plus);
    FakeBodyNode b1(2), b2(3);
    insertOneSuccessor(b1);
    insertOneSuccessor(b2);
    callDoRestoreColor(Color::none);
    CPPUNIT_ASSERT_EQUAL(true, b1.onePredRestoredCalled(Color::plus, Color::none));
    CPPUNIT_ASSERT_EQUAL(true, b2.onePredRestoredCalled(Color::plus, Color::none));
  }

	void testFireColorEvent() {
		event::ChannelManager m;
		FakeHandler h;
		m.getChannelFor(event::Event<HeadNodeColored>()).connect(h);

		fireColorEvent(m, Color::plus);

		CPPUNIT_ASSERT_EQUAL(true, h.colorEventReceived(Color::plus));
	}
	void testFireRestoreEvent() {
		event::ChannelManager m;
		FakeHandler h;
		m.getChannelFor(event::Event<HeadNodeRestored>()).connect(h);

		fireRestoreEvent(m, Color::plus);

		CPPUNIT_ASSERT_EQUAL(true, h.restoreEventReceived(Color::plus));
	}
private:
  
	void callDoSetColor(Color::ColorValue newC) {
    HeadNode::setColor(newC);
  }
  void callDoRestoreColor(Color::ColorValue newC) {
    HeadNode::restoreColor(newC);
  }

  class FakeHandler {
	public:
		FakeHandler() : ceReceived_(false), reReceived_(false) {
		}
		
		void handle(const HeadNodeColored& e) {
			ceReceived_ = true;
			color_ = e.oldColor_;
		}
		void handle(const HeadNodeRestored& e) {
			reReceived_ = true;
			color_ = e.oldColor_;
		}
		bool colorEventReceived(Color::ColorValue expected) const {
			return ceReceived_ && color_ == expected;
		}
		bool restoreEventReceived(Color::ColorValue expected) const {
			return reReceived_ && color_ == expected;
		}
	private:
		bool ceReceived_, reReceived_;
		Color::ColorValue color_;
	};
	class FakeBodyNode : public BodyNode {
  public:
    FakeBodyNode(long id)
      : BodyNode(id)
      , succColored_(false)
      , zeroPredColored_(false)
      , onePredColored_(false)
      , succRestored_(false)
      , zeroPredRestored_(false)
      , onePredRestored_(false)
      , oldColor_(Color::none)
      , newColor_(Color::none) {
    }

    void successorColored(Color::ColorValue o, Color::ColorValue n) {
      succColored_ = true;
      oldColor_ = o;
      newColor_ = n;
    }
    void zeroPredecessorColored(Color::ColorValue o, Color::ColorValue n) {
      zeroPredColored_ = true;
      oldColor_ = o;
      newColor_ = n;
    }
    void onePredecessorColored(Color::ColorValue o, Color::ColorValue n) {
      onePredColored_ = true;
      oldColor_ = o;
      newColor_ = n;
    }
    void successorRestored(Color::ColorValue o, Color::ColorValue n) {
      succRestored_ = true;
      oldColor_ = o;
      newColor_ = n;
    }
    void zeroPredecessorRestored(Color::ColorValue o, Color::ColorValue n) {
      zeroPredRestored_ = true;
      oldColor_ = o;
      newColor_ = n;
    }
    void onePredecessorRestored(Color::ColorValue o, Color::ColorValue n) {
      onePredRestored_ = true;
      oldColor_ = o;
      newColor_ = n;
    }

    // Test Interface
    bool successorColoredCalled(Color::ColorValue oldC, Color::ColorValue newC) {
      return succColored_ && oldColor_ == oldC && newColor_ == newC;
    }
    bool zeroPredColoredCalled(Color::ColorValue oldC, Color::ColorValue newC) {
      return zeroPredColored_ && oldColor_ == oldC && newColor_ == newC;
    }
    bool onePredColoredCalled(Color::ColorValue oldC, Color::ColorValue newC) {
      return onePredColored_ && oldColor_ == oldC && newColor_ == newC;
    }

    bool successorRestoredCalled(Color::ColorValue oldC, Color::ColorValue newC) {
      return succRestored_ && oldColor_ == oldC && newColor_ == newC;
    }
    bool zeroPredRestoredCalled(Color::ColorValue oldC, Color::ColorValue newC) {
      return zeroPredRestored_ && oldColor_ == oldC && newColor_ == newC;
    }
    bool onePredRestoredCalled(Color::ColorValue oldC, Color::ColorValue newC) {
      return onePredRestored_ && oldColor_ == oldC && newColor_ == newC;
    }
  private:
    bool succColored_;
    bool zeroPredColored_;
    bool onePredColored_;
    bool succRestored_;
    bool zeroPredRestored_;
    bool onePredRestored_;
    Color::ColorValue oldColor_;
    Color::ColorValue newColor_;
  };
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestHeadNode);
}	// end namespace NS_NOMORE_TESTS
