//
//  ProgramOptions
//  (C) Copyright Benjamin Kaufmann, 2004 - 2005
//	Permission to copy, use, modify, sell and distribute this software is 
//	granted provided this copyright notice appears in all copies. 
//	This software is provided "as is" without express or implied warranty, 
//	and with no claim as to its suitability for any purpose.
//
//  ProgramOptions is a scaled-down version of boost::program_options
//  see: http://boost-sandbox.sourceforge.net/program_options/html/
// 
#ifndef PROGRAM_OPTIONS_VALUE_H_INCLUDED
#define PROGRAM_OPTIONS_VALUE_H_INCLUDED
#ifdef _MSC_VER
#pragma warning (disable : 4786)
#pragma warning (disable : 4503)
#endif
#include "program_options.h"
#include "value_base.h"
#include "errors.h"
#include <string>
#include <memory>
#include <sstream>
#include <vector>
#include <stdexcept>
namespace ProgramOptions {

//! typed value of an option
template <class T>
class Value : public ValueBase
{
public:
	/*!
     * \param storeTo where to store the value once it is known.
     * \note if storeTo is 0 Value allocates new memory for the value.
     */
    Value(T* storeTo = 0)
		: value_(storeTo)
		, defaultValue_(0)
		, deleteValue_(storeTo == 0)
		, implicit_(false)
		, defaulted_(false)
		, composing_(false)
		, hasValue_(false)
	{}
	
    Value(const Value<T>& other)
		: value_(other.value_ ? new T(*other.value_) : 0)
		, defaultValue_(other.defaultValue_ ? new T(*other.defaultValue_) : 0)
		, deleteValue_(true)
		, implicit_(other.implicit_)
		, defaulted_(other.defaulted_)
		, composing_(other.composing_)
		, hasValue_(other.hasValue_)
	{}
	
    Value& operator=(const Value<T>& other)
	{
		if (this != &other)
		{
			std::auto_ptr<T> newVal(other.value_ ? new T(*other.value_) : 0);
			std::auto_ptr<T> newDefVal(other.defaultValue_ ? new T(*other.defaultValue_) : 0);
			if (deleteValue_)
				delete value_;
			delete defaultValue_;
			value_ = newVal.release();
			deleteValue_ = true;
			defaultValue_ = newDefVal.release();
			implicit_ = other.implicit_;
			defaulted_ = other.defaulted_;
			composing_ = other.composing_;
			hasValue_ = other.hasValue_;
		}
		return *this;
	}
	~Value()
	{
		if (deleteValue_)
			delete value_;
		delete defaultValue_;
	}

	bool isImplicit() const {return implicit_;}
	bool isComposing() const {return composing_;}
	bool isDefaulted() const {return defaulted_;}
	
    void parse(const std::string&);

	const T& value() const
	{
		if (!value_)
			throw BadValue("no value");
		return *value_;
	}
	T& value()
	{
		if (!value_)
			throw BadValue("no value");
		return *value_;
	}

	//! sets a defaultValue for this value
    Value<T>* defaultValue(const T& t)
	{
		T* nd = new T(t);
		delete defaultValue_;
		defaultValue_ = nd;
		return this;

	}
    
    bool applyDefault()
	{
		if (defaultValue_)
		{
			if (value_)
			{
				*value_ = *defaultValue_;
			}
			else
			{
				value_ = new T(*defaultValue_);
				deleteValue_ = true;
			}
			defaulted_ = true;
			return true;
		}
		return false;
	}

	
	
    //! marks this value as implicit
    Value<T>* setImplicit()
	{
		implicit_ = true;
		return this;
	}
	
    //! marks this value as composing
    Value<T>* setComposing()
	{
		composing_ = true;
		return this;
	}
private:
	T* value_;
	T* defaultValue_;
	mutable bool deleteValue_;
	bool implicit_;
	bool defaulted_;
	bool composing_;
	bool hasValue_;
};


std::string checkSingleValue(const std::string& s, bool allowEmpty);

template <class T>
void parseValue(const std::string& s, T& t, double)
{
	std::stringstream str;
    str << ProgramOptions::checkSingleValue(s, false);
	if ( (str >> t).fail() || !str.eof() )
	{
		throw BadValue(std::string("value '") + s + "' does not look like a valid " + typeid(T).name());
	}
}

void parseValue(const std::string& s, bool& b, int);
void parseValue(const std::string& s, std::string& r, int);
void parseValue(const std::string& s, char& c, int);

template <class T>
void parseValue(const std::string& s, std::vector<T>& result, int)
{
	std::stringstream str(s);
	for (std::string item; str >> item; )
	{
		T temp;
        ProgramOptions::parseValue(item, temp, 1);
		result.push_back(temp);
	}
	if (!str.eof())
	{
		throw BadValue(std::string("value: ") + s + " does not look like a vector<T>");
	}
}



template <class T>
void Value<T>::parse(const std::string& s)
{
	if (!value_)
	{
		value_ = new T();
		deleteValue_ = true;
	}
	if (!hasValue_ || composing_)
	{
        ProgramOptions::parseValue(s, *value_, 1);
		defaulted_ = false;
		hasValue_ = true;
	}
	else
		throw MultipleOccurences("");
}

///////////////////////////////////////////////////////////////////////////////
// value creation functions
///////////////////////////////////////////////////////////////////////////////
//! creates a new Value-object for values of type T
template <class T>
Value<T>* value(T* v = 0)
{
	return new Value<T>(v);
}

//! creates a Value-object for options that are bool-switches like --help or --version
/*!
 * \note same as value<bool>()->setImplicit()
 */
Value<bool>* bool_switch(bool* b = 0);

///////////////////////////////////////////////////////////////////////////////
// option to value functions
///////////////////////////////////////////////////////////////////////////////

//! down_cast for Value-Objects.
/*!
 * \throw BadValue if opt is not of type Value<T>
 */
template <class T>
const T& value_cast(const ValueBase& opt, T* = 0)
{
	if (const Value<T>* p = dynamic_cast<const Value<T>*>(&opt))
		return p->value();
	std::string err = "value is not an ";
	err += typeid(T).name();
	throw BadValue(err.c_str());
}

template <class T>
T& value_cast(ValueBase& opt, T* = 0)
{
	if (Value<T>* p = dynamic_cast<Value<T>*>(&opt))
		return p->value();
	std::string err = "value is not an ";
	err += typeid(T).name();
	throw BadValue(err.c_str());
}

template <class T, class U>
const T& option_as(const U& container, const char* name, T* = 0)
{
	try {
        return ProgramOptions::value_cast<T>(container[name]);
	}
	catch(const BadValue& v)
	{
		std::string msg = "Option ";
		msg += name;
		msg += ": ";
		msg += v.what();
		throw BadValue(msg);
	}
}

}
#endif
