/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_NODE_H
#define LIBNOMORE_NODE_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif


#include <color.h>

namespace event {
	class ChannelManager;
}
namespace NS_NOMORE {

class Graph;

struct NodeType {
  enum Value {
    head_node = 1,
    body_node = 2
  };
};

//! Base class for all nodes to be stored in nomore's body-head graph.
/*! This base node holds common information (like identifier and node color)
 * 	needed by all subclasses e.g. HeadNode or BodyNode. 
 * \see Graph
 */
class Node {
  friend class Graph;

public:
  //! Creates a new node with the given id.
  /* \param id The id for the new node.
   * \param type The type of this node.
   */
  explicit Node(long id, NodeType::Value type);

  //! Destroys a node.
  virtual ~Node();

  //! Returns the current color value of the node.
  Color::ColorValue getColor() const {
    return color_;
  }

  //! Returns the unique identifier of the node.
  long getId() const {
    return id_;
  }

  //! Returns the type of this node.
  NodeType::Value getType() const {
    return type_;
  }

  void setComponentNumber(int nr) {
    componentNumber_ = nr;
  }

  //! returns the number of the component this node is a member of.
  /*!
   * \note the default is to return -1 meaning that no component number was set
   */
  int getComponentNumber() const {
    return componentNumber_;
  }

  //! returns true if this node is not strongly connected to any other node.
  /*!
   * \note if component numbers were not set the function returns false.
   */ 
  virtual bool trivialComponent() const = 0;

  //! Returns the color which preserves the recursive support used for the U operator.
  /*! The method returns the value of the supportedness of the node coded in the color
   *  weak_plus and plus.
   */
  virtual Color::ColorValue recursiveSupportColor() const = 0;

  //! Returns the preferred choice point color of this node.
  virtual Color::ColorValue preferredChoicePointColor() const = 0;

  //! Returns the alternative choice point color for this node.
  virtual Color::ColorValue alternativeChoicePointColor() const = 0;


  virtual void propagate(Graph& g) = 0;
protected:
  //! Colors the node.
  /*! Sets the node's color to c and calls doSetColor(oldColor) if the transition
   *  from the current color to c is valid and c differs from the current color.
   *  \note Refer to \ref NS_NOMORE::isValidColoring "isValidColoring" for information
   *  on valid colorings.
   *  \param c The new color for the node.
   *  \pre c is not Color::none
   *  \return 
   *    - ColorOperation::succeeded if a new color was set.
   *    - ColorOperation::unchanged if c and the node's current color are equal.
   *    - ColorOperation::failed    if coloring the node to c would be illegal.
   */
  ColorOpResult setColor(Color::ColorValue c);

  //! Restores the node's color to c.
  /*! Sets the color of the node to c and calls doRestoreColor(oldColor).
   *  \param c The new color of the node.
   *  \pre c is either Color::ignore, Color::none or Color::weak_plus. */
  void restoreColor(Color::ColorValue c) {
    Color::ColorValue oldColor = color_;
    color_ = c;
    doRestoreColor(oldColor);
  }

  //! Called by setColor if the color was changed.
  /*! Subclasses can implement necessary update-operations here.
   * \param oldColor The old color before coloring this node. */
  virtual void doSetColor(Color::ColorValue oldColor) = 0;

  //! Called by restoreColor.
	/*! Subclasses can implement necessary update-operations here.
   * \param oldColor The old color before restoring this node. */
  virtual void doRestoreColor(Color::ColorValue oldColor) = 0;

  //! Should fire a node-specific color event.
  /*! \ref _event "More information on events." */
  virtual void fireColorEvent(event::ChannelManager& c, Color::ColorValue oldColor) = 0;

  //! Should fire a node-specific color choice point event.
  /*! \ref _event "More information on events." */
  virtual void fireColorChoicePointEvent(event::ChannelManager& c, Color::ColorValue oldColor, bool recolored) = 0;
private:
  //! Copying of nodes is not supported
  Node(const Node& n);
  Node& operator=(const Node& n);

  //! The unique identifier of the node.
  long id_;

  NodeType::Value type_;

  //! The color of the node. Initialized to Color::none.
  Color::ColorValue color_;

  //! The number of the strong component this node is a member of.
  int componentNumber_;
  
};

//! id-based comparison of two nodes
inline bool operator==(const Node& lhs, const Node& rhs) {
  return lhs.getId() == rhs.getId();
}

//! Checks if a transition from oldColor to newColor is valid.
/*!
 * Valid colorings are:
 * - none -> X
 * - ignore	-> X
 * - weak_plus	-> plus
 * .
 * where X is any color value except Color::none.
 **/
inline bool isValidColoring(Color::ColorValue oldColor, Color::ColorValue newColor) {
	return 
    newColor != Color::none
    && (oldColor == Color::none || oldColor == Color::ignore)
    || (oldColor == Color::weak_plus && newColor == Color::plus);
}

//! Struct for checking the color of a node against a color mask.
struct HasColor {
	/*!
   * \param colorMask An ORed enumeration of color values
   * \see Color::ColorValue
   */
  explicit HasColor(int colorMask)
		: colorMask_(colorMask) {
	}

  //! returns true if n has a color that was specified in the color mask.
	bool operator()(Node* n) const {
		return (n->getColor() & colorMask_) != 0;
	}

private:
	int colorMask_;
};

//! Struct for creating a hash for a node and for comparing two nodes.
struct HashHelper {
  bool operator()(const Node* n1, const Node* n2) const {return n1->getId() == n2->getId();}
  long operator()(const Node* n1) const {
    return n1->getId();
  }
};
  
} // NS_NOMORE

#endif
