/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_FORWARD_PROP_OPERATOR_H
#define NOMORE_FORWARD_PROP_OPERATOR_H

#include <operator.h>
#include <color.h>
#include <queue>


namespace NS_NOMORE {

class BodyNode;
class HeadNode;
struct HeadNodeColored;
struct BodyNodeColored;
struct HeadNodeColoredChoicePoint;
struct BodyNodeColoredChoicePoint;

//! forward inference operator for the nomore body-head graph.
/*! The forward inference or forward propagation (also called \c P-operator)
 * 	tries to infer a color for successor-nodes starting from the coloring
 * 	of one node. That is, whenever a node is colored this operator tries
 * 	to infer a color for all zero, one and unlabeled successors of that node.
 *  The specific conditions under which a color can be infered are described below.
 *
 * 	The ForwardPropagator calls Graph::color to color a node. If Graph::color
 *	detects an invalid coloring the corresponding \ref ColorError "ColorError-Exception"
 *	is propagated.
 *
 * 	The ForwardPropagator learns about the nodes for which forward inference makes sense
 *	by listenig to the following four events: HeadNodeColored, BodyNodeColored, 
 *	HeadNodeColoredChoicePoint, BodyNodeColoredChoicePoint.
 *	It therefore registers itself with the graph as a handler for these events.
 *	
 *	\par Conditions \n
 *  The ForwardPropagator distinguishes between head- and body nodes
 *	when propagating. 
 *
 *	- If a \ref HeadNode "head node" changes its color, all its zero/one-successors (body nodes)
 *	are examined and colored using the following rules:	
 *		- succcessors that are \ref BodyNode::isSupported "supported" and \ref BodyNode::isUnblocked "unblocked"
 *		are colored to \ref Color::plus "plus".
 *		- succcessors that are \ref BodyNode::isWeakSupported "weak supported" and \ref BodyNode::isUnblocked "unblocked"
 *		are color to \ref Color::weak_plus "weak_plus".
 *		- successors that are \ref BodyNode::isUnsupported "unsupported" <b>or</b>
 *		\ref BodyNode::isBlocked "blocked" are colored to \ref Color::minus "minus".
 *    .
 *
 *	- If a \ref BodyNode "body node" changes its color, all its successors (head nodes)
 *	are examined and colored using the following rules:
 *		- If the \ref BodyNode "body node" was colored to \ref Color::plus "plus" 
 *		(\ref Color::weak_plus "weak plus") all successors are colored to
 *		\ref Color::plus "plus" (\ref Color::weak_plus "weak plus").
 *		- successors that are \ref HeadNode::isUnsupported "unsupported" are
 *		colored to \ref Color::minus "minus".
 *		.
 *	.
 *
 * 	\note \n
 *	Since the inference-process always starts from a colored node you must
 * 	make sure that at least one node gets a color by a different means.
 *	One way to get a starting node for the ForwardPropagator is to use
 *  the LparseReader which colors facts and self blockers.
 */
class ForwardPropagator : public PropOperator {
public:
  //! Creates the operator.
  /*! \param grp The graph the operator should work on. */ 
  ForwardPropagator(Graph& grp);
  ~ForwardPropagator();

  //! Handles a HeadNodeColored event fired by any head node which has been colored.
  void handle(const HeadNodeColored& e);

  //! Handles a BodyNodeColored event fired by any body node which has been colored.
  void handle(const BodyNodeColored& e);

  //! Handles a HeadNodeColoredChoicePoint event fired by any head node which has been colored as choice point.
  void handle(const HeadNodeColoredChoicePoint& e);

  //! Handles a BodyNodeColoredChoicePoint event fired by any body node which has been colored as choice point.
  void handle(const BodyNodeColoredChoicePoint& e);

  //! Returns the name of the operator.
  static std::string getOpName() {
    return "P";
  }

private:
  //! This method implements forward propagation as defined in the detailed section of ForwardPropagator.
  void execute();

  //! Resets the propagation queues.
  virtual void reset();
  
  //! Forward propagation for all head nodes.
  bool propagateHeadNodes();

  //! Forward propagation for all body nodes.
  bool propagateBodyNodes();

  std::queue<HeadNode*> headsChanged_;
  std::queue<BodyNode*> bodiesChanged_;
};

}

#endif
