/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_SOLVER_H
#define NOMORE_SOLVER_H

#include <memory>

namespace NS_NOMORE {

class Operator;
class ChoiceOperator;
class Printer;
class Graph;

//! Nomore's answer set solver.
class Solver {
public:
  //! the default constructor initializes all members to 0.
  /*!
   * \note This constructor constructs a not yet usable object.
   * To be usable one first needs to set operators and a printer.
   */
  Solver();
  
  /*! 
   * \param as Maximal number of answer sets to compute (0=all answer sets).
   * \param det Deterministic operators for propagation.
   * \param choice Choice operator for selecting nodes.
   * \param printer Printer instance to be used for printing answer sets.
   *
   * \pre pointer arguments are either != 0 and point to objects that were
   * allocated using new or 0.
   *
   * \pre choice->extractOperator(choice->getName()) returns an operator 
   * that can be downcasted to ChoiceOperator
   *
   * \note the Solver takes over ownership of the objects
   */
  Solver(int as, Operator *det, Operator *choice, Printer *printer);

  //! Copies a solver.
  Solver(const Solver& other);

  //! destroys the solver, all contained operators and the printer
  ~Solver();

  int  getNrOfAnswerSetsToCompute() const;
  void setNrOfAnswerSetsToCompute(int as);
  
  Operator* getPropagationOperator() const;

  //! sets the propagation operator to be used by the solver.
  /*!
   * \pre det is != 0 and points to an object that was allocated using new
   * \post the solver owns det
   */
  void setPropagationOperator(Operator* det);
  
  ChoiceOperator* getChoiceOperator() const;

  //! sets the choice operator to be used by the solver.
  /*!
   * \pre choice is != 0 and points to an object that was allocated using new
   * \pre choice->extractOperator(choice->getName()) returns an operator 
   * that can be downcasted to ChoiceOperator
   *
   * \post the solver owns choice
   */
  void setChoiceOperator(Operator* choice);
  
  //! sets the printer to be used by the solver.
  /*!
   * \pre printer is != 0 and points to an object that was allocated using new
   * \post the solver owns printer
   */
  void setAnswerSetPrinter(Printer* printer);

  //! Solves the logic program represented by the given graph using the set operators.
  /*! 
   * \pre Operators for propgation, post-processing and choosing were set. 
   * \pre A printer for printing answer sets were set.
   *
   *  The solver uses the following algorithm:
   *  \code
   *    error := false;         // - no more answer sets?
   *
   *    while(!error) {
   *      try {
   *        det-Operator();
   *        if(graph is totally colored) {
   *          print_answer_set;
   *          
   *          // error is true if no choice can be recolored -> no more answer sets
   *          error := try_recolor_choice();
   *        } else {
   *
   *          // graph is not total colored -> select a choice point and color
   *          if(!select-Operator()) {
   * 
   *            // no choice point is found -> execute the post processing step
   *            post-Operator();
   *          }
   *        }
   *      } catch(ColorError) {
   *        // catches all color error of the operators det, post and select/choice and
   *        // the try_recolor_choice method
   *     
   *        // error is true if no choice can be recolored -> no more answer sets
   *        error = try_recolor_choice();
   *      }
   *    }
   *  \endcode
   *  \return False if all answer sets were computed. True if possibly more 
   *  answer sets could be computed. 
   */
  bool solve(Graph& grp);
private:
  //! Number of answer sets to solve (0=all answer sets).
  mutable int as_;

  //! Pointer to the deterministic operator (or sequence containing several operators).
  mutable std::auto_ptr<Operator> det_;

  //! Pointer to the choice operator to select a choice point.
  mutable std::auto_ptr<Operator> choice_;

  //! User defined printer object for printing the answer sets.
  mutable std::auto_ptr<Printer> printer_;

};
}

#endif
