#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <event/channel_manager.h>
#include "test_common.h"
using namespace std;
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestBodyNode : public CppUnit::TestFixture, public BodyNode {
public:
  TestBodyNode()
    : BodyNode(1) {
  }

  void testInit() {
    CPPUNIT_ASSERT(successorsBegin() == successorsEnd());
    CPPUNIT_ASSERT(zeroPredecessorsBegin() == zeroPredecessorsEnd());
    CPPUNIT_ASSERT(onePredecessorsBegin() == onePredecessorsEnd());
  }

  void testEmptyIsSupported() {
    CPPUNIT_ASSERT_EQUAL(true, isSupported());
  }

  void testSupported() {
    HeadNode h(1, 1);
    insertZeroPredecessor(h);
    CPPUNIT_ASSERT_EQUAL(false, isSupported());

    zeroPredecessorColored(Color::none, Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, isSupported());

    zeroPredecessorRestored(Color::plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isSupported());
  }

  void testEmptyIsWeakSupported() {
    CPPUNIT_ASSERT_EQUAL(true, isWeakSupported());
  }

  void testIsWeakSupportedByPlus() {
    HeadNode h(1, 1);
    insertZeroPredecessor(h);
    CPPUNIT_ASSERT_EQUAL(false, isWeakSupported());

    zeroPredecessorColored(Color::none, Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, isWeakSupported());

    zeroPredecessorRestored(Color::plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isWeakSupported());
  }

  void testIsWeakSupportedByWeakPlus() {
    HeadNode h(1, 1);
    insertZeroPredecessor(h);
    CPPUNIT_ASSERT_EQUAL(false, isWeakSupported());

    zeroPredecessorColored(Color::none, Color::weak_plus);
    CPPUNIT_ASSERT_EQUAL(true, isWeakSupported());

    zeroPredecessorRestored(Color::weak_plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isWeakSupported());
  }

  void testEmptyIsNotUnsupported() {
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());
  }

  void testIsUnsupportedByOne() {
    HeadNode h1(1,1), h2(2,2);
    insertZeroPredecessor(h1);
    insertZeroPredecessor(h2);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());

    zeroPredecessorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(true, isUnsupported());

    zeroPredecessorRestored(Color::minus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());
  }

  void testIsUnsupportedByAll() {
    HeadNode h1(1,1), h2(2,2);
    insertZeroPredecessor(h1);
    insertZeroPredecessor(h2);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());

    zeroPredecessorColored(Color::none, Color::minus);
    zeroPredecessorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(true, isUnsupported());

    zeroPredecessorRestored(Color::minus, Color::none);
    CPPUNIT_ASSERT_EQUAL(true, isUnsupported());
    zeroPredecessorRestored(Color::minus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isUnsupported());
  }

  void testEmptyIsNotBlocked() {
    CPPUNIT_ASSERT_EQUAL(false, isBlocked());
  }

  void testIsBlockedByOnePlus() {
    HeadNode h1(1,1), h2(2,2);
    insertOnePredecessor(h1);
    insertOnePredecessor(h2);
    CPPUNIT_ASSERT_EQUAL(false, isBlocked());

    onePredecessorColored(Color::none, Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, isBlocked());

    onePredecessorRestored(Color::plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isBlocked());
  }

  void testIsBlockedByOneWeakPlus() {
    HeadNode h1(1,1), h2(2,2);
    insertOnePredecessor(h1);
    insertOnePredecessor(h2);
    CPPUNIT_ASSERT_EQUAL(false, isBlocked());

    onePredecessorColored(Color::none, Color::weak_plus);
    CPPUNIT_ASSERT_EQUAL(true, isBlocked());

    onePredecessorRestored(Color::weak_plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isBlocked());
  }

  void testIsBlockedByAll() {
    HeadNode h1(1,1), h2(2,2);
    insertOnePredecessor(h1);
    insertOnePredecessor(h2);
    CPPUNIT_ASSERT_EQUAL(false, isBlocked());

    onePredecessorColored(Color::none, Color::weak_plus);
    onePredecessorColored(Color::none, Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, isBlocked());

    onePredecessorRestored(Color::weak_plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(true, isBlocked());
    onePredecessorRestored(Color::plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isBlocked());
  }

  void testEmptyIsUnblocked() {
    CPPUNIT_ASSERT_EQUAL(true, isUnblocked());
  }

  void testIsUnblocked() {
    HeadNode h1(1,1), h2(2,2);
    insertOnePredecessor(h1);
    insertOnePredecessor(h2);
    CPPUNIT_ASSERT_EQUAL(false, isUnblocked());

    onePredecessorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(false, isUnblocked());
    onePredecessorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(true, isUnblocked());

    onePredecessorRestored(Color::minus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, isUnblocked());
  }

  void testEmptyCanBeIgnored() {
    CPPUNIT_ASSERT_EQUAL(true, canBeIgnored());
  }

  void testCanBeIgnoredByPlus() {
    HeadNode h1(1, 1);
    insertSuccessor(h1);
    CPPUNIT_ASSERT_EQUAL(false, canBeIgnored());

    successorColored(Color::none, Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, canBeIgnored());

    successorRestored(Color::plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, canBeIgnored());
  }

  void testCanBeIgnoredByMinus() {
    HeadNode h1(1, 1);
    insertSuccessor(h1);
    CPPUNIT_ASSERT_EQUAL(false, canBeIgnored());

    successorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(true, canBeIgnored());

    successorRestored(Color::minus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, canBeIgnored());
  }

  void testCanBeIgnored() {
    HeadNode h1(1,1), h2(2,2);
    insertSuccessor(h1);
    insertSuccessor(h2);
    successorColored(Color::none, Color::plus);
    successorColored(Color::none, Color::minus);
    CPPUNIT_ASSERT_EQUAL(true, canBeIgnored());

    successorRestored(Color::plus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, canBeIgnored());
    successorColored(Color::none, Color::plus);
    CPPUNIT_ASSERT_EQUAL(true, canBeIgnored());
    successorRestored(Color::minus, Color::none);
    CPPUNIT_ASSERT_EQUAL(false, canBeIgnored());
  }

  void testInsertSuccessor() {
    HeadNode h(1, 1);
    insertSuccessor(h);
    CPPUNIT_ASSERT(findNode(successorsBegin(), successorsEnd(), h));
  }

  void testInsertSuccessorCallsHeadInsertPred() {
    HeadNode h(1, 1);
    insertSuccessor(h);
    CPPUNIT_ASSERT(findNode(h.predecessorsBegin(), h.predecessorsEnd(), *this));
  }

  void testInsertZeroPred() {
    HeadNode h(1, 1);
    insertZeroPredecessor(h);
    CPPUNIT_ASSERT(findNode(zeroPredecessorsBegin(), zeroPredecessorsEnd(), h));

    // redundant - siehe testEmptyIsSupported und testIsSupported
    CPPUNIT_ASSERT_EQUAL(false, isSupported());
    CPPUNIT_ASSERT_EQUAL(false, isWeakSupported());
  }

  void testInsertOnePred() {
    HeadNode h(1, 1);
    insertOnePredecessor(h);
    CPPUNIT_ASSERT(findNode(onePredecessorsBegin(), onePredecessorsEnd(), h));

    // redundant - siehe Tests fr unblocked
    CPPUNIT_ASSERT_EQUAL(false, isUnblocked());
  }

  void testSetColorNotifiesSuccs() {
    FakeHeadNode h1(1), h2(2);
    insertSuccessor(h1);
    insertSuccessor(h2);

    callSetColor(Color::plus);

    CPPUNIT_ASSERT_EQUAL(true, h1.predColoredCalled(Color::none, Color::plus));
    CPPUNIT_ASSERT_EQUAL(true, h2.predColoredCalled(Color::none, Color::plus));
  }

  void testRestoreColorNotifiesSuccs() {
    FakeHeadNode h1(1), h2(2);
    insertSuccessor(h1);
    insertSuccessor(h2);

    callSetColor(Color::plus);
    callRestoreColor(Color::none);

    CPPUNIT_ASSERT_EQUAL(true, h1.predRestoredCalled(Color::plus, Color::none));
    CPPUNIT_ASSERT_EQUAL(true, h2.predRestoredCalled(Color::plus, Color::none));
  }

	void testFireColorEvent() {
		event::ChannelManager m;
		FakeHandler h;
		m.getChannelFor(event::Event<BodyNodeColored>()).connect(h);

		fireColorEvent(m, Color::plus);

		CPPUNIT_ASSERT_EQUAL(true, h.colorEventReceived(Color::plus));
	}
	
	
	CPPUNIT_TEST_SUITE(TestBodyNode);
  CPPUNIT_TEST(testInit); 
  CPPUNIT_TEST(testEmptyIsSupported); 
  CPPUNIT_TEST(testSupported);  
  CPPUNIT_TEST(testEmptyIsWeakSupported);
  CPPUNIT_TEST(testIsWeakSupportedByPlus);
  CPPUNIT_TEST(testIsWeakSupportedByWeakPlus);
  CPPUNIT_TEST(testEmptyIsNotUnsupported);
  CPPUNIT_TEST(testIsUnsupportedByOne);
  CPPUNIT_TEST(testIsUnsupportedByAll);

  CPPUNIT_TEST(testEmptyIsNotBlocked);
  CPPUNIT_TEST(testIsBlockedByOnePlus);
  CPPUNIT_TEST(testIsBlockedByOneWeakPlus);
  CPPUNIT_TEST(testIsBlockedByAll);

  CPPUNIT_TEST(testEmptyIsUnblocked);
  CPPUNIT_TEST(testIsUnblocked);

  CPPUNIT_TEST(testEmptyCanBeIgnored);
  CPPUNIT_TEST(testCanBeIgnoredByPlus);
  CPPUNIT_TEST(testCanBeIgnoredByMinus);
  CPPUNIT_TEST(testCanBeIgnored);

  CPPUNIT_TEST(testInsertSuccessor);
  CPPUNIT_TEST(testInsertSuccessorCallsHeadInsertPred);

  CPPUNIT_TEST(testInsertZeroPred);
  CPPUNIT_TEST(testInsertOnePred);

  CPPUNIT_TEST(testSetColorNotifiesSuccs);
  CPPUNIT_TEST(testRestoreColorNotifiesSuccs);

	CPPUNIT_TEST(testFireColorEvent);
  CPPUNIT_TEST_SUITE_END();
private:
  void callSetColor(Color::ColorValue c) {
    BodyNode::setColor(c);
  }
  void callRestoreColor(Color::ColorValue c) {
    BodyNode::restoreColor(c);
  }

  class FakeHandler {
	public:
		FakeHandler() : ceReceived_(false) {
		}
		
		void handle(const BodyNodeColored& e) {
			ceReceived_ = true;
			color_ = e.oldColor_;
		}
		bool colorEventReceived(Color::ColorValue expected) const {
			return ceReceived_ && color_ == expected;
		}
	private:
		bool ceReceived_;
		Color::ColorValue color_;
	};

	class FakeHeadNode : public HeadNode {
  public:
    FakeHeadNode(long id)
      : HeadNode(id, id)
      , predColoredCalled_(false)
      , predRestoredCalled_(false) {
    }
    void predecessorColored(Color::ColorValue old, Color::ColorValue n) {
      predColoredCalled_ = true;
      old_ = old;
      new_ = n;
    }
    void predecessorRestored(Color::ColorValue old, Color::ColorValue n) {
      predRestoredCalled_ = true;
      old_ = old;
      new_ = n;
    }

    bool predColoredCalled(Color::ColorValue old, Color::ColorValue n) const {
      return predColoredCalled_ && old_ == old && new_ == n;
    }
    bool predRestoredCalled(Color::ColorValue old, Color::ColorValue n) const {
      return predRestoredCalled_ && old_ == old && new_ == n;
    }
  private:
    bool predColoredCalled_;
    bool predRestoredCalled_;
    Color::ColorValue old_;
    Color::ColorValue new_;
  };
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestBodyNode);

}	// end namespace NS_NOMORE_TESTS
