#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/unfounded_set.h>
#include <graph.h>
#include <head_node.h>
#include <body_node.h>
#include <strong_components.h>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestMaxSupport : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(TestMaxSupport);
  CPPUNIT_TEST(testExample);
  CPPUNIT_TEST_SUITE_END();

public:
  /* example:
   *
   * B1          B8--->H5<---B7
   * |                 |      ^
   * |                 0      |
   * |                 |      0
   * +                 +      |
   * H1<---B4<-0-H3    B6    H6
   * |           ^     |      ^
   * 0           |     |      |
   * |           |     |      |
   * +           |     +      |
   * B2--->H2-0->B3--->H4-0->B5
   *
   * H1 :- B1.
   * H1 :- B4.
   * H2 :- B2.
   * H3 :- B3.
   * H4 :- B3.
   * H4 :- B6.
   * H5 :- B7.
   * H5 :- B8.
   * H6 :- B5.
   *
   * B1 := {}
   * B2 := {H1}
   * B3 := {H2}
   * B4 := {H3}
   * B5 := {H4}
   * B6 := {H5}
   * B7 := {H6}
   * B8 := {}
   *
   * heads: 1,2,3,4,5,6
   * bodies: 1,2,3,4,5,6,7,8
   */
	void testExample() {
    Graph grp;
    UnfoundedSetOperator v(grp);

    HeadNode *H1 = grp.insertHeadNode(1);
    HeadNode *H2 = grp.insertHeadNode(2);
    HeadNode *H3 = grp.insertHeadNode(3);
    HeadNode *H4 = grp.insertHeadNode(4);
    HeadNode *H5 = grp.insertHeadNode(5);
    HeadNode *H6 = grp.insertHeadNode(6);

    BodyNode *B1 = grp.insertBodyNode();
    BodyNode *B2 = grp.insertBodyNode();
    BodyNode *B3 = grp.insertBodyNode();
    BodyNode *B4 = grp.insertBodyNode();
    BodyNode *B5 = grp.insertBodyNode();
    BodyNode *B6 = grp.insertBodyNode();
    BodyNode *B7 = grp.insertBodyNode();
    BodyNode *B8 = grp.insertBodyNode();

    B1->insertSuccessor(*H1);
    B4->insertSuccessor(*H1);
    B2->insertSuccessor(*H2);
    B3->insertSuccessor(*H3);
    B3->insertSuccessor(*H4);
    B6->insertSuccessor(*H4);
    B7->insertSuccessor(*H5);
    B8->insertSuccessor(*H5);
    B5->insertSuccessor(*H6);

    H1->insertZeroSuccessor(*B2);
    H2->insertZeroSuccessor(*B3);
    H3->insertZeroSuccessor(*B4);
    H4->insertZeroSuccessor(*B5);
    H5->insertZeroSuccessor(*B6);
    H6->insertZeroSuccessor(*B7);

    grp.color(*B8, Color::minus);
    // first call to V -> nothing has been changed
    CPPUNIT_ASSERT_EQUAL( ColorOperation::unchanged, v() );

    CPPUNIT_ASSERT_EQUAL( Color::none, B2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, B3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, B4->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, B5->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, B6->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, B7->getColor() );

    CPPUNIT_ASSERT_EQUAL( Color::none, H1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, H2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, H3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, H4->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, H5->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, H6->getColor() );

    grp.color(*B1, Color::minus);
    // second call to V -> all nodes are minus
    CPPUNIT_ASSERT_EQUAL( ColorOperation::succeeded, v() );

    CPPUNIT_ASSERT_EQUAL( Color::minus, B2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, B3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, B4->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, B5->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, B6->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, B7->getColor() );

    CPPUNIT_ASSERT_EQUAL( Color::minus, H1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, H2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, H3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, H4->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, H5->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, H6->getColor() );

    CPPUNIT_ASSERT_EQUAL(4, identifyStrongComponents(grp));
  }
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestMaxSupport);

}	// end namespace NS_NOMORE_TESTS
