/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_NEG_ARCS_COUNT_HEURISTIC_H
#define LIBNOMORE_NEG_ARCS_COUNT_HEURISTIC_H
#include <static_heuristic.h>
namespace NS_NOMORE {

//! Heuristic-Strategy that compares negative preconditions and negative successors of body and head nodes.
/*! The contained EvaluatorType generates heuristic values of the each uncolored body and head node.
 *  The value consists of the number of negative preconditions (the number of 1-predecessors) 
 *  and the number of negative successors (of all heads for a body node).
 *  The count of negative successors is weighted by 1024.
 *
 *  \note This Strategy may be used with either a Static- or a Dynamic-Heuristic.
 *  But since the heuristic value won't change over the time, the former
 *  is more sensible.
 *
 *  \note This strategy works with body and head nodes.
 */
class NegativeWeightArcsCountStrat {
public:
  //! the value type of the strategie containing two counters for negative successors and predecessors
  struct ValueType {
    ValueType(size_t ns = 0, size_t np = 0)
      : negSuccessors_(ns)
      , negPredecessors_(np) {
    }
    size_t negSuccessors_;
    size_t negPredecessors_;
  };

  //! the greater comparator for the value type
  struct Compare {
    bool operator()(const ValueType& v1, const ValueType& v2) const {
      return v1.negSuccessors_ > v2.negSuccessors_ 
        || (v1.negSuccessors_ == v2.negSuccessors_  && v1.negPredecessors_ > v2.negPredecessors_); 
    }
  };
  
  //! evalutes the heuristic value of each uncolored node
  struct Evaluate {
    ValueType operator()(Node* n, double tf) const;
  };
  
  //! the comparator type
  typedef Compare ComparatorType;

  //! the evaluator of the strategie
  typedef Evaluate EvaluatorType;
};

//! A static heuristic that uses NegativeWeightArcsCountStrat as strategy.
typedef StaticHeuristic<NegativeWeightArcsCountStrat> NegativeWeightArcsCount;

}
#endif
