/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_OBJECT_FACTORY_H
#define LIBNOMORE_OBJECT_FACTORY_H

#include <string>
#include <stdexcept>
#include <map>
#include <utility>
#include <vector>
#include <memory>

namespace NS_NOMORE {

class Graph;

class Operator;
class ChoiceOperator;
class Lookahead;

class Heuristic;
class OperatorDecoratorFactory;
class Constraint;

//! Factory that maps logical operator (and heuristic) names to their C++ classes.
/*!
 * This Factory can be used to create operators and heuristic by only specifying
 * their logical name.
 * A DecoratorFactory can be installed to decorate each constructed operator.
 *
 */
class ObjectFactory {
public:
  typedef Operator* (*OpCreateFunction)(Graph& );
  typedef Lookahead* (*LookOpCreateFunction)(Graph& );
  typedef ChoiceOperator* (*ChoiceOpCreateFunction)(Graph&, Heuristic*);
  typedef Heuristic* (*HeuristicCreateFunction)();

  //! class holding information necessary for registering objects in the object factory.
  template <class CF>
  struct RegInfo {
    typedef CF Creator;
    RegInfo() : name(""), description(""), creator(0){}
    /*!
     * \param _name the unique name of the class to be registered.
     * \param cf a creator for objects of type specified by _name.
     * \param _description an optional description of the type
     */
    RegInfo(const std::string& _name, Creator cf, const char* _description = "")
      : name(_name)
      , description(_description)
      , creator(cf) {
    }
    std::string name;
    std::string description;
    Creator     creator;
  };
  
  typedef RegInfo<OpCreateFunction> OpInfo;
  typedef RegInfo<LookOpCreateFunction> LookOpInfo;
  typedef RegInfo<ChoiceOpCreateFunction> ChoiceOpInfo;
  typedef RegInfo<HeuristicCreateFunction> HeuristicInfo;

  
  //! creates a new factory object
  ObjectFactory();
  
  /*!
   * creates a new object that decorates requested operators using the given
   * DecoratorFactory
   * \pre df points to a dynamically allocated object
   * \post The factory owns the DecoratorFactory
   */
  explicit ObjectFactory(OperatorDecoratorFactory* df);
  
  /*!
   * creates a new object that decorates requested operators using the given
   * DecoratorFactory
   * \note the factory only stores a reference to the DecoratorFactory but
   * does not take over ownership. 
   */
  ObjectFactory(OperatorDecoratorFactory& df);
  
  ~ObjectFactory();

  /*!
   * \pre df points to a dynamically allocated object
   * \post The factory owns the DecoratorFactory
   */
  void setOperatorDecoratorFactory(OperatorDecoratorFactory* df);
  
  /*!
   * \note the factory only stores a reference to the DecoratorFactory but
   * does not take over ownership. 
   */
  void setOperatorDecoratorFactory(OperatorDecoratorFactory& df);

  
  //! object types this factory can create
  enum ObjectType {
    DET_OP        = 1,
    LOOKAHEAD_OP  = 2,
    CHOICE_OP     = 4,
    HEURISTIC     = 8,
    ALL_OPERATORS = DET_OP | LOOKAHEAD_OP | CHOICE_OP,
    ALL_OBJECTS   = ALL_OPERATORS | HEURISTIC
  };
  
  //! type used as result type for the factory's createOperators operation
  struct Operators {
    ~Operators();
    Operators();
    
    Operators(const Operators& other);
      
    mutable std::auto_ptr<Operator>       detOp_;
    mutable std::auto_ptr<Operator>       lookOp_;
    mutable std::auto_ptr<Operator>       choiceOp_;
  };

  //! registers all known operators/heuristics.in this factory object,
  void registerKnownClasses();
  
  
  
  //! registers the operator described by info in the factory.
  /*!
   * \return true if registration was successful. false if name already exists.
   */
  bool registerClass(const OpInfo& info);

  /*!
   * \overload registerClass(const OpInfo&);
   */
  bool registerClass(const LookOpInfo& info);
  
  /*!
   * \overload registerClass(const OpInfo&);
   */
  bool registerClass(const ChoiceOpInfo& info);
  
  /*!
   * \overload registerClass(const OpInfo&);
   */
  bool registerClass(const HeuristicInfo& info);

  //! creates an operator of type specified by name.
  /*!
   * \throw UnknownObject if name is not a known operator name
   */
  std::auto_ptr<Operator> createOperator(Graph& g, const std::string& name) const;

  //! creates a lookahead operator of type specified by name.
  /*!
   * \throw UnknownObject if name is not a known operator name
   *
   * \note If a DecoratorFactory is used the returned operator is *not* of 
   * type Lookahead. It merely decorates such an operator. 
   */
  std::auto_ptr<Operator> createLookaheadOperator(Graph& g, const std::string& name) const;

  //! creates a choice operator of type specified by name using the heuristic with the name heu.
  /*!
   * \throw UnknownObject if name is not a known choice operator or heu is not a known heuristic
   *
   * \note If a DecoratorFactory is used the returned operator is *not* a 
   * ChoiceOperator. It merely decorates one.
   */
  std::auto_ptr<Operator> createChoiceOperator(Graph& g, const std::string& name, const std::string& heu) const;
  
  //! creates a heuristic of type specified by name.
  /*!
   * \throw UnknownObject if name is not a known heuristic
   */
  std::auto_ptr<Heuristic> createHeuristic(Graph& g, const std::string& name) const;

  //! creates the operators described in objStr to be used with the given Graph g.
  /*!
   * \param g The graph the operators should be applied on.
   * \param objStr a string containing the names of the to be created operators.
   * objStr must have the following format: ChoiceOp:DetOp:LOp[DetOp]:Heuristic
   * where:
   *  - ChoiceOp is the name of the requested Choice operator, e.g. D
   *  - DetOp is the requested set of propagation operators
   *  - LOp[DetOp] is the name of the requested Lookahead Operator followed
	 *  by the set of propagation operators the Lookahead Operator should use
	 *  enclosed in square brackets
   * - Heuristic is the name of the requested heuristic the Choice operator should use
   * .
   *
   * \param validate if true, then this method checks whether the created operators can
   * be used together.
   *
   * \return An object of type Operators that contains the requested operators.
   *
   * \note 
   * - ChoiceOp and DetOp are required.
	 * - All other parts may be set to "None" if not needed.
   * - DetOp must be specified in a format that is understood by the \ref OperatorStringParser "OperatorStringParser class".
   *
   * \throw FormatError if objStr does not have a valid format.
   * \throw UnknownObject if objStr contains an unknown name
   * \throw InvalidOperatorCombination if validate is true and objStr contains
   * an invalid operator combination.
   * \throw OperatorStringParser::SyntaxError if the DetOp part of objStr is invalid.
   * 
   */
  Operators createOperators(Graph& g, const std::string& objStr, bool validate = true) const;
  

  //! returns the names of all registered classes of type ot.
  std::vector<std::string> getRegisteredClasses(ObjectType ot) const;

  //! returns the description for the type specified by name
  std::string getDescription(const std::string& name) const;
private:
  std::auto_ptr<Operator> getLookahead(Graph& g,
                                       const std::auto_ptr<Operator>& detOp, 
                                       const std::auto_ptr<Operator>& choice,
                                       const std::vector<std::string>& opStrings) const;
  ObjectFactory(const ObjectFactory&);
  ObjectFactory& operator=(const ObjectFactory&);
  
  OperatorDecoratorFactory* opDecorator_;
  bool owner_;
  
  typedef std::map<std::string, std::pair<OpCreateFunction, std::string> > OpMap;
  typedef std::map<std::string, std::pair<LookOpCreateFunction, std::string> > LookOpMap;
  typedef std::map<std::string, std::pair<ChoiceOpCreateFunction, std::string> > ChoiceOpMap;
  typedef std::map<std::string, std::pair<HeuristicCreateFunction, std::string> > HeuristicsMap;

  OpMap         detOps_;
  LookOpMap     lookaheadOps_;
  ChoiceOpMap   choiceOps_;
  HeuristicsMap heuristics_;

};

//! returns the name of class T
/*!
 * specialize this function for each class T to be registered in the object factory
 * 
 * \attention The parameter is needed to workaround a bug in VC 6.0 but should
 * otherwise be ignored.
 *
 * \code
 * // in the cpp-file of the class to be registered
 * template <>
 * const char* getName(YourClass*) {
 *    return "YourClass" // or whatever name YourClass should have
 * }
 */
template <class T>
const char* getName(T*);

//! creation function for propagation operators
/*!
 * specialize this function for each propagation operator to be registered in 
 * the object factory.
 *
 * \attention The parameter is needed to workaround a bug in VC 6.0 but should
 * otherwise be ignored.
 *  
 * \code
 * // in the cpp-file implementing the operator
 * template <>
 * Operator* createDetOp(Graph& g, YourOperator*) {
 *    return new YourOperator(g);
 * }
 * \endcode
 */
template <class T>
Operator* createDetOp(Graph&, T*);

//! creation function for lookahead operators
/*!
 * \see createDetOp
 */
template <class T>
Lookahead* createLookaheadOp(Graph&, T*);

//! creation function for choice operators
/*!
 * \see createDetOp
 */
template <class T>
ChoiceOperator* createChoiceOp(Graph&, Heuristic*, T*);

//! creation function for heuristics
/*!
 * \see createDetOp
 */
template <class T>
Heuristic* createHeuristic(T*);
  

//! Exception type thrown if the factory is requested to create an unknown object.
class UnknownObject : public std::runtime_error {
public:
  UnknownObject(const std::string& name, ObjectFactory::ObjectType t)
    : std::runtime_error(name)
    , ot_(t) {

  }
  //! returns the type of the unknown object
  /*!
   * \see ObjectFactory::ObjectType
   */
  ObjectFactory::ObjectType getObjectType() const {
    return ot_;
  }
private:
  ObjectFactory::ObjectType ot_;
};

//! Exception type thrown when a string has an invalid/unexpected format
class FormatError : public std::runtime_error {
public:
  FormatError(const std::string& desc)
    : std::runtime_error(desc) {

  }
};

//! Exception type thrown when an operator string contains an invalid operator combination
class InvalidOperatorCombination : public std::runtime_error {
public:
  InvalidOperatorCombination() 
    : std::runtime_error("invalid operator combination") {
  }
};

}

#endif
