/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         *
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         *
 ***************************************************************************/

#ifndef INCLUDED_IN_OBJECT_FACTORY

#error "This is an internal header!"


#endif

#include <object_factory.h>
#include <vector>
namespace NS_NOMORE {

template <class T>
class Register {
public:
  Register(const char* desc = "")
    : desc_(desc) {

  }
  static Operator* createOp(Graph& g) {
    return NS_NOMORE::createDetOp<T>(g, 0);
  }

  static Lookahead* createLookaheadOp(Graph& g) {
    return NS_NOMORE::createLookaheadOp<T>(g, 0);
  }

  static ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h) {
    return NS_NOMORE::createChoiceOp<T>(g, h, 0);
  }
  static Heuristic* createHeu() {
    return NS_NOMORE::createHeuristic<T>(0);
  }
  static const char* name() {
    return NS_NOMORE::getName<T>(0);
  }

  operator ObjectFactory::OpInfo() const {
    return ObjectFactory::OpInfo(name(), createOp, desc_);
  }

  operator ObjectFactory::ChoiceOpInfo() const {
    return ObjectFactory::ChoiceOpInfo(name(), createChoiceOp, desc_);
  }

  operator ObjectFactory::HeuristicInfo() const {
    return ObjectFactory::HeuristicInfo(name(), createHeu, desc_);
  }
  ObjectFactory::HeuristicInfo hi() const {
    return ObjectFactory::HeuristicInfo(name(), createHeu, desc_);
  }

  operator ObjectFactory::LookOpInfo() const {
    return ObjectFactory::LookOpInfo(name(), createLookaheadOp, desc_);
  }
private:
  const char* desc_;
};

template <class Info>
class Registrar {
public:
  Registrar& add(const Info& info) {
    classes_.push_back(info);
    return *this;
  }
  Registrar operator()(const Info& info) {
    return add(info);
  }
  void registerClasses(ObjectFactory& f) {
    for (size_t i = 0; i < classes_.size(); ++i) {
      f.registerClass(classes_[i]);
    }
  }
private:
  std::vector< Info > classes_;
};

typedef Registrar<ObjectFactory::OpInfo> PropOpRegistrar;
typedef Registrar<ObjectFactory::LookOpInfo> LookaheadOpRegistrar;
typedef Registrar<ObjectFactory::ChoiceOpInfo> ChoiceOpRegistrar;
typedef Registrar<ObjectFactory::HeuristicInfo> HeuristicRegistrar;

///////////////////////////////////////////////////////////////////////////////
// Propagation Operators
///////////////////////////////////////////////////////////////////////////////
static PropOpRegistrar detOps_s = PropOpRegistrar()
  .add( Register<class ForwardPropagator >("Standard forward propation.") )
  .add( Register<class BackwardPropagator>("Standard backward propagation.") )
  .add( Register<class UnfoundedLoopsOperator>("Unfounded Loop finder.") )
;
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Lookahead Operators
///////////////////////////////////////////////////////////////////////////////
static LookaheadOpRegistrar lookaheadOps_s = LookaheadOpRegistrar()
  .add(Register<class HybridLookahead>("Lookahead on head and body nodes."))
  .add(Register<class BodyLookahead>("Lookahead on body nodes only."))
;
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Choice Operators
///////////////////////////////////////////////////////////////////////////////
static ChoiceOpRegistrar choiceOps_s = ChoiceOpRegistrar()
  .add(Register<class SelectUncolored>("Selects body as well as head nodes."))
  .add(Register<class SelectUncoloredBody>("Selects body nodes only."))
  .add(Register<class SelectSupported>("Selects supported body nodes only."))
  .add(Register<class HybridSelect>("Selects body nodes that are supported or only trivially connected."))
  .add(Register<class SelectSuccessors>("Selects nodes and their neighbors."))
;
///////////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////////////////////
// Heuristics
///////////////////////////////////////////////////////////////////////////////
template <class T> class StaticHeuristic;
template <class T> class DynamicHeuristic;
template <class T> class LookaheadHeuristic;


static HeuristicRegistrar heuristics_s = HeuristicRegistrar() 
  .add(Register<class SelectFirst>("always selects the first node"))
  .add(Register<StaticHeuristic<class NegativePreconditionCountStrat> >("rates nodes based on their negative predecessors (static)"))
  .add(Register<StaticHeuristic<class NegativeWeightArcsCountStrat> >("rates nodes based on their negative predecessors and successors (static)"))
  .add(Register<StaticHeuristic<class RandomStrat> >("rates nodes randomly (static)"))
  .add(Register<StaticHeuristic<class StaticLexOrderingStrat> >("rates nodes lexicographically (static)"))
  .add(Register<StaticHeuristic<class RandomHeadNodeStrat> >("like SRand but prefers head nodes"))

  .add(Register<DynamicHeuristic<class RandomStrat> >("selects a node randomly (dynamic)"))
  .add(Register<DynamicHeuristic<class RandomHeadNodeStrat> >("like DRand but prefers head nodes (dynamic)"))
  .add(Register<class Neighborhood>("rates nodes based on lookahead restricted to their neighborhood"))

  .add(Register<LookaheadHeuristic<class HybridLookaheadHeuristicStrat> >("rates nodes based on the result of hybrid lookahead (lookahead)"))
  .add(Register<LookaheadHeuristic<class BodyLookaheadHeuristicStrat> >("rates nodes based on the result of body lookahead (lookahead)"))
;
///////////////////////////////////////////////////////////////////////////////

}

