/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#include <heuristics/neighborhood.h>
#include <graph.h>
#include <operators/forward_prop.h>
#include <operators/backward_prop.h>

#include <limits>
#include <algorithm>
#include <operators/lookahead.h>



namespace NS_NOMORE { namespace {
  void release(Operator* o, bool doDelete) { if (doDelete) delete o; }

  struct PAdapter {
    PAdapter(ForwardPropagator& p) : p_(&p) {}
    bool operator()(const HeadNode& h) const { return p_->propagate(h); }
    bool operator()(const BodyNode& b) const { return p_->propagate(b); }
    ForwardPropagator* p_;
  };
  struct BAdapter {
    BAdapter(BackwardPropagator& b) : b_(&b) {}
    bool operator()(const HeadNode& h) const { return b_->backpropagate(h); }
    bool operator()(const BodyNode& b) const { return b_->backpropagate(b); }
    BackwardPropagator* b_;
  };

  template <class Q, class Op>
  bool propagate(Q& q, const Op& op) {
    typedef typename Q::value_type NT;
    while (!q.empty()) {
      NT node = q.front();
      q.pop();
      if (! op(*node) ) {
        return false;
      }
    }
    return true;
  } 

}
///////////////////////////////////////////////////////////////////////////////
template <>
const char* getName(Neighborhood*) {
  return "Neighborhood";
}

template <>
Heuristic* createHeuristic(Neighborhood*) {
  return new Neighborhood();
}
///////////////////////////////////////////////////////////////////////////////
Neighborhood::Neighborhood(int m)
  : graph_(0)
  , bestNode_(0, ValueType())
  , currentValue_(0)
  , max_(m)
  , lastStep_(false) {

}

Neighborhood::~Neighborhood() {
  release(p_.first, p_.second);
  release(b_.first, b_.second);
}

void Neighborhood::setGraph(Graph& g) {
  graph_ = &g;
  graph_->checkTightness();
}

void Neighborhood::setMaxDepth(int m) {
  if (m > 0) {
    max_ = m;
  }
}

void Neighborhood::setPropagationOperator(Operator& op) {
  Operator* p = op.extractOperator(NS_NOMORE::getName<ForwardPropagator>(0));
  Operator* b = op.extractOperator(NS_NOMORE::getName<BackwardPropagator>(0));
  if (p_.first != p) {
    release(p_.first, p_.second);
  }
  if (b_.first != b) {
    release(b_.first, b_.second);
  }
  // share existing operators but own new ones.
  p_.first = p ? static_cast<ForwardPropagator*>(p) : new ForwardPropagator(*graph_);
  p_.second = p == 0;

  b_.first = b ? static_cast<BackwardPropagator*>(b) : new BackwardPropagator(*graph_);
  b_.second = b == 0;
}

void Neighborhood::registerEventHandlers() {
  event::ChannelManager& m = graph_->getEventManager();
  m.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);
  
}

void Neighborhood::unregisterEventHandlers() {
  event::ChannelManager& m = graph_->getEventManager();
  m.getChannelFor(event::Event<HeadNodeColored>()).disconnect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
}

Choice Neighborhood::selectNode(const Constraint& c) {
  bestNode_.first = 0;
  bestNode_.second = ValueType();
  currentValue_ = 0;

  registerEventHandlers();
  graph_->getEventManager().dispatch(LookaheadExecution(ActivityState::started));

  typedef std::pair<ForwardPropagator::BodyQueue, ForwardPropagator::HeadQueue> PQueue;
  typedef std::pair<BackwardPropagator::BodyQueue, BackwardPropagator::HeadQueue> BQueue;
  PQueue oldPQueue;
  BQueue oldBQueue;
  p_.first->swapQueues(oldPQueue.first, oldPQueue.second);
  b_.first->swapQueues(oldBQueue.first, oldBQueue.second);
  Graph::UncoloredNodeIterator end = graph_->uncoloredNodesEnd();
  Constraint::Type type = c.getType();
  for (Graph::UncoloredNodeIterator it = graph_->uncoloredNodesBegin(); it != end; ++it) {
    if (c.isSatisfiedBy(**it) && !propagate(*it, type)) {
      break; // select the node that causes a conflict
    }
  }

  graph_->getEventManager().dispatch(LookaheadExecution(ActivityState::finished));
  unregisterEventHandlers();

  p_.first->swapQueues(oldPQueue.first, oldPQueue.second);
  b_.first->swapQueues(oldBQueue.first, oldBQueue.second);
  return getBestChoice(type);
}

Choice Neighborhood::getBestChoice(Constraint::Type t) const {
  if (!bestNode_.first)
    return Choice();
 
  if (t == Constraint::uniform_constraint && bestNode_.second.second > bestNode_.second.first) {
    return Choice(bestNode_.first
      , bestNode_.first->alternativeChoicePointColor()
      , bestNode_.first->preferredChoicePointColor()
    ); 
  }
  return Choice(bestNode_.first); 
}

bool Neighborhood::propagate(Node* n, Constraint::Type t) {
  // color node as choice point -> a new stack is used
  Color::ColorValue colors[2] = {n->preferredChoicePointColor(), n->alternativeChoicePointColor()};
  int counter[2] = {0, 0};
  int maxColors = t == Constraint::uniform_constraint ? 2 : 1;
  bool conflict = false;
  for (int i = 0; i < maxColors && !conflict; ++i) {
    currentValue_ = 0;    
    bool cpColored = graph_->colorChoicePoint(*n, colors[i], Color::none);
    if (!cpColored || !propagateNewColor(n)) {
      counter[i] = std::numeric_limits<int>::max();
      printf("conflict found!\n");
      conflict = true;
    }
    else {
      counter[i] = currentValue_;
    }
    if (cpColored)
      graph_->restoreChoicePoint();
  }
  
  evaluate(n, ValueType(counter[0], counter[1]));
  return !conflict;

}

bool Neighborhood::propagateNewColor(Node* n) {
  typedef std::pair<ForwardPropagator::BodyQueue, ForwardPropagator::HeadQueue> PQueue;
  typedef std::pair<BackwardPropagator::BodyQueue, BackwardPropagator::HeadQueue> BQueue;
  PQueue runPQueue;
  BQueue runBQueue;
  
  if (n->getType() == NodeType::body_node) {
    runPQueue.first.push(static_cast<BodyNode*>(n));
    runBQueue.first.push(static_cast<BodyNode*>(n));
  } 
  else if (n->getType() == NodeType::head_node) {
    runPQueue.second.push(static_cast<HeadNode*>(n));
    runBQueue.second.push(static_cast<HeadNode*>(n));
  }
  bool error = false;
  PAdapter pa(*p_.first);
  BAdapter ba(*b_.first);
  for (int i = 0; i < max_ && !error; ++i) {
    lastStep_ = i == (max_ - 1);
    error = !NS_NOMORE::propagate(runPQueue.first, pa)
          || !NS_NOMORE::propagate(runPQueue.second, pa)
          || !NS_NOMORE::propagate(runBQueue.first, ba)
          || !NS_NOMORE::propagate(runBQueue.second, ba);
    p_.first->swapQueues(runPQueue.first, runPQueue.second);
    b_.first->swapQueues(runBQueue.first, runBQueue.second);
  }
  return !error;
}

void Neighborhood::handle(const HeadNodeColored& e) {
  ++currentValue_;
  if (lastStep_ && e.node_->getColor() == Color::minus) {
    ++currentValue_;
  }
}

void Neighborhood::handle(const BodyNodeColored& e) {
  ++currentValue_;
  if (lastStep_ && HasColor(Color::plus | Color::weak_plus)(e.node_)) {
    ++currentValue_;
  }
}

void Neighborhood::evaluate(Node* n, ValueType vt) {
  double tf = graph_->getTightnessFactor(n);
  vt.first *= tf;
  vt.second *= tf;
  if (bestNode_.first == 0 || vt.first + vt.second > bestNode_.second.first + bestNode_.second.second) {
    bestNode_.first = n;
    bestNode_.second = vt;
  }
}

const char* Neighborhood::getName() const {
  return NS_NOMORE::getName<Neighborhood>(0);
}

}

