/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_DYNAMIC_HEURISTIC_H
#define LIBNOMORE_DYNAMIC_HEURISTIC_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <heuristic.h>
#include <graph.h>
namespace NS_NOMORE {

//! base class for dynamic heuristics
/*!
 * To determine the best choice point a dynamic heuristic first evaluates 
 * the heuristic value of each candidate node and then selects the node with 
 * the "best" heuristic value.
 * 
 * I.e. a dynamic heuristic reevaluates the heuristic value of nodes each time
 * a choice point is requested.
 *
 * \par Template parameter: \n
 * - HeuStrat: The policy (aka strategy) to be used to evaluate nodes and to
 * compare two heuristic values.
 * .
 * HeuStrat must provide the following types/typedefs:
 * - ValueType: The type of the heuristic value used by this heuristic (i.e. int)
 * - ComparatorType: A function-type that accepts two values of type ValueType and
 * returns true if the first value is "better" than the second.
 * - EvaluatorType: A function-type that accepts a node.
 * The EvaluatorType shall install a heuristic value of type ValueType in each
 * node for which it is called.
 *
 */
template <class HeuStrat>
class DynamicHeuristic : public Heuristic {
public:
  //! the type of the binary predicate to be used for value comparisons
  typedef typename HeuStrat::ComparatorType ComparatorType;
  
  //! the type of the unary predicate to be used for valuation of nodes.
  typedef typename HeuStrat::EvaluatorType EvaluatorType;
  
  //! the type of the heuristic value
  typedef typename HeuStrat::ValueType ValueType;

  typedef typename HeuStrat::ChoiceCreatorType CreatorType;
  
  //! creates a new object
  /*! 
   * \param c A binary predicate used to compare two values of type ValueType
   * \param e An unary predicate used to valuate nodes.
   */
  DynamicHeuristic(const ComparatorType& c = ComparatorType(), const EvaluatorType& e = EvaluatorType())
    : comparator_(c)
    , evaluator_(e) {
  }
  const char* getName() const {
    return NS_NOMORE::getName<DynamicHeuristic>(0);
  }
private:  
  void doSetGraph() {
    // needed because this heuristic makes use of the tightness factor
    getGraph().checkTightness();
  }
  
  Choice doSelectNode(const Constraint& c) {
    ValueType bestValue = ValueType();
    Node* bestNode = 0;
    
    for (Graph::UncoloredNodeIterator it = getGraph().uncoloredNodesBegin(); it != it.end(); ++it) {
      if (c.isSatisfiedBy(**it)) {
        ValueType currentValue = evaluator_(*it, getGraph().getTightnessFactor(*it));
        if (bestNode == 0 || comparator_(currentValue, bestValue)) {
          bestNode = *it;
          bestValue = currentValue;
        } 
      }
    }
    return bestNode ? choiceCreator_.createChoice(bestNode) : Choice();
  }
  ComparatorType comparator_;
  EvaluatorType evaluator_;
  CreatorType choiceCreator_;
};

}
#endif
