/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef LIBNOMORE_HEURISTIC_H
#define LIBNOMORE_HEURISTIC_H
#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif
#include <node.h>
#include <list>
#include <vector>
#include <constraint.h>
#include <cassert>

namespace event {
  template <class E> class Channel;
}
namespace NS_NOMORE {

class BodyNode;
class HeadNode;
class Graph;
class Operator;

class Choice {
public:
  Choice()
    : node_(0) {
  }
  Choice(Node* n) 
    : node_(n) {
    if (node_) {
      pref_ = node_->preferredChoicePointColor();
      alt_ = node_->alternativeChoicePointColor();
    }
  }
  Choice(Node* n, Color::ColorValue preferred, Color::ColorValue alternative)
    : node_(n)
    , pref_(preferred)
    , alt_(alternative) {
  }
  Node* node() const {return node_;}
  Color::ColorValue preferredColor() const {assert(isValid()); return pref_;}
  Color::ColorValue alternativeColor() const {assert(isValid()); return alt_;}
  bool isValid() const { return node_ != 0; }
  
  operator bool() const { return isValid();}
private:
  Node* node_;
  Color::ColorValue pref_;
  Color::ColorValue alt_;
};

//! Defines the possible states of an activity for example the LookaheadExecution event.
struct ActivityState {

  //! Values of possible state.
  enum Value {
    //! the activity is started.
    started = 1,
    //! the activity is finished.
    finished = 2,
  };
};

//! Event which is fired whenever a heuristic starts or stops execution.
struct HeuristicExecution {
  explicit HeuristicExecution(ActivityState::Value state) : state_(state) {
	}
	ActivityState::Value state_;
};


//! Base class for Heuristics.
/*!
 * heuristics guide the selection of choice points. They are used together
 * with a \ref ChoiceOperator "choice-operator".
 */
class Heuristic {
public: 
  //! Constructs the heuristics.
  Heuristic();

  //! Destroys the heuristics.
  virtual ~Heuristic();
  
  //! sets the graph on which this heuristic should work.
  void setGraph(Graph& g);
  
  //! gives the heuristic a chance to see the propagation operators used.
  /*!
   * The default implementation does nothing.
   *
   * \note this method is a hack currently needed only in the Neighborhood-Heuristic.
   * \todo find a better place for this method
   */
  virtual void setPropagationOperator(Operator& op);

  virtual const char* getName() const = 0;

  Choice selectNode(const Constraint& c);
protected:
  Graph& getGraph()  {
    assert(graph_);
    return *graph_;
  }
private:
  //! returns the "best" choice according to this heuristic that satisfies the given constraint.
  /*!
   * \param c the constraint a potential best node must satisfy.
   * \return the "best" choice (containing a head or body node) according to this heuristic or 
   * invalid choice if no such node exists.
   *
   * \see Choice::isValid
   */
  virtual Choice doSelectNode(const Constraint& c) = 0;

  virtual void doSetGraph();

  Graph* graph_;

  event::Channel<HeuristicExecution>* channel_;

};

/*!
 * This heuristic selects the first uncolored node that satisfies the
 * the given constraint.
 */
class SelectFirst : public Heuristic {
public:
  SelectFirst();
private:
  Choice doSelectNode(const Constraint& c);

  virtual const char* getName() const;
};

typedef SelectFirst NoHeuristic;

struct HeuristicValuesEvent {
  enum Action {
    save,
    restore
  };

  HeuristicValuesEvent(Action action) : action_(action) {    
  }

  Action action_;
};

struct DefaultChoiceCreator {
  Choice createChoice(Node* n) const {
    return Choice(n);
  }
};


template <class T>
const char* getName(T*);

template <class T>
Heuristic* createHeuristic(T*);



}

#endif
