/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#include <heuristics/neighborhood.h>
#include <graph.h>
#include <operators/forward_prop.h>
#include <operators/backward_prop.h>

#include <limits>
#include <algorithm>
#include <operators/lookahead.h>
#include <util/overlay_list.h>

namespace NS_NOMORE { namespace {
  
  void release(Operator* o, bool doDelete) { if (doDelete) delete o; }

  template <class QueuePair>
  inline void clear(QueuePair& p) {
    while (!p.first.empty()) p.first.pop();
    while (!p.second.empty()) p.second.pop();
  }

  struct PAdapter {
    PAdapter(ForwardPropagator& p) : p_(&p) {}
    bool operator()(const HeadNode& h) const { return p_->propagate(h); }
    bool operator()(const BodyNode& b) const { return p_->propagate(b); }
    ForwardPropagator* p_;
  };
  struct BAdapter {
    BAdapter(BackwardPropagator& b) : b_(&b) {}
    bool operator()(const HeadNode& h) const { return b_->backpropagate(h); }
    bool operator()(const BodyNode& b) const { return b_->backpropagate(b); }
    BackwardPropagator* b_;
  };

  template <class Q, class Op>
  bool propagate(Q& q, const Op& op) {
    typedef typename Q::value_type NT;
    while (!q.empty()) {
      NT node = q.front();
      q.pop();
      if (! op(*node) ) {
        return false;
      }
    }
    return true;
  } 

  static StaticOverlayList overlayList_g(0);
}
///////////////////////////////////////////////////////////////////////////////
template <>
const char* getName(Neighborhood*) {
  return "Neighborhood";
}

template <>
Heuristic* createHeuristic(Neighborhood*) {
  return new Neighborhood();
}
///////////////////////////////////////////////////////////////////////////////
Neighborhood::Neighborhood(int m)
  : p_(0, false)
  , b_(0, false)
  , max_(m)
  , currentValue_(0)
  , lastStep_(false)
  , bestNode_(0, ValueType()) {

}

Neighborhood::~Neighborhood() {
  release(p_.first, p_.second);
  release(b_.first, b_.second);
}

void Neighborhood::doSetGraph() {
  getGraph().checkTightness();
}

void Neighborhood::setMaxDepth(int m) {
  if (m > 0) {
    max_ = m;
  }
}

void Neighborhood::setPropagationOperator(Operator& op) {
  Operator* p = op.extractOperator(NS_NOMORE::getName<ForwardPropagator>(0));
  Operator* b = op.extractOperator(NS_NOMORE::getName<BackwardPropagator>(0));
  if (p_.first != p) {
    release(p_.first, p_.second);
  }
  if (b_.first != b) {
    release(b_.first, b_.second);
  }
  // share existing operators but own new ones.
  p_.first = p ? static_cast<ForwardPropagator*>(p) : new ForwardPropagator(getGraph());
  p_.second = p == 0;

  b_.first = b ? static_cast<BackwardPropagator*>(b) : new BackwardPropagator(getGraph());
  b_.second = b == 0;
}

void Neighborhood::registerEventHandlers() {
  event::ChannelManager& m = getGraph().getEventManager();
  m.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);
  
}

void Neighborhood::unregisterEventHandlers() {
  event::ChannelManager& m = getGraph().getEventManager();
  m.getChannelFor(event::Event<HeadNodeColored>()).disconnect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
}

Choice Neighborhood::doSelectNode(const Constraint& c) {
  overlayList_g.resize(getGraph().countNodes());
  bestNode_.first = 0;
  bestNode_.second = ValueType();
  currentValue_ = 0;

  registerEventHandlers();

  clear(savePQueue_);
  clear(saveBQueue_);
  
  p_.first->swapQueues(savePQueue_.first, savePQueue_.second);
  b_.first->swapQueues(saveBQueue_.first, saveBQueue_.second);
  Graph::UncoloredNodeIterator end = getGraph().uncoloredNodesEnd();
  Constraint::Type type = c.getType();
  for (Graph::UncoloredNodeIterator it = getGraph().uncoloredNodesBegin(); it != end; ++it) {
    if (c.isSatisfiedBy(**it) && !propagate(*it, type)) {
      break; // select the node that causes a conflict
    }
  }

  unregisterEventHandlers();

  p_.first->swapQueues(savePQueue_.first, savePQueue_.second);
  b_.first->swapQueues(saveBQueue_.first, saveBQueue_.second);
  overlayList_g.clear();
  
  return getBestChoice(type);
}

Choice Neighborhood::getBestChoice(Constraint::Type t) const {
  if (!bestNode_.first)
    return Choice();
 
  if (t == Constraint::uniform_constraint && bestNode_.second.second > bestNode_.second.first) {
    return Choice(bestNode_.first
      , bestNode_.first->alternativeChoicePointColor()
      , bestNode_.first->preferredChoicePointColor()
    ); 
  }
  return Choice(bestNode_.first); 
}

bool Neighborhood::propagate(Node* n, Constraint::Type t) {
  // color node as choice point -> a new stack is used
  Color::ColorValue colors[2] = {n->preferredChoicePointColor(), n->alternativeChoicePointColor()};
  int counter[2] = {0, 0};
  int maxColors = t == Constraint::uniform_constraint ? 2 : 1;
  bool conflict = false;
  for (int i = 0; i < maxColors && !conflict; ++i) {
    if (overlayList_g.isOverlayed(*n, colors[i]))
      continue;

    currentValue_ = 0;    
    bool cpColored = getGraph().colorChoicePoint(*n, colors[i], Color::none, ColorOpType::first_choice);
    if (!cpColored || !propagateNewColor(n)) {
      counter[i] = std::numeric_limits<int>::max();
      conflict = true;
    }
    else {
      counter[i] = currentValue_;
    }
    if (cpColored)
      getGraph().restoreChoicePoint();
  }
  
  evaluate(n, ValueType(counter[0], counter[1]));
  return !conflict;

}

bool Neighborhood::propagateNewColor(Node* n) {
  clear(runPQueue_);
  clear(runBQueue_);
  
  if (n->getType() == NodeType::body_node) {
    runPQueue_.first.push(static_cast<BodyNode*>(n));
    runBQueue_.first.push(static_cast<BodyNode*>(n));
  } 
  else if (n->getType() == NodeType::head_node) {
    runPQueue_.second.push(static_cast<HeadNode*>(n));
    runBQueue_.second.push(static_cast<HeadNode*>(n));
  }
  bool error = false;
  PAdapter pa(*p_.first);
  BAdapter ba(*b_.first);
  for (int i = 0; i < max_ && !error; ++i) {
    lastStep_ = i == (max_ - 1);
    error = !NS_NOMORE::propagate(runPQueue_.first, pa)
          || !NS_NOMORE::propagate(runPQueue_.second, pa)
          || !NS_NOMORE::propagate(runBQueue_.first, ba)
          || !NS_NOMORE::propagate(runBQueue_.second, ba);
    p_.first->swapQueues(runPQueue_.first, runPQueue_.second);
    b_.first->swapQueues(runBQueue_.first, runBQueue_.second);
  }
  return !error;
}

void Neighborhood::handle(const HeadNodeColored& e) {
  ++currentValue_;
  if (lastStep_ && e.node_->getColor() == Color::minus) {
    ++currentValue_;
  }
  overlayList_g.add(*e.node_, e.node_->getColor());
}

void Neighborhood::handle(const BodyNodeColored& e) {
  ++currentValue_;
  if (HasColor(Color::plus | Color::weak_plus)(e.node_)) {
    overlayList_g.add(*e.node_, Color::plus);
    overlayList_g.add(*e.node_, Color::weak_plus);
    if (lastStep_)
      ++currentValue_;
  }
  else
    overlayList_g.add(*e.node_, e.node_->getColor());
}

void Neighborhood::evaluate(Node* n, ValueType vt) {
  double tf = getGraph().getTightnessFactor(n);
  vt.first = (int)(vt.first * tf);
  vt.second = (int)(vt.second * tf);
  if (bestNode_.first == 0 || vt.first + vt.second > bestNode_.second.first + bestNode_.second.second) {
    bestNode_.first = n;
    bestNode_.second = vt;
  }
}

const char* Neighborhood::getName() const {
  return NS_NOMORE::getName<Neighborhood>(0);
}

}

