/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#include <statistics/timing_statistics.h>
#include <event/channel_manager.h>
#include <operators/lookahead.h>
#include <heuristic.h>
#include <graph.h>
#include <iomanip>

namespace NS_NOMORE {

TimingStatistics::TimingStatistics(event::ChannelManager& m) 
  : StatisticsObject()
  , startGraphConstruction_(0) 
  , endGraphConstruction_(0) 
  , graphConstruction_(0) 

  , startPropagation_(0) 
  , endPropagation_(0) 
  , propagation_(0) 

  , startLookahead_(0) 
  , endLookahead_(0) 
  , lookahead_(0) 

  , startHeuristic_(0) 
  , endHeuristic_(0) 
  , heuristic_(0) 

  , startExecution_(0) 
  , endExecution_(0) 
  , execution_(0) {

  startExecution_ = readTimeStampCounter();
  startGraphConstruction_ = readTimeStampCounter();

  registerWith(m);  
}

TimingStatistics::~TimingStatistics() {
}

void TimingStatistics::registerWith(event::ChannelManager& m) {
  m.getChannelFor(event::Event<LookaheadExecution>()).connect(*this);
  m.getChannelFor(event::Event<HeuristicExecution>()).connect(*this);
  m.getChannelFor(event::Event<GraphConstructionFinished>()).connect(*this);
}

void TimingStatistics::handle(const LookaheadExecution& e) {
  if(e.state_ == ActivityState::started) {
    endPropagation_ = readTimeStampCounter();
    propagation_ += endPropagation_ - startPropagation_;

    startLookahead_ = readTimeStampCounter();
  }
  else if(e.state_ == ActivityState::finished) {
    endLookahead_ = readTimeStampCounter();
    lookahead_ += endLookahead_ - startLookahead_;

    startPropagation_ = readTimeStampCounter();
  }
}

void TimingStatistics::handle(const HeuristicExecution& e) {
  if(e.state_ == ActivityState::started) {
    endPropagation_ = readTimeStampCounter();
    propagation_ += endPropagation_ - startPropagation_;

    startHeuristic_ = readTimeStampCounter();
  }
  else if(e.state_ == ActivityState::finished) {
    endHeuristic_ = readTimeStampCounter();
    heuristic_ += endHeuristic_ - startHeuristic_;

    startPropagation_ = readTimeStampCounter();
  }
}

void TimingStatistics::handle(const GraphConstructionFinished& e) {
  endGraphConstruction_ = readTimeStampCounter();
  graphConstruction_ = endGraphConstruction_ - startGraphConstruction_;

  startPropagation_ = readTimeStampCounter();
}

void TimingStatistics::print(std::ostream& os) const {
  uint64 prop = propagation_ + readTimeStampCounter() - startPropagation_;
  uint64 exec = readTimeStampCounter() - startExecution_;
#if defined(_MSC_VER) && _MSC_VER <= 1200
  __int64 iexec = exec;
  float ratePropagation = static_cast<__int64>(prop) * 100.0f / iexec;
  float rateLookahead = static_cast<__int64>(lookahead_) * 100.0f / iexec;
  float rateHeuristic = static_cast<__int64>(heuristic_) * 100.0f / iexec;
  float rateGraphConstruction = static_cast<__int64>(graphConstruction_) * 100.0f / iexec;
#else
  float ratePropagation = prop * 100.0f / exec;
  float rateLookahead = lookahead_ * 100.0f / exec;
  float rateHeuristic = heuristic_ * 100.0f / exec;
  float rateGraphConstruction = graphConstruction_ * 100.0f / exec;
#endif
  
  char sRateProp[5]; sprintf(sRateProp, "%.2f", ratePropagation);
  char sRateLA[5];   sprintf(sRateLA, "%.2f", rateLookahead);
  char sRateHeu[5];  sprintf(sRateHeu, "%.2f", rateHeuristic);
  char sRateGC[5];   sprintf(sRateGC, "%.2f", rateGraphConstruction);

  os << "Timings\n";
  os << "  graph construction: " << sRateGC << "%\n";
  os << "  lookahead execution: " << sRateLA << "%\n";
  os << "  node selection: " << sRateHeu << "%\n";
  os << "  model generation: " << sRateProp << "%\n";
}

}
