#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <node.h>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

// MockNode is a dummy node that checks some expectations
class MockNode : public NS_NOMORE::Node {
public:
  MockNode(long id)
    : Node(id, NodeType::head_node)
    , dscExpected_(new Color::ColorValue(Color::none))
    , drcExpected_(new Color::ColorValue(Color::none)) {
  }
  ~MockNode() {
    delete dscExpected_;
    delete drcExpected_;
  }

  virtual Color::ColorValue recursiveSupportColor() const {
    return Color::plus;
  }

  // Node Interface
  ColorOpResult setColor(Color::ColorValue c) {
    return Node::setColor(c);
  }

  void doSetColor(Color::ColorValue oldColor) {
    CPPUNIT_ASSERT(dscExpected_);
    CPPUNIT_ASSERT_EQUAL(*dscExpected_, oldColor);
  }

  void restoreColor(Color::ColorValue c) {
    Node::restoreColor(c);
  }

  void doRestoreColor(Color::ColorValue oldColor) {
    CPPUNIT_ASSERT(drcExpected_);
    CPPUNIT_ASSERT_EQUAL(*drcExpected_, oldColor);
  }
  void fireColorEvent(event::ChannelManager&, Color::ColorValue) {
  }
  void fireRestoreEvent(event::ChannelManager&, Color::ColorValue) {
  }
  void fireColorChoicePointEvent(event::ChannelManager&, Color::ColorValue, ColorOpType::Value) {
  }
  void propagate(Graph&) {}

  // Test Interface
  void setColorExpectation(Color::ColorValue* c) {
    delete dscExpected_;
    dscExpected_ = c;
  }

  virtual Color::ColorValue preferredChoicePointColor() const {
    return recursiveSupportColor();
  }

  virtual Color::ColorValue alternativeChoicePointColor() const {
    return Color::minus;
  }

  void restoreColorExpectation(Color::ColorValue* c) {
    delete drcExpected_;
    drcExpected_ = c;
  }
  bool trivialComponent() const {
    return false;
  }
private:
  Color::ColorValue* dscExpected_;
  Color::ColorValue* drcExpected_;
};

class TestNode : public CppUnit::TestFixture {
public:
  CPPUNIT_TEST_SUITE(TestNode);
  CPPUNIT_TEST(testInit);
  CPPUNIT_TEST(testIsValidColoring);
  CPPUNIT_TEST(testSetColorValid);
  CPPUNIT_TEST(testSetColorInvalid);
  CPPUNIT_TEST(testSetColorNoop);
  CPPUNIT_TEST(testRestoreColorToNone);
  CPPUNIT_TEST(testRestoreColorToWeakPlus);
  CPPUNIT_TEST_SUITE_END();

  void testInit() {
    MockNode n(1);
    CPPUNIT_ASSERT_EQUAL(1l, n.getId());
    CPPUNIT_ASSERT_EQUAL(Color::none, n.getColor());
  }

  // Vollstndiger Test fr IsValidColoring
  // Auf diesen Test bauen wir auf.
  void testIsValidColoring() {
    struct Coloring {
      Color::ColorValue to;
      bool valid;
    };
    
    // X -> none is not allowed
    Coloring colorings[12] = {
      // none -> x
        {Color::minus, true}
      , {Color::plus, true}
      , {Color::weak_plus, true}
      // minus -> x
      , {Color::minus, false}
      , {Color::plus, false}
      , {Color::weak_plus, false}
      // plus -> x
      , {Color::minus, false}
      , {Color::plus, false}
      , {Color::weak_plus, false}
      // weak_plus -> x
      , {Color::minus, false}
      , {Color::plus, true}
      , {Color::weak_plus, false}
    };
    Color::ColorValue colors[4] = {
      Color::none, Color::minus, Color::plus, Color::weak_plus
    };
    for (int i = 0; i != 4; ++i) {
      for (int j = 0; j != 3; ++j) {
        int testCase = (i * 3) + j;
        CPPUNIT_ASSERT_EQUAL(colorings[testCase].valid
          , isValidColoring(colors[i], colorings[testCase].to));
      }
    }
  }

  void testSetColorValid() {
    MockNode n(1);
    n.setColorExpectation(new Color::ColorValue(Color::none));
    CPPUNIT_ASSERT_EQUAL(ColorOperation::succeeded, n.setColor(Color::plus));
    CPPUNIT_ASSERT_EQUAL(Color::plus, n.getColor());
  }

  void testSetColorInvalid() {
    MockNode n(1);
    n.setColor(Color::plus);
    n.setColorExpectation(0); // wir erwarten das doSetColor nicht aufgerufen wird
    
    CPPUNIT_ASSERT_EQUAL(ColorOperation::failed, n.setColor(Color::minus));
    CPPUNIT_ASSERT_EQUAL(Color::plus, n.getColor());
  }

  
  void testSetColorNoop() {
    MockNode n(1);
    n.setColor(Color::plus);

    n.setColorExpectation(0);
    CPPUNIT_ASSERT_EQUAL(ColorOperation::unchanged, n.setColor(Color::plus));
  }

  
  void testRestoreColorToNone() {
    MockNode n(1);
    n.setColor(Color::plus);

    n.restoreColorExpectation(new Color::ColorValue(Color::plus));
    n.restoreColor(Color::none);
    CPPUNIT_ASSERT_EQUAL(Color::none, n.getColor());
  }

  void testRestoreColorToWeakPlus() {
    MockNode n(1);
    n.setColor(Color::minus);

    n.restoreColorExpectation(new Color::ColorValue(Color::minus));
    n.restoreColor(Color::weak_plus);
    CPPUNIT_ASSERT_EQUAL(Color::weak_plus, n.getColor());
  }
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestNode);

}	// end namespace NS_NOMORE_TESTS
