/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef NOMORE_CHOICE_OPERATOR_H
#define NOMORE_CHOICE_OPERATOR_H

#include <operator.h>
#include <color.h>
#include <heuristic.h>

namespace NS_NOMORE {

class Node;
class Constraint;

//! Base class for all non deterministic choice operators.
/*! 
 *  ChoiceOperator implements the operator()() method of the base class 
 *  Operator. Concrete sub classes must override the method selectChoice for 
 *  choosing the next node to be colored as choice point. 
 *  
 *  There are two ways to use a concrete choice operator.
 *
 *  - <b>Method 1:</b> Use the choice operator just like any other operator
 *    by calling its function operator. This way an uncolored node is selected
 *    (using the installed heuristics) and automatically colored as choice point. 
 *    \code
 *      ColorOpResult result = anChoiceOperator();
 *      if (result == ColorOperation::succeeded) {
 *        // choice point found and successfully colored
 *      }
 *      else if (result == ColorOperation::failed) {
 *        // choice point found but color error while coloring the node
 *      }
 *      else {
 *        // no choice point found
 *      }
 *    \endcode
 *    
 *  - <b>Method 2:</b> Select and color a node manually. To do this call the 
 *    selectChoice method then color the returned node (e.g. by calling 
 *    graph::colorChoicePoint). selectChoice returns a null-ptr if no choices 
 *    are left.
 *    \code
 *      node = anChoiceOperator.selectChoice();
 *      if(node == NULL) {
 *        // no choice point found
 *      } 
 *      else {
 *        graph.colorChoicePoint(*node, myFirstColor, mySecondColor);
 *      }
 *    \endcode */
class ChoiceOperator : public Operator {
public:
  //! Creates the choice operator.
  /*! 
   *  \param grp The graph the operator should work on. 
   *  \param h The heuristic to use for evaluating possible choice points */
  ChoiceOperator(Graph& grp, Heuristic* h);

  //! Destroys the choice operator.
  ~ChoiceOperator();

  //! Executes this choice operator.
  /*!
   *  The method selected a node according to the heuristics and colors the 
   *  node if possible as choice point. This base class implements the 
   *  following algorithm:
   *  \code
   *    bool result = false;
   *    Node* n = selectNode();
   *    if (n != 0) {
   *      if (graph_->colorChoicePoint(n, n->preferredChoicePointColor(), n->alternativeChoicePointColor()))
   *        result = true
   *    }
   *    return result;
   *  \endcode
   *
   * \pre choiceType is either ColorOpType::first_choice or ColorOpType::next_choice
   *
   * \note Concrete sub classes shall not override this method. Instead they 
   *       should override the hook method ChoiceOperator::selectChoice. */
  bool operator()();

  //! Executes this choice operator using the color operation type.
  /*!
   *  The method selected a node according to the heuristics and colors the 
   *  node if possible as choice point. This base class implements the 
   *  following algorithm:
   *  \code
   *    bool result = false;
   *    Node* n = selectNode();
   *    if (n != 0) {
   *      if (graph_->colorChoicePoint(n, n->preferredChoicePointColor(), n->alternativeChoicePointColor()))
   *        result = true
   *    }
   *    return result;
   *  \endcode
   *
   * \param choiceType The type the choice point is colored as.
   *
   * \note Concrete sub classes shall not override this method. Instead they 
   *       should override the hook method ChoiceOperator::selectChoice. */
  bool operator()(ColorOpType::Value choiceType);
    
  //! Selects a choice point.
  /*! 
   *  Returns a choice object containing the node to be colored as choice 
   *  point as well the choice's preferred and alternative choice color. 
   *
   *  \return The choice which is made using the heuristics. */
  virtual Choice selectChoice() = 0;

  //! Returns the constraint the choice must satisfy.
  /*!
   *  \return The constraint of the choice operator. */
  virtual const Constraint& getConstraint() const = 0;

  //! Returns the heuristic set by the constructor.
  /*!
   *  \return The installed heuristics of the choice operator. */
  const Heuristic& getHeuristic() const {return *heuristic_;}

  //! Returns the heuristic set by the constructor.
  /*!
   *  \return The installed heuristics of the choice operator. */
  Heuristic& getHeuristic() {return *heuristic_;}
  
private:
  // instance variables
  Graph* graph_;
  Heuristic* heuristic_;
};

//! SimpleChoiceOperator uses a given constraint and a given heuristic to choose nodes.
/*!
 *  The heuristic is used to evaluate those nodes that satisfy the given
 *  constraint. The SimpleChoiceOperator then returns the node with the best
 *  heuristic value.
 *
 *  \par Known Combinations
 *  - C  = SimpleChoiceOperator + NoConstraint: Unconditionally selects both 
 *    head and body nodes.
 *  - CH = SimpleChoiceOperator + HeadConstraint: Selects only head nodes.
 *  - CB = SimpleChoiceOperator + BodyConstraint: Select only body nodes.
 *  - D  = SimpleChoiceOperator + SupportedBodyConstraint: Selects only 
 *    supported body nodes.
 *  - DT = SimpleChoiceOperator + SupportedOrTrivialBodyConstraint: Selects 
 *    only body nodes that are either supported or trivally connected. */
class SimpleChoiceOperator : public ChoiceOperator {
public:
  //! Constructs the choice operator and sets the member variables.
  /*!
   *  \param theConstraint The constraint which must be satisfied by the 
   *                       selected node. 
   *  \param name          The name of the choice operator used for the user 
   *                       interface. 
   *  \param g             The graph the operator should work on.
   *  \param h             The heuristics used to select the best node. */
  SimpleChoiceOperator(Constraint* theConstraint, const char* name, Graph& g, Heuristic* h);

  //! Destroys the object.
  ~SimpleChoiceOperator();

  //! Selects a choice point.
  /*! 
   *  Returns a choice object containing the node to be colored as choice 
   *  point as well the choice's preferred and alternative choice color. 
   *
   *  \return The choice which is made using the heuristics. */
  Choice selectChoice();

  //! Returns the constraint the choice must satisfy.
  /*!
   *  \return The constraint of the choice operator. */
  const Constraint& getConstraint() const;

  //! Returns the name of the current choice operator.
  /*!
   *  \return The name of the choice operator. */
  const char* getName() const;

private:
  // instance variables
  Constraint* theConstraint_;
  const char* name_;
};

///////////////////////////////////////////////////////////////////////////////
// Basic choice constraints
///////////////////////////////////////////////////////////////////////////////

//! Constraint which allows all uncolored nodes.
/*!
 *  All uncolored nodes satisfying this constraint. The only condition that 
 *  must hold is the color. */
class NoConstraint : public Constraint {
public:
  //! Returns true if the given node is uncolored.
  /*!
   *  \param n The node tested for satisfying the constraint.
   *  \return True if n is uncolored. */
  bool isSatisfiedBy(const Node& n) const;

  //! Returns true if node n2 supersedes node n1 for the corresponding constraint.
  /*!
   *  \param n1 The first node.
   *  \param n2 The second node.
   *  \return Always true. */
  bool supersedes(const Node&, const Node&) const;

  //! Returns the type of the constraint.
  /*!
   *  \return Constraint::hybrid_constraint
   *
   *  \see Constraint::Type */
  Type getType() const;
};

//! Constraint which allows all uncolored head nodes.
/*!
 *  All uncolored head nodes satisfying this constraint. */
class HeadConstraint : public Constraint {
public:
  //! Returns true if the given node is uncolored and of type HeadNode.
  /*!
   *  \param n The node tested for satisfying the constraint.
   *  \return True if n is uncolored and a head node. */
  bool isSatisfiedBy(const Node& n) const;

  //! Returns true if node n2 supersedes node n1 for the corresponding constraint.
  /*!
   *  \param n1 The first node.
   *  \param n2 The second node.
   *  \return Always true. */
  bool supersedes(const Node&, const Node&) const;

  //! Returns the type of the constraint.
  /*!
   *  \return Constraint::uniform_constraint
   *
   *  \see Constraint::Type */
  Type getType() const;
};

//! Constraint which allows all uncolored body nodes.
/*!
 *  All uncolored body nodes satisfying this constraint. */
class BodyConstraint : public Constraint {
public:
  //! Returns true if the given node is uncolored and of type BodyNode.
  /*!
   *  \param n The node tested for satisfying the constraint.
   *  \return True if n is uncolored and a body node. */
  bool isSatisfiedBy(const Node& n) const;

  //! Returns true if node n2 supersedes node n1 for the corresponding constraint.
  /*!
   *  \param n1 The first node.
   *  \param n2 The second node.
   *  \return Always true. */
  bool supersedes(const Node&, const Node&) const;

  //! Returns the type of the constraint.
  /*!
   *  \return Constraint::uniform_constraint
   *
   *  \see Constraint::Type */
  Type getType() const;
};

//! Constraint which allows all uncolored and supported body nodes.
/*!
 *  All uncolored body nodes which are supported satisfying this constraint. */
class SupportedBodyConstraint : public Constraint {
public:
  //! Returns true if the given node a uncolored and supported body node.
  /*!
   *  \param n The node tested for satisfying the constraint.
   *  \return True if n is uncolored, a body node and supported. */
  bool isSatisfiedBy(const Node& n) const;
  
  //! Returns true if node n2 supersedes node n1 for the corresponding constraint.
  /*!
   *  \param n1 The first node.
   *  \param n2 The second node.
   *  \return True iff
   *          - n2 is a head node or
   *          - n1 is a head node and n2 is a supported body node or
   *          - n1 is a supported body node or
   *          - n2 is a body node that is not supported. */
  bool supersedes(const Node& n1, const Node& n2) const;

  //! Returns the type of the constraint.
  /*!
   *  \return Constraint::uniform_constraint
   *
   *  \see Constraint::Type */
  Type getType() const;
};

//! Constraint which allows all uncolored and supported or trivial body nodes.
/*!
 *  All uncolored body nodes which are supported (or in a trivial strongly 
 *  connected component) satisfying this constraint. */
class SupportedOrTrivialBodyConstraint : public Constraint {
public:
  //! Returns true if the given node a uncolored and supported (or trivial) body node.
  /*!
   *  \param n The node tested for satisfying the constraint.
   *  \return True if n is uncolored, a body node and supported (or in a 
   *          trivial component). */
  bool isSatisfiedBy(const Node& n) const;
  
  //! Returns true if node n2 supersedes node n1 for the corresponding constraint.
  /*!
   *  \param n1 The first node.
   *  \param n2 The second node.
   *  \return True iff
   *          - n2 is a head node or
   *          - n1 is a head node and n2 is a supported or trivial connected body node or
   *          - n1 is a supported or trivial connected body node or
   *          - n2 is a body node that is neither supported nor trivial connected. */
  bool supersedes(const Node& n1, const Node& n2) const;

  //! Returns the type of the constraint.
  /*!
   *  \return Constraint::uniform_constraint
   *
   *  \see Constraint::Type */
  Type getType() const;
  
};

//! Method for creating choice point operators.
/*! 
 *  \param g The graph the operator should work on.
 *  \param h The heuristics used to get the best nodes.
 *  \param T The choice point operator. 
 *  \return An instance of the choice point operator. */
template <class T>
ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h, T*);

}

#endif
