/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef NOMORE_LOOKAHEAD_OPERATOR_H
#define NOMORE_LOOKAHEAD_OPERATOR_H

#include <operator.h>
#include <color.h>
#include <heuristic.h>
#include <propagator.h>
#include <graph.h>

namespace NS_NOMORE {

class BodyNode;
class HeadNode;
class Node;
class Constraint;

//! This abstract class is the base class for all lookahead operators.
/*!
 *  Lookahead operators work in a conflict driven way. They assign a color to
 *  an uncolored node whenever the opposite assignment led to a conflict.
 *  Conflict detection is based on propagation using a given propagator. */
class Lookahead : public Operator {
public:
  //! Creates the lookahead operator.
  /*! 
   *  \param grp The graph the operator should work on. */ 
  Lookahead(Graph& grp);

  //! Destroys the object.
  ~Lookahead();
  
  //! Sets the propagator to be used by this lookahead operator.
  /*!
   *  \param prop The propagator used to extend the parial assignment.
   *  \param m    The propagation mode. Use Propagator::local to disable the
   *              unfounded set check in lookahead. */
  void setPropagator(Propagator& prop, Propagator::Mode m = Propagator::full);

  //! Sets the constraint to be used for supersede tests.
  /*!
   *  The constraint is given by the choice operator and is used to check if
   *  a node must be tested or not.
   *
   *  \param con The constraint used for supersede tests. */
  void setConstraint(const Constraint& con) {
    constraint_ = &con;
  }

  //! Gets the constraint used for supersede-tests.
  /*!
   *  \return The constraint used for supersede tests. */
  const Constraint& getConstraint() const  {
    return *constraint_;
  }
  
  //! Sets the heuristics.
  /*!
   *  \param heu The heuristics to select nodes. */
  virtual void setHeuristic(Heuristic& heu) = 0;

  //! Executes the lookahead operator.
  /*!
   *  This base class implements the following algorithm:
   *  \code
   *    1. fire an event to signal the start of this lookahead operation
   *    2. call execute();
   *    3. fire an event to signal the end of this lookahead operation
   *  \endcode
   *
   *  \note Concrete sub classes shall not override this method. Instead they 
   *        should override the hook method Operator::execute(). */
  bool operator()();

  //! Returns the graph.
  /*!
   *  \return The Graph set by the constructor. */
  Graph& getGraph() {
    assert(graph_); 
    return *graph_; 
  }
  
  //! Returns the graph.
  /*!
   *  \return The Graph set by the constructor. */
  const Graph& getGraph() const {
    assert(graph_); 
    return *graph_; 
  }

protected:
  //! Checks whether the given color would be a valid color for the given node.
  /*!
   *  This method first colors the given node to the given color and then 
   *  propagates the new color using the stored operator. If the propagation 
   *  is successful (i.e. the color is valid for the node) the method returns 
   *  true. Otherwise false is returned.
   *
   *  \param node The node to check.
   *  \param color The color to which the node should be colored.
   *  \return True if the color is a valid color for node. False if setting 
   *          node to color leads to a conflict.
   *
   *  \post All temporarily made colorings are restored before this method 
   *        returns. */
  bool checkColor(Node &node, Color::ColorValue color);
  
  //! Colors the given node to the given color and propagates the new color using the stored operator.
  /*!
   *  This method first colors the given node to the given color and than 
   *  propagates using the stored propagator. The resulting assignment is not 
   *  restored before existing this method.
   *
   *  \param node The node to color.
   *  \param color The color to which the node should be colored.
   *  \return False if coloring node to color leads to a conflict. True 
   *          otherwise. */
  bool color(Node &node, Color::ColorValue color);

  //! Abstract method for the concrete lookahead implementation.
  /*!
   *  This method must be override by a sub class and implements the concrete
   *  lookahead using the methods checkColor() and color() of this base class. */
  virtual void execute() = 0;

private:
  // instance variables
  Graph* graph_;
  Propagator* propagator_;
  const Constraint* constraint_;
  Propagator::Mode propMode_;
};

//! Method for executing lookahead over a list of nodes for a specified lookahead.
/*!
 *  Takes all nodes used for lookahead and checks every node with the 
 *  specified lookahead operator. 
 *
 *  \param op The lookahead operator used to look ahead. */
template <class LookOp>
void lookahead(LookOp& op) {
  typedef typename LookOp::iterator_type iterator_type;
  Graph& g = op.getGraph();
  bool repeat;
  do {
    repeat = false;
    iterator_type end = op.end();
    for (iterator_type first = op.begin(); first != end; ++first) {
      if (op.lookahead(*first)) {
        repeat = true;
      }
      if (g.hasConflict())
        return;
    } // for each node
  } while (repeat);
}

//! Event which is fired if the lookahead is started or finished.
struct LookaheadExecution {
  //! Initializes the event.
  /*!
   *  \param state The current state of computation.
   *  \see ActivityState */
  explicit LookaheadExecution(ActivityState::Value state) : state_(state) {
	}
  
  //! The activity state in which the solver currently is.
	ActivityState::Value state_; 
};

//! Creates the lookahead operator.
/*!
 *  \param g The graph used to look ahead.
 *  \param T The lookahead operator to create.
 *  \return An instance of the lookahead operator T. */
template <class T>
Lookahead* createLookaheadOp(Graph& g, T*);

}

#endif
