/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#include <statistics/timing_statistics.h>
#include <operators/lookahead.h>
#include <heuristic.h>

#include <iomanip>

namespace NS_NOMORE {

TimingStatistics::TimingStatistics(event::ChannelManager& m) 
  : StatisticsObject()
  , startGraphConstruction_(0) 
  , endGraphConstruction_(0) 
  , graphConstruction_(0) 

  , startPropagation_(0) 
  , endPropagation_(0) 
  , propagation_(0) 

  , startLookahead_(0) 
  , endLookahead_(0) 
  , lookahead_(0) 

  , startHeuristic_(0) 
  , endHeuristic_(0) 
  , heuristic_(0) 

  , startExecution_(0) 
  , endExecution_(0) 
  , execution_(0) {

  startExecution_ = readMicroSecCounter();
  startGraphConstruction_ = readMicroSecCounter();

  registerWith(m);  
}

TimingStatistics::~TimingStatistics() {
}

void TimingStatistics::registerWith(event::ChannelManager& m) {
  m.getChannelFor(event::Event<LookaheadExecution>()).connect(*this);
  m.getChannelFor(event::Event<HeuristicExecution>()).connect(*this);
  m.getChannelFor(event::Event<GraphConstructionFinished>()).connect(*this);
}

void TimingStatistics::handle(const LookaheadExecution& e) {
  if(e.state_ == ActivityState::started) {
    endPropagation_ = readMicroSecCounter();
    propagation_ += endPropagation_ - startPropagation_;

    startLookahead_ = readMicroSecCounter();
  }
  else if(e.state_ == ActivityState::finished) {
    endLookahead_ = readMicroSecCounter();
    lookahead_ += endLookahead_ - startLookahead_;

    startPropagation_ = readMicroSecCounter();
  }
}

void TimingStatistics::handle(const HeuristicExecution& e) {
  if(e.state_ == ActivityState::started) {
    endPropagation_ = readMicroSecCounter();
    propagation_ += endPropagation_ - startPropagation_;

    startHeuristic_ = readMicroSecCounter();
  }
  else if(e.state_ == ActivityState::finished) {
    endHeuristic_ = readMicroSecCounter();
    heuristic_ += endHeuristic_ - startHeuristic_;

    startPropagation_ = readMicroSecCounter();
  }
}

void TimingStatistics::handle(const GraphConstructionFinished&) {
  endGraphConstruction_ = readMicroSecCounter();
  graphConstruction_ = endGraphConstruction_ - startGraphConstruction_;

  startPropagation_ = readMicroSecCounter();
}

void TimingStatistics::print(std::ostream& os) const {
  uint64 prop = propagation_ + readMicroSecCounter() - startPropagation_;
  uint64 exec = readMicroSecCounter() - startExecution_;
  if (exec == 0) {  // unlikely, but just in case ;)
    ++exec;         // prevent devision-by-zero
  }
  float ratePropagation = prop * 100.0f / exec;
  float rateLookahead = lookahead_ * 100.0f / exec;
  float rateHeuristic = heuristic_ * 100.0f / exec;
  float rateGraphConstruction = graphConstruction_ * 100.0f / exec;

  char sRateProp[7]; sprintf(sRateProp, "%.2f", ratePropagation);
  char sRateLA[7];   sprintf(sRateLA, "%.2f", rateLookahead);
  char sRateHeu[7];  sprintf(sRateHeu, "%.2f", rateHeuristic);
  char sRateGC[7];   sprintf(sRateGC, "%.2f", rateGraphConstruction);

  os << "Timings\n";
  os << "  graph construction: " << sRateGC << "%\n";
  os << "  lookahead execution: " << sRateLA << "%\n";
  os << "  node selection: " << sRateHeu << "%\n";
  os << "  model generation: " << sRateProp << "%\n";
}

}
