/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_NEG_ARCS_COUNT_HEURISTIC_H
#define LIBNOMORE_NEG_ARCS_COUNT_HEURISTIC_H

#include <static_heuristic.h>

namespace NS_NOMORE {

//! Heuristic strategy that compares negative preconditions and successors.
/*! 
 *  The contained EvaluatorType generates heuristic values of each uncolored 
 *  body and head node. For a body node the value is composed of the number 
 *  of negative preconditions (the number of 1-predecessors) and the number 
 *  of negative successors (of all heads for a body node). For a head the 
 *  values is composed of the number of negative successors and the number 
 *  of negative preconditions of all its bodies. The count of negative 
 *  successors is heigher rated.
 *
 *  \note This strategy is only useful as static heuristic.
 *  \note The strategy can be used with body and head nodes. */
class NegativeWeightArcsCountStrat {
public:
  //! The value type containing two counters for neg successors and predecessors.
  struct ValueType {
    //! Initializes the value type.
    /*!
     *  \param ns The number of negative successors for a node. 
     *  \param np The number of negative predecessors for a node. */
    ValueType(size_t ns = 0, size_t np = 0)
      : negSuccessors_(ns)
      , negPredecessors_(np) {
    }

    //! The number of negative successors.
    size_t negSuccessors_;

    //! The number of negative predecessors.
    size_t negPredecessors_;
  };

  //! The greater comparator for the value type.
  struct Compare {
    //! Comparator rates the negative successors and predecessors of two nodes.
    /*!
     *  The negative successors are higher rated.
     *
     *  \param v1 The value type of the first node. 
     *  \param v2 The value type of the second node.
     *  \return True if the first node is "better" than the second node. */
    bool operator()(const ValueType& v1, const ValueType& v2) const {
      return v1.negSuccessors_ > v2.negSuccessors_ 
        || (v1.negSuccessors_ == v2.negSuccessors_  && v1.negPredecessors_ > v2.negPredecessors_); 
    }
  };
  
  //! Evalutes the heuristic value of each uncolored node.
  struct Evaluate {
    //! Computes the heuristic value of a node.
    /*!
     *  \param n The node for generating the heuristic value.
     *  \param tf The tightness factor multipying with the heuristic value.
     *  \return The heuristic value of the node n. */
    ValueType operator()(Node* n, double tf) const;
  };
  
  //! The comparator type defined above.
  typedef Compare ComparatorType;

  //! The evaluator of the strategie defined above.
  typedef Evaluate EvaluatorType;

  //! The choice creator type used to create a Choice.
  typedef DefaultChoiceCreator ChoiceCreatorType;
};

//! A static heuristic that uses NegativeWeightArcsCountStrat as strategy.
typedef StaticHeuristic<NegativeWeightArcsCountStrat> NegativeWeightArcsCount;

}
#endif
