/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <operators/body_lookahead.h>
#include <body_node.h>
#include <head_node.h>
#include <graph.h>
#include <constraint.h>
#include <heuristic.h>

namespace NS_NOMORE {


///////////////////////////////////////////////////////////////////////////////
template <>
const char* getName(BodyLookahead*) {
  return "BLA";
}

template <>
Lookahead* createLookaheadOp(Graph& g, BodyLookahead*) {
  return new BodyLookahead(g);
}
///////////////////////////////////////////////////////////////////////////////

  
BodyLookahead::BodyLookahead(Graph& grp) 
  : Lookahead(grp)
  , currentColor_(Color::none)
  , counterPlus_(0)
  , counterMinus_(0)
  , currentNode_(0)
  , heuristic_(0) {  
  grp.checkTightness();
}
  
const char* BodyLookahead::getName() const {
  return NS_NOMORE::getName<BodyLookahead>(0);
}

void BodyLookahead::execute() {
  
  event::ChannelManager &em = getGraph().getEventManager();
  em.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
  em.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);

  NS_NOMORE::lookahead(*this);

  em.getChannelFor(event::Event<HeadNodeColored>()).disconnect(*this);
  em.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
  
  overlayMap_.clear();

}


void BodyLookahead::handle(const HeadNodeColored& e) {
  if(e.node_->oneSuccessorsBegin() != e.node_->oneSuccessorsEnd()) {
    currentColor_ == Color::minus ? ++counterMinus_ : ++counterPlus_;
  }    
}

inline void BodyLookahead::updateOverlayMap(Node& node, char color) {
  std::pair<OverlayMap::iterator, bool> r = overlayMap_.insert(OverlayMap::value_type(node.getId(), color));
  if (!r.second) {
    // node is already in map,
    // thus update color
    r.first->second |= color;
  }
}

void BodyLookahead::handle(const BodyNodeColored& e) {
  if (getConstraint().supersedes(*currentNode_, *e.node_)) {
    if(e.node_->getColor() == Color::plus || e.node_->getColor() == Color::weak_plus) {
      updateOverlayMap(*e.node_, Color::plus | Color::weak_plus);
    }    
    else if(e.node_->getColor() == Color::minus) {
      updateOverlayMap(*e.node_, Color::minus);
    }
  }
}

void BodyLookahead::storeHeuristic() {
  if (heuristic_) {
    double tf = getGraph().getTightnessFactor(currentNode_);
    heuristic_->addHeuristicValue(
      currentNode_, 
      HeuristicValueType((long)(tf * counterPlus_), (long)(tf * counterMinus_))
    );
  }
}


bool BodyLookahead::lookahead(BodyNode* n) {
  counterPlus_ = counterMinus_ = 0;
  const size_t maxColors = 2;
  Color::ColorValue checkColors[maxColors] = { n->preferredChoicePointColor(), n->alternativeChoicePointColor() };
  bool conflict = false;
  bool checkedOne = false;
  OverlayIterator it = overlayMap_.find(n->getId());
  currentNode_ = n;
  size_t i;
  for (i = 0; i < maxColors && !conflict; ++i) {
    if ( !isOverlayed(it, checkColors[i]) ) {
      currentColor_ = checkColors[i];
      checkedOne = true;
      conflict = !checkColor(*currentNode_, currentColor_);
    }
  }
  if (conflict) {
    color(*n, checkColors[ i % maxColors]);
    overlayMap_.clear();
  }
  else if (checkedOne) {
    storeHeuristic();
  }
  return conflict;
}

void BodyLookahead::setHeuristic(Heuristic& h) {
  heuristic_ = dynamic_cast<BodyLAHeuristicBase*>(&h);
}


}
