#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <operators/hybrid_lookahead.h>
#include <heuristics/hybrid_lookahead_heuristic.h>
#include <operators/no_unfounded_check.h>
#include <graph.h>
#include "test_common.h"

using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class HybridLookaheadProp : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE(HybridLookaheadProp);
  CPPUNIT_TEST(testSuccess);
  CPPUNIT_TEST(testDefaultHeuristic);
  CPPUNIT_TEST(testException);
  CPPUNIT_TEST(testFailure1);
  CPPUNIT_TEST(testFailure2);
  CPPUNIT_TEST_SUITE_END();

public:
  void setUp() {
    grp = new Graph();

    body1 = grp->insertBodyNode();
    body2 = grp->insertBodyNode();
    body3 = grp->insertBodyNode();
    head1 = grp->insertHeadNode(1);
    head2 = grp->insertHeadNode(2);

    det = new NoUnfoundedCheck(*grp);    
    lookahead = new HybridLookahead(*grp);
		lookahead->setPropagator(*det);
		lookahead->setHeuristic(heu);
		lookahead->setConstraint(con);
  }

  void tearDown() {
    delete lookahead;
    delete det;
    delete grp;
  }

  void testSuccess() {
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    grp->finishConstruction();
    
		CPPUNIT_ASSERT_EQUAL(true, (*lookahead)());

    CPPUNIT_ASSERT( heu.valueFor(body1, 1) );
		CPPUNIT_ASSERT( heu.valueFor(body3, 1) );
    
		// the following nodes are dependent
		CPPUNIT_ASSERT( heu.noValueFor(body2) );	// overlayed by body1 (propagation)
		CPPUNIT_ASSERT( heu.noValueFor(head1) );	// overlayed by body3 (back-propagation)
		CPPUNIT_ASSERT( heu.noValueFor(head2) );	// overlayed by body3 (back-propagation)
		
		
    
    CPPUNIT_ASSERT_EQUAL( Color::none, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, body3->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::none, head2->getColor() );
  }

  void testDefaultHeuristic() {
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    
    HybridLookaheadHeuristic heu;
    heu.setGraph(*grp);
    lookahead->setHeuristic(heu);
    grp->finishConstruction();
    (*lookahead)();

    CPPUNIT_ASSERT_EQUAL(static_cast<Node*>(body1), heu.selectNode(OnlyBodies()).node());
    // both heads depend on body3
		CPPUNIT_ASSERT_EQUAL(static_cast<Node*>(0), heu.selectNode(OnlyHeads()).node());
  }

  void testException() {
    HeadNode *head3 = grp->insertHeadNode(3);
  
    body1->insertSuccessor(*head1);
    body1->insertSuccessor(*head2);
    head2->insertZeroSuccessor(*body2);
    head2->insertOneSuccessor(*body3);    
    body3->insertSuccessor(*head3);
    head3->insertZeroSuccessor(*body1);
    
    CPPUNIT_ASSERT_EQUAL( false, (*lookahead)() );
    CPPUNIT_ASSERT_EQUAL( true, grp->hasConflict() );

  }

  void testFailure1() {    
    body1->insertSuccessor(*head1);
    head1->insertZeroSuccessor(*body2);
    body2->insertSuccessor(*head2);
    head2->insertOneSuccessor(*body2);
    
    grp->color(*head2, Color::minus);
    grp->color(*body2, Color::minus);
    
    CPPUNIT_ASSERT_EQUAL( true, (*lookahead)() );

    CPPUNIT_ASSERT_EQUAL( Color::minus, body1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, body2->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head1->getColor() );
    CPPUNIT_ASSERT_EQUAL( Color::minus, head2->getColor() );
  }

  void testFailure2() {    
    body1->insertSuccessor(*head1);
    head1->insertOneSuccessor(*body2);
    body2->insertSuccessor(*head2);
    head2->insertOneSuccessor(*body2);
    grp->checkTightness();
    grp->finishConstruction();
    grp->color(*head2, Color::minus);
    grp->color(*body2, Color::minus);
    
    CPPUNIT_ASSERT_EQUAL( true, (*lookahead)() );

    CPPUNIT_ASSERT_EQUAL(head1->getColor(), head1->alternativeChoicePointColor());
  }
  
private:
  struct FakeHeuristic : public HybridLAHeuristicBase {
  public:
    const char* getName() const { return "Fake"; }
    Choice doSelectNode(const Constraint& c) {return Choice();}
    void addHeuristicValue(Node* n, const HeuristicValueType& v) {
      values_.push_back(NVPair(n, v));
    }
    bool valueFor(Node* n, long v) const {
      for (size_t i = 0; i != values_.size(); ++i) {
        if (values_[i].first == n && values_[i].second == v)
          return true;
      }
      return false;
    }
    bool noValueFor(Node* n) const {
      for (size_t i = 0; i != values_.size(); ++i) {
        if (values_[i].first == n)
          return false;
      }
      return true;
    }
    typedef std::pair<Node*, long> NVPair;
    std::vector<NVPair> values_;
  };

  Graph *grp;

  Propagator *det;
  HybridLookahead *lookahead;
  SuppConstraint con;  
  FakeHeuristic heu;
  BodyNode *body1;
  BodyNode *body2;
  BodyNode *body3;
  HeadNode *head1;
  HeadNode *head2;
};

CPPUNIT_TEST_SUITE_REGISTRATION(HybridLookaheadProp);

} // end namespace NS_NOMORE_TESTS
