#include <cppunit/TestFixture.h>
#include <cppunit/TestAssert.h>
#include <cppunit/extensions/HelperMacros.h>
#include <lparsereader.h>
#include <graph.h>
#include <sstream>
#include <head_node.h>
using namespace NS_NOMORE;

namespace NS_NOMORE_TESTS {

class TestLparseReader : public CppUnit::TestFixture {
private:
  typedef std::pair<std::string, long> Atom;
  bool hasAtom(const Graph& g, const Atom& a) const {
    for (Graph::HeadNodeIterator it = g.headNodesBegin(); it != g.headNodesEnd(); ++it)
      if ( (*it)->getAtomId() == a.second && (*it)->getAtomName() == a.first)
        return true;
    return false;
  }
public:
  CPPUNIT_TEST_SUITE(TestLparseReader);
  CPPUNIT_TEST(testRead);
  CPPUNIT_TEST(testWrite);
	CPPUNIT_TEST(testBug);
	CPPUNIT_TEST_SUITE_END();

	void testRead() {
		LparseReader reader;
		Graph g;
		std::stringstream prg;
		
		prg << "1 1 1 1 2\n"	// d:-not c.
			  << "1 2 1 1 3\n"	// c:-not a.
				<< "1 3 1 1 4\n"	// a:-not b.
				<< "1 4 1 1 3\n"	// b:-not a.
				<< "0\n1 d\n2 c\n3 a\n4 b\n"
				<< "0\nB+\n0\nB-\n0\n1\n";
		reader.readProgram(prg, g);
		CPPUNIT_ASSERT_EQUAL(3u, g.countBodyNodes());
		CPPUNIT_ASSERT_EQUAL(4u, g.countHeadNodes());
		
    Atom atoms[4] = {Atom("d", 1l), Atom("c", 2l), Atom("a", 3l), Atom("b", 4l)};
    for (int i = 0; i != 4; ++i) {
      CPPUNIT_ASSERT(hasAtom(g, atoms[i]));
    }
	}

	void testWrite() {
		LparseReader reader;
		Graph g;
		std::stringstream prg;
		std::stringstream res;
		
		prg << "1 1 1 1 2 \n"	// d:-not c.
			  << "1 2 1 1 3 \n"	// c:-not a.
				<< "1 3 1 1 4 \n"	// a:-not b.
				<< "1 4 1 1 3 \n"	// b:-not a.
				<< "0\n1 d\n2 c\n3 a\n4 b\n"
				<< "0\nB+\n0\nB-\n0\n1\n";
		reader.readProgram(prg, g);

    reader.writeProgram(res, g);

    CPPUNIT_ASSERT_EQUAL(prg.str(), res.str());
	}

	void testBug() {
		std::stringstream prg;
		std::stringstream res;
		
		prg << "1 1 1 0 2 \n"			// d :- c
			  << "1 2 2 0 3 1 \n"		// c :- b, d.
				<< "1 3 1 0 4 \n"			// b:-a.
				<< "1 4 1 1 3 \n"			// a:-not b.
				<< "0\n1 d\n2 c\n3 b\n4 a\n"
				<< "0\nB+\n0\nB-\n0\n1\n";
		
		LparseReader reader;
		Graph g;
		reader.readProgram(prg, g);
		reader.writeProgram(res, g);

		std::stringstream exp;
		exp << "1 3 1 0 4 \n"			// b:-a.
				<< "1 4 1 1 3 \n"			// a:-not b.
				<< "0\n3 b\n4 a\n"
				<< "0\nB+\n0\nB-\n0\n1\n";
		CPPUNIT_ASSERT_EQUAL(exp.str(), res.str());

	}
};

CPPUNIT_TEST_SUITE_REGISTRATION(TestLparseReader);

}	// end namespace NS_NOMORE_TESTS
