/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_NODE_H
#define LIBNOMORE_NODE_H

#if defined (_MSC_VER) && _MSC_VER <= 1300
#pragma warning (disable : 4786)
#endif

#include <color.h>
#include <strong_components.h>
#include <functional>
#include <cassert>

namespace event {
	class ChannelManager;
}

namespace NS_NOMORE {

class Graph;

//! Enumeration of node types.
struct NodeType {

  //! Node types.
  enum Value {
    //! Head node representing heads of rules.
    head_node = 1,

    //! Body node representing bodies of rules.
    body_node = 2
  };
};

//! Type of color operations for the color method.
struct ColorOpType {

  //! Color type for a node.
  enum Value {
    //! The node has to be colored as a normal node.
    no_choice,

    //! The node has to be colored as the first choice point.    
    /*!
     *  \note This type is typically used for n-ary backtracking search.
     *        For the binary backtracking search the next_choice is the
     *        same color type. */
    first_choice,

    //! The node has to be colored as the first choice point.    
    /*!
     *  \note This type is typically used for n-ary backtracking search.
     *        For the binary backtracking search the first_choice is the
     *        same color type. */
    next_choice,

    //! The node has to be colored as a normal node.
    /*!
     *  \note This color operation indicates a recoloring of a choice point.*/
    last_choice
  };
};

//! Base class for all nodes to be stored in nomore++'s body head graph.
/*! 
 *  This base node holds common information (like identifier and node color)
 * 	needed by all subclasses, e.g. HeadNode or BodyNode. 
 *
 *  \see Graph */
class Node {
  friend class Graph;

public:
  //! Creates a new node with the given id and node type.
  /* 
   *  \param id The id for the new node.
   *  \param type The type of this node. */
  explicit Node(long id, NodeType::Value type);

  //! Destroys a node.
  virtual ~Node();

  //! Returns the current color value of the node.
  /*!
   *  \return The color value of the node. */
  Color::ColorValue getColor() const {
    return color_;
  }

  //! Returns the unique identifier of the node.
  /*!
   *  \return The idenfier of the node set by the constructor. */
  long getId() const {
    return id_;
  }

  //! Returns the type of this node.
  /*!
   *  \return The type of the node set by the constructor. */
  NodeType::Value getType() const {
    return type_;
  }

  //! Sets the component number of the node.
  /*!
   *  \param nr The number of the stronly connected component. */
  void setComponentNumber(int nr) {
    componentNumber_ = nr;
  }

  //! Returns the component this node is a member of.
  /*!
   *  \note The default is to return 0 meaning that no component number was 
   *        set.
   *  \return The number of the stronly connected component (0 if not set). */
  int getComponentNumber() const {
    return componentNumber_;
  }

  //! Returns true if this node is not strongly connected to any other node.
  /*!
   *  \note If component numbers were not set the function returns false.
   *  \return True if the node is in a trivial component. */ 
  bool trivialComponent() const {
    return componentNumber_ == NS_NOMORE::trivialComponent;
  }

  //! Returns the color which preserves the recursive support.
  /*!
   *  \return The color value preserves the recursive support. */
  virtual Color::ColorValue recursiveSupportColor() const = 0;

  //! Returns the preferred choice point color of this node.
  /*!
   *  \return The preferred color value for the node used for the choice 
   *          point. */
  virtual Color::ColorValue preferredChoicePointColor() const = 0;

  //! Returns the alternative choice point color for this node.
  /*!
   *  \return The alternative color value for the node used for the choice 
   *          point. */
  virtual Color::ColorValue alternativeChoicePointColor() const = 0;

  //! The tightness factor for the node.
  /*!
   *  This factor represents the supported state of the node.
   *
   *  \param gradient The gradient used to compute the factor.
   *  \note The default implementation returns 1.0.
   *  \return The tightness factor. */
  virtual double getTightnessFactor(int gradient) const;

protected:
  //! Colors the node.
  /*! 
   *  Sets the node's color to c and calls doSetColor(oldColor) if the 
   *  transition from the current color to c is valid and c differs from 
   *  the current color. 
   *
   *  \note Refer to \ref NS_NOMORE::isValidColoring "isValidColoring" for 
   *        information on valid colorings.
   *
   *  \param c The new color for the node.
   *
   *  \pre c is not Color::none.
   *
   *  \return 
   *    - ColorOperation::succeeded if a new color was set.
   *    - ColorOperation::unchanged if c and the node's current color are equal.
   *    - ColorOperation::failed    if coloring the node to c would be illegal. */
  ColorOpResult setColor(Color::ColorValue c);

  //! Restores the node's color to c.
  /*! 
   *  Sets the color of the node to c and calls doRestoreColor(oldColor).
   *
   *  \param c The new color of the node.
   *  \pre c is either Color::none or Color::weak_plus. */
  void restoreColor(Color::ColorValue c) {
    Color::ColorValue oldColor = color_;
    color_ = c;
    doRestoreColor(oldColor);
  }

  //! Called by setColor if the color was changed.
  /*! 
   *  Subclasses can implement necessary update operations here.
   *
   *  \param oldColor The old color before coloring this node. */
  virtual void doSetColor(Color::ColorValue oldColor) = 0;

  //! Called by restoreColor.
	/*! 
   *  Subclasses can implement necessary update operations here.
   *  \param oldColor The old color before restoring this node. */
  virtual void doRestoreColor(Color::ColorValue oldColor) = 0;

  //! Should fire a node-specific color event.
  /*! 
   *  \param c The channel manager for firing the events.
   *  \param oldColor The old color of the node before coloring.
   *
   *  \ref _event "More information on events." */
  virtual void fireColorEvent(event::ChannelManager& c, Color::ColorValue oldColor) = 0;

  //! Should fire a node-specific color choice point event.
  /*! 
   *  \param c The channel manager for firing the events.
   *  \param oldColor The old color of the node before coloring.
   *  \param v The choice point color operation.
   *
   *  \ref _event "More information on events." */
  virtual void fireColorChoicePointEvent(event::ChannelManager& c, Color::ColorValue oldColor, ColorOpType::Value v) = 0;

private:
  // deny constructors
  Node(const Node& n);
  Node& operator=(const Node& n);

  // instance variables  
  long id_; 
  NodeType::Value type_;  
  Color::ColorValue color_; // Initialized to Color::none.
  int componentNumber_; 
};

//! Id based comparison of two nodes.
inline bool operator==(const Node& lhs, const Node& rhs) {
  return lhs.getId() == rhs.getId();
}

//! Checks if a transition from oldColor to newColor is valid.
/*!
 *  Valid colorings are:
 *  - none -> X
 *  - weak_plus	-> plus
 *  .
 *  Where X is any color value except Color::none. */
inline bool isValidColoring(Color::ColorValue oldColor, Color::ColorValue newColor) {
	assert(newColor != Color::none);
  return oldColor == Color::none || (oldColor == Color::weak_plus && newColor == Color::plus);
}

//! Struct for checking the color of a node against a color mask.
struct HasColor : public std::unary_function<Node*, bool>{
  //! Initializes the instance by the color mask.
	/*!
   *  \param colorMask An ORed enumeration of color values.
   *  \see Color::ColorValue */
  explicit HasColor(int colorMask)
		: colorMask_(colorMask) {
	}

  //! Returns true if n has a color that was specified in the color mask.
  /*!
   *  \param n The node tested for the specified colors. 
   *  \return True if the node has a color specified by the color mask else 
   *          false. */
	bool operator()(const Node* n) const {
		return (n->getColor() & colorMask_) != 0;
	}

private:
  // instance variables
	int colorMask_;
};

//! Struct for creating a hash for a node and for comparing two nodes.
struct HashHelper {
  //! Comparing two nodes using its hash value.
  /*!
   *  \param n1 The first node for the comparison.
   *  \param n2 The second node for the comparison.
   *  \return True if both hash values are equal else false. */
  bool operator()(const Node* n1, const Node* n2) const {
    return n1->getId() == n2->getId();
  }

  //! Returns the hash value of a node.
  /*!
   *  \param n1 The node.
   *  \return The hash values (as long) for the node n1. */
  long operator()(const Node* n1) const {
    return n1->getId();
  }
};

//! Comparison of two nodes by id.
struct EqualId {
  //! Initializes the object and sets the first node.
  /*!
   *  \param n The first node for the comparison. */
  EqualId(const Node* n) : id_(n->getId()) {}

  //! Compares two node ids.
  /*!
   *  \param n The second node for comparison. 
   *  \return True if both node ids are equal. */
  bool operator()(const Node* n) const {
    return n->getId() == id_;
  }

private:
  // instance variables
  long id_;
};
  
} // NS_NOMORE

#endif
