/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_OBJECT_FACTORY_H
#define LIBNOMORE_OBJECT_FACTORY_H

#include <string>
#include <stdexcept>
#include <map>
#include <utility>
#include <vector>
#include <memory>
#include <propagator.h>
namespace NS_NOMORE {

class Graph;

class Propagator;
class ChoiceOperator;
class Lookahead;

class Heuristic;
class Constraint;

//! Factory that maps logical operator (and heuristic) names to their C++ classes.
/*!
 *  This Factory can be used to create operators and heuristic by only 
 *  specifying their logical name. */
class ObjectFactory {
public:
  //! Typedef of the propagator create function.
  typedef Propagator* (*PropCreateFunction)(Graph& );

  //! Typedef of the lookahead create function.
  typedef Lookahead* (*LookOpCreateFunction)(Graph& );

  //! Typedef of the choice point operator create function.
  typedef ChoiceOperator* (*ChoiceOpCreateFunction)(Graph&, Heuristic*);

  //! Typedef of the heuristic create function.
  typedef Heuristic* (*HeuristicCreateFunction)();

  //! Class holding information necessary for registering objects in the object factory.
  template <class CF>
  struct RegInfo {
    //! Typedef for the creator.
    typedef CF Creator;

    //! Initializes the information object.
    RegInfo() : name(""), description(""), creator(0){}

    //! Initializes the information object by parameters.
    /*!
     *  \param _name The unique name of the class to be registered.
     *  \param cf A creator for objects of type specified by _name.
     *  \param _description An optional description of the type */
    RegInfo(const std::string& _name, Creator cf, const char* _description = "")
      : name(_name)
      , description(_description)
      , creator(cf) {
    }

    //! The name of the operator (or heuristic).
    std::string name;

    //! Short description of the operator (or heuristic).
    std::string description;

    //! The creator used to create the operator (or heuristic).
    Creator     creator;
  };
  
  //! Typedef of the information object for propagators.
  typedef RegInfo<PropCreateFunction> PropInfo;

  //! Typedef of the information object for lookaheads.
  typedef RegInfo<LookOpCreateFunction> LookOpInfo;

  //! Typedef of the information object for choice point operators.
  typedef RegInfo<ChoiceOpCreateFunction> ChoiceOpInfo;

  //! Typedef of the information object for heuristics.
  typedef RegInfo<HeuristicCreateFunction> HeuristicInfo;
  
  //! Creates a new factory object.
  ObjectFactory();
  
  //! Destroys the factory.
  ~ObjectFactory();
  
  //! Object types this factory can create.
  enum ObjectType {
    PROPAGATOR    = 1,
    LOOKAHEAD_OP  = 2,
    CHOICE_OP     = 4,
    HEURISTIC     = 8,
    ALL_OPERATORS = PROPAGATOR | LOOKAHEAD_OP | CHOICE_OP,
    ALL_OBJECTS   = ALL_OPERATORS | HEURISTIC
  };
  
  //! Type used as result type for the factory's createOperators operation.
  struct Operators {
    //! Initialize the operators object.
    Operators();

    //! Destroys the operators object.
    ~Operators();
    
    //! Initialize the operators object by other operators.
    /*!
     *  Copy constructor.
     *
     *  \param other Another object which member variables are copied to this 
     *               object. */
    Operators(const Operators& other);
      
    //! The deterministic operators (propagator).
    mutable std::auto_ptr<Propagator>     detOp_;

    //! The lookahead operator.
    mutable std::auto_ptr<Lookahead>      lookOp_;

    //! The non deterministic operator.
    mutable std::auto_ptr<ChoiceOperator> choiceOp_;
  };

  //! Registers all known operators and heuristics in this factory object.
  void registerKnownClasses();
  
  //! Registers the operator described by info in the factory.
  /*!
   *  \param info The class informations of a propagator.
   *  \return True if registration was successful. False if name already exists. */
  bool registerClass(const PropInfo& info);

  //! Registers the operator described by info in the factory.
  /*!
   *  \param info The class informations of a lookahead.
   *  \return True if registration was successful. False if name already exists. */
  bool registerClass(const LookOpInfo& info);
  
  //! Registers the operator described by info in the factory.
  /*!
   *  \param info The class informations of a choice operator.
   *  \return True if registration was successful. False if name already exists. */
  bool registerClass(const ChoiceOpInfo& info);
  
  //! Registers the operator described by info in the factory.
  /*!
   *  \param info The class informations of a heuristics.
   *  \return True if registration was successful. False if name already exists. */
  bool registerClass(const HeuristicInfo& info);

  //! Creates a propagator of type specified by name.
  /*!
   *  \param g The graph used to create the propagator.
   *  \param name The name of the propagator to create.
   *  \return The propagator if found.
   *
   *  \throw UnknownObject if name is not a known operator name. */
  std::auto_ptr<Propagator> createPropagator(Graph& g, const std::string& name) const;

  //! Creates a lookahead of type specified by name.
  /*!
   *  \param g The graph used to create the lookahead.
   *  \param name The name of the lookahead to create.
   *  \return The lookahead operator if found.
   *
   *  \throw UnknownObject if name is not a known operator name. */
  std::auto_ptr<Lookahead> createLookaheadOperator(Graph& g, const std::string& name) const;

  //! Creates a choice operator of type specified by name.
  /*!
   *  \param g The graph used to create the choice operator.
   *  \param name The name of the choice operator to create.
   *  \param heu The heuristic used to create the choice operator.
   *  \return The choice operator if found.
   *
   *  \throw UnknownObject if name is not a known operator name */
  std::auto_ptr<ChoiceOperator> createChoiceOperator(Graph& g, const std::string& name, const std::string& heu) const;
  
  //! Creates a heuristics object of type specified by name.
  /*!
   *  \param g The graph used to create the heuristics object.
   *  \param name The name of the heuristics object to create.
   *  \return The heuristics object if found.
   *
   *  \throw UnknownObject if name is not a known heuristic name */
  std::auto_ptr<Heuristic> createHeuristic(Graph& g, const std::string& name) const;

  //! Creates the operators described in the given strings to be used with the given Graph g.
  /*!
   *  \param g The graph the operators should be applied on.
   *  \param propName Propagator to create.
   *  \param lookName Lookahead operator to create.
   *  \param lookPropMode Propagation mode to be used in lookahead.
   *  \param choiceName Choice operator to create.
   *  \param heuName Heuristic to create.
   *  \return An object of type Operators that contains the requested objects.
   *
   *  \note propName and choiceName are required.
	 *  \note All other parts may be set to "None" if not needed.
   *
   *  \throw UnknownObject if one of the strings contain an unknown name. */
  Operators createObjects(Graph& g, const std::string& propName, const std::string& lookName, Propagator::Mode lookPropMode, const std::string& choiceName, const std::string& heuName) const;
  
  //! Returns the names of all registered classes of type ot.
  /*!
   *  \param ot The type of objects to return.
   *  \return A list of objects of type ot.
   *
   *  \see ObjectType*/
  std::vector<std::string> getRegisteredClasses(ObjectType ot) const;

  //! Returns the description for the type specified by name.
  /*!
   *  \param name The name of the object getting its description.
   *  \return The description of the object. */
  std::string getDescription(const std::string& name) const;

private:
  // instance methods
	void setLookahead(Graph& g, Operators& ops, const std::string& name, Propagator::Mode m) const;
  ObjectFactory(const ObjectFactory&);
  ObjectFactory& operator=(const ObjectFactory&);
  
  // typedefs
  typedef std::map<std::string, std::pair<PropCreateFunction, std::string> > PropMap;
  typedef std::map<std::string, std::pair<LookOpCreateFunction, std::string> > LookOpMap;
  typedef std::map<std::string, std::pair<ChoiceOpCreateFunction, std::string> > ChoiceOpMap;
  typedef std::map<std::string, std::pair<HeuristicCreateFunction, std::string> > HeuristicsMap;

  // instance variables
  bool          owner_;
  PropMap       detOps_;
  LookOpMap     lookaheadOps_;
  ChoiceOpMap   choiceOps_;
  HeuristicsMap heuristics_;
};

//! Returns the name of class T.
/*!
 *  Specialize this function for each class T to be registered in the object 
 *  factory.
 *  \attention The parameter is needed to workaround a bug in VC 6.0 but 
 *             should otherwise be ignored.
 *
 *  \code
 *    // in the cpp-file of the class to be registered
 *    template <>
 *    const char* getName(YourClass*) {
 *      return "YourClass" // or whatever name YourClass should have
 *    }
 *  \endcode */
template <class T>
const char* getName(T*);

//! Creation function for lookahead operators.
/*!
 *  \param g The graph used to create the object.
 *  \param T The type of the object.
 *  \see createDetOp */
template <class T>
Lookahead* createLookaheadOp(Graph& g, T*);

//! Creation function for choice operators.
/*!
 *  \param g The graph used to create the object.
 *  \param h The heuristics used to create the object.
 *  \param T The type of the object.
 *  \see createDetOp */
template <class T>
ChoiceOperator* createChoiceOp(Graph& g, Heuristic* h, T*);

//! Creation function for heuristics.
/*!
 *  \param T The type of the object.
 *  \see createDetOp */
template <class T>
Heuristic* createHeuristic(T*);
  
//! Exception type thrown if the factory is requested to create an unknown object.
class UnknownObject : public std::runtime_error {
public:
  //! Initializes the exception by parameters.
  /*!
   *  \param name The name of the exception.
   *  \param t Type of the object which is not found in the factory.
   *
   *  \see ObjectFactory::ObjectType */
  UnknownObject(const std::string& name, ObjectFactory::ObjectType t)
    : std::runtime_error(name)
    , ot_(t) {
  }

  //! Returns the type of the unknown object.
  /*!
   *  \return The type of the object that is not found.
   *
   *  \see ObjectFactory::ObjectType */
  ObjectFactory::ObjectType getObjectType() const {
    return ot_;
  }

private:
  // instance variables
  ObjectFactory::ObjectType ot_;
};

//! Exception type thrown when an operator string contains an invalid operator combination.
class InvalidOperatorCombination : public std::runtime_error {
public:
  //! Initializes the exception.
  InvalidOperatorCombination() 
    : std::runtime_error("invalid operator combination") {
  }
};

}

#endif
