/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_UNFOUNDED_LOOPS_H
#define NOMORE_UNFOUNDED_LOOPS_H

#include <operators/default_local_propagator.h>
#include <color.h>
#include <head_node.h>
#include <deque>
#include "../src/detail/functors.h"
#include <set>
#include <map>

namespace NS_NOMORE {

class HeadNode;
class BodyNode;
struct BodyNodeColored;
struct BodyNodeColoredChoicePoint;
struct HeadNodeColoredChoicePoint;
struct GraphConstructionFinished;

//! The implementation of a local propagation and an loops finder operator.
/*! 
 *  This propagator uses the default local propagation implemented in the 
 *  class DefaultLocalPropagator. This operation is extended by the unfounded 
 *  set operator by Martin Gebser which finds loops which are unfounded. 
 *
 *  <b>Trigger:</b> The unfounded set algorithm operates on head node of 
 *                  minus colored body predecessors.
 *  <b>Init:</b> 
 *    - <i>Set</i>:      set to one head node h of a minus colored body 
 *                       predecessor which has no external bodies -- if set 
 *                       is after computation filled the node containing to a 
 *                       unfounded set.
 *    - <i>External</i>: set to body predecessors of h which are uncolored
 *    - <i>Sink</i>:     empty -- set of probably justifying head nodes
 *    - <i>Source</i>:   empty -- set of probably justifying body nodes
 *    - <i>scc</i>:      component number of h -- scc(h)
 *  
 *  <b>Algorithm:</b>
 *    \todo Algorithmus anpassen!!!!!!!
 *    \code
 *    while (External not empty) {
 *      b = External.pop();
 *      if (there is a in 0-preds of b such that scc(a) = scc and color(a) != Color::plus and a not in Sink) {
 *        J.set(body preds b of a s.t. color(b) != Color::minus and (scc(b) != scc or b in Source))
 *        if {J is empty) {
 *          Set.add(a)
 *          External.remove(0-succs of head node a)
 *          External.add(preds b of a where color(b) != Color::minus and 0-preds of b contains no head nodes from Set)
 *        } else {
 *          Sink.add(a)
 *          External.add(b)
 *        }
 *      } else {
 *        Source.add(b)
 *        R.set(0-preds of b which part of the set Set)
 *        while (R not empty) {
 *          Set.remove(R)
 *          Sink.add(R)
 *          J.set(all 0-succs bodies of head nodes of R which have no more 0-preds useful for unfounded sets)
 *          Source.push(J)
 *          R.set(0-preds of body nodes in J which part of the set Set)
 *        }
 *        External.set(body nodes which are not (probably) justified by some atom in the set Set)
 *      }
 *    }
 *    \endcode 
 *  The resulting set Set can be colored with Color::minus. */
class UnfoundedLoops : public DefaultLocalPropagator {
public:
  //! Initializes the propagator.
  /*! 
   *  \param grp The graph this propagator should work on. */ 
  explicit UnfoundedLoops(Graph& grp);

	//! Propagates using the local propagation and the improved unfounded lopps operator.
	/*!
   *  \return The value true is returned if the propagator colored one node. 
   *          If no node can be colored the method returns false.
	 *  \note If the graph does not contain any non-trivial strong components 
   *        (i.e. the propgram is tight) propagateFull is equivalent to 
   *        propagateLocal since tight programs can never contain loops which 
   *        are unfounded. */
  virtual bool propagateFull();

	//! Resets all members of the propagator.
  virtual void reset();

  //! Handles the GraphConstructionFinished event.
  /*! 
   *  Checks for tightness and enables or disables the unfounded set search.
   *
   *  \param e The raised event. */
  void handle(const GraphConstructionFinished& e);
  
  //! Handles a BodyNodeColored event fired.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColored& e);
  
  //! Handles a BodyNodeColoredChoicePoint event.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColoredChoicePoint& e);

  //! Returns the name of the operator.
  /*!
   *  \return Returns the name of the propagator. */
  const char* getName() const;
  
  //! Returns the state of the unfounded set finder.
  /*!
   *  \return Returns true if the propagator has disabled itself because the 
   *          graph does not contain any non-trivial strong components. */
  bool isDisabled() const;

private:
  // typedefs
  typedef std::deque<HeadNode*> TodoList;
  typedef std::vector<HeadNode*> Stack;

  typedef std::set<HeadNode*, DETAIL::LessDerefId> WellFoundedHeadList;
  typedef std::multimap<BodyNode*, HeadNode*, DETAIL::LessDerefId> WellFoundedBodyList;

  // instance methods
  void updateTodoList(BodyNode*);
  void updateWFList(BodyNode*);

  bool checkLoop(HeadNode* h);

  BodyNode* getWellfounded(HeadNode* head, int masterComponent) const;
  bool loopIsUnfounded(const Stack& loop) const;

  void color(const Stack& loop);

  HeadNode::BodyNodeIterator pickBody(HeadNode* head, HeadNode::BodyNodeIterator start, const Stack& dfsStack, const Stack& loop) const;
  HeadNode* pickHead(BodyNode* body, int component) const;

  void addToWFList(HeadNode* h, BodyNode* b);
  inline bool isInWFList(HeadNode* h) const;

  // instance variables
  bool disabled_;  
  TodoList todo_;
  WellFoundedHeadList wellFoundedHeads_;
  WellFoundedBodyList wellFoundedBodies_;
};

}

#endif
