/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_READER_H
#define LIBNOMORE_READER_H

#include <iosfwd>
#include <stdexcept>

namespace NS_NOMORE {

class Graph;

//! Interface for classes that transform logic programs to body head graphs and vice versa.
/*!
 *  This class represents two abstract methods which are used to transform the 
 *  program into a body head graph: readProgram() and writeProgram(). 
 *  Overwrite this methods for reading and writing different logic program 
 *  formats. */
class Reader {
public:
  //! Constructs the reader.
  Reader();

  //! Destroys the reader.
  virtual ~Reader();

  //! Transforms a logic program into a corresponding nomore++ body head graph. 
  /*! 
   *  \param input The input stream from which the logic program should be read.
   *  \param[out] graph The graph corresponding to the logic program.
   * 
   *  \pre Input is in good state. It is readable.
   *  \pre It is save to add new Nodes to the graph, i.e. the graph is not yet 
   *       finalized.
   *
   *  \return Returns its second parameter, i.e. the graph object.
   *
   *  \throws ReadError if an error occurs while reading from input. */
  virtual Graph& readProgram(std::istream& input, Graph& graph) = 0;

	//! Transforms the given body head dependency graph back to a logic program.
	/*! 
   *  \param os The output stream the logic program should be written to.
   *  \param g The graph to be transformed. 
   * 
   *  \throws WriteError if an error occurs while writing to os. */
  virtual void writeProgram(std::ostream& os, const Graph& g) = 0;
};

//! Exception type thrown to signal errors while reading a logic program.
class ReadError : public std::runtime_error {
public:
  //! Initializes the exception using a description string.
  /*!
   *  \param desc The description of the exception. */
  ReadError(const std::string& desc)
    : std::runtime_error(desc) {
  }
};

//! Exception type thrown to signal errors while writing a logic program.
class WriteError : public std::runtime_error {
public:
  //! Initializes the exception using a description string.
  /*!
   *  \param desc The description of the exception. */
  WriteError(const std::string& desc)
    : std::runtime_error(desc) {
  }
};

} // NS_NOMORE

#endif
