/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         *
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         *
 ***************************************************************************/
#include <heuristics/neighborhood.h>
#include <graph.h>

#include <limits>
#include <algorithm>
#include <operators/lookahead.h>

namespace NS_NOMORE {

///////////////////////////////////////////////////////////////////////////////
template <>
const char* getName(Neighborhood*) {
  return "Neighborhood";
}

template <>
Heuristic* createHeuristic(Neighborhood*) {
  return new Neighborhood();
}
///////////////////////////////////////////////////////////////////////////////
Neighborhood::Neighborhood(int m)
  : max_(m)
  , currentValue_(0)
  , lastStep_(false)
  , bestNode_(0, ValueType())
  , propagator_(0) {
  assert(m > 0);
}

Neighborhood::~Neighborhood() {
}

void Neighborhood::doSetGraph() {
  getGraph().checkTightness();
}

void Neighborhood::setMaxDepth(int m) {
  if (m > 0) {
    max_ = m;
  }
}

void Neighborhood::setPropagator(Propagator& op) {
  propagator_ = &op;
}

void Neighborhood::registerEventHandlers() {
  event::ChannelManager& m = getGraph().getEventManager();
  m.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);

}

void Neighborhood::unregisterEventHandlers() {
  event::ChannelManager& m = getGraph().getEventManager();
  m.getChannelFor(event::Event<HeadNodeColored>()).disconnect(*this);
	m.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
}

Choice Neighborhood::doSelectNode(const Constraint& c) {
  overlayList_.resize(getGraph().countNodes());
  bestNode_.first = 0;
  bestNode_.second = ValueType();
  currentValue_ = 0;

  registerEventHandlers();

  Graph::UncoloredNodeIterator end = getGraph().uncoloredNodesEnd();
  Constraint::Type type = c.getType();
  for (Graph::UncoloredNodeIterator it = getGraph().uncoloredNodesBegin(); it != end; ++it) {
    if (c.isSatisfiedBy(**it) && !propagate(*it, type)) {
      break; // select the node that causes a conflict
    }
  }

  unregisterEventHandlers();
  overlayList_.clear();

  return getBestChoice(type);
}

Choice Neighborhood::getBestChoice(Constraint::Type t) const {
  if (!bestNode_.first)
    return Choice();

  if (t == Constraint::uniform_constraint && bestNode_.second.second > bestNode_.second.first) {
    return Choice(bestNode_.first
      , bestNode_.first->alternativeChoicePointColor()
      , bestNode_.first->preferredChoicePointColor()
    );
  }
  return Choice(bestNode_.first);
}

bool Neighborhood::propagate(Node* n, Constraint::Type t) {
  assert(0 != n);

  // color node as choice point -> a new stack is used
  Color::ColorValue colors[2] = {n->preferredChoicePointColor(), n->alternativeChoicePointColor()};
  int counter[2] = {0, 0};
  int maxColors = t == Constraint::uniform_constraint ? 2 : 1;
  bool conflict = false;
  for (int i = 0; i < maxColors && !conflict; ++i) {
    if (overlayList_.isOverlayed(*n, colors[i]))
      continue;

    currentValue_ = 0;
    bool cpColored = getGraph().colorChoicePoint(*n, colors[i], Color::none, ColorOpType::first_choice);
    if (!cpColored || !propagator_->propagateLocal(max_)) {
      counter[i] = std::numeric_limits<int>::max();
      conflict = true;
    }
    else {
      counter[i] = currentValue_;
    }
    if (cpColored)
      getGraph().restoreChoicePoint();
  }

  evaluate(n, ValueType(counter[0], counter[1]));
  return !conflict;
}

void Neighborhood::handle(const HeadNodeColored& e) {
  ++currentValue_;
  if (lastStep_ && e.node_->getColor() == Color::minus) {
    ++currentValue_;
  }
  overlayList_.add(*e.node_, e.node_->getColor());
}

void Neighborhood::handle(const BodyNodeColored& e) {
  ++currentValue_;
  if (HasColor(Color::plus | Color::weak_plus)(e.node_)) {
    overlayList_.add(*e.node_, Color::plus);
    overlayList_.add(*e.node_, Color::weak_plus);
    if (lastStep_)
      ++currentValue_;
  }
  else
    overlayList_.add(*e.node_, e.node_->getColor());
}

void Neighborhood::evaluate(Node* n, ValueType vt) {
  assert(0 != n);

  double tf = getGraph().getTightnessFactor(n);
  vt.first = (int)(vt.first * tf);
  vt.second = (int)(vt.second * tf);
  if (bestNode_.first == 0 || vt.first + vt.second > bestNode_.second.first + bestNode_.second.second) {
    bestNode_.first = n;
    bestNode_.second = vt;
  }
}

const char* Neighborhood::getName() const {
  return NS_NOMORE::getName<Neighborhood>(0);
}

}
