/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#include <solver.h>
#include <printer.h>
#include <graph.h>
#include <operator.h>
#include <choice_operator.h>
#include <operators/lookahead.h>
#include <stdexcept>
namespace NS_NOMORE {

Solver::Solver() 
  : as_(0)
  , det_(0)
  , lookahead_(0)
  , choice_(0)
  , printer_(0) {  
}  

Solver::Solver(int as, Propagator *det, Lookahead* look, ChoiceOperator *choice, Printer *printer) 
  : as_(as)
  , det_(det)
  , lookahead_(look)
  , choice_(choice)
  , printer_(printer) {
}

Solver::~Solver() {  
}

void Solver::setAnswerSetPrinter(Printer* printer) {
  assert(printer);
  printer_ = std::auto_ptr<Printer>(printer);
}

void Solver::setChoiceOperator(ChoiceOperator* choice) {
  assert(choice);
  choice_ = std::auto_ptr<ChoiceOperator>(choice);
}

ChoiceOperator* Solver::getChoiceOperator() const {
  return choice_.get();
}


void Solver::setNrOfAnswerSetsToCompute(int as) {
  as_ = as;
}

int Solver::getNrOfAnswerSetsToCompute() const {
  return as_;
}


void Solver::setPropagator(Propagator* det) {
  assert(det);
  det_ = std::auto_ptr<Propagator>(det);  
}

void Solver::setLookahead(Lookahead* look) {
  lookahead_ = std::auto_ptr<Lookahead>(look);
}

const Propagator* Solver::getPropagator() const {
  return det_.get();
}

const Lookahead* Solver::getLookahead() const {
  return lookahead_.get();
}

bool Solver::solve(Graph& grp) {
  grp.finishConstruction();

  if (as_ == 0)
    as_ = -1;
  
  bool restoreOK = true;
  for(int asNr = 0; asNr != as_ && restoreOK; ) {
    if (propagate() == false) {
      det_->reset();
      if (lookahead_.get()) lookahead_->reset();
      restoreOK = backtrack(grp);
    }
    else if (grp.totalColoring()) {
      (*printer_)(++asNr, grp);
      restoreOK = backtrack(grp);
    }
    else if (!choose(grp)) { 
      if (!grp.hasConflict()) { // no choices left
        if (!grp.totalize()) {
          restoreOK = backtrack(grp);
        }
        else{
          assert(grp.totalColoring());
        }
      }
      else { // illegal choice
        restoreOK = backtrack(grp);
      }
    }
  }
  return restoreOK;
}

bool Solver::propagate() {
  bool ret = false;
  if ( det_->propagateFull()) {
    if (lookahead_.get()) {
      lookahead_->reset();
      if ((*lookahead_)()) {
        det_->reset();  // necessary because det_ may contain ops that
                        // are not part of lookahead_ and vice versa.
        ret = true;
      }
    }
    else {
      ret = true;
    }
  }
  return ret;
}

int Solver::getNrOfFoundAS() const {
  return printer_->getNrOfFoundAS();
}


}
