//Copyright (C) 2010 Simon Kiertscher
//
//This program is free software; you can redistribute it and/or modify it 
//under the terms of the GNU General Public License as published by the 
//Free Software Foundation; either version 3 of the License, or (at your option) 
//any later version.
//
//This program is distributed in the hope that it will be useful, but WITHOUT 
//ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS 
//FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License along with 
//this program; if not, see <http://www.gnu.org/licenses/>.

#include <Python.h>
#include <frameobject.h>
#include "cherub_api.h"
#include <netdb.h>
#include <assert.h>

#include <unistd.h>
#include <syslog.h>

#include <string.h>
#include <time.h>
#include <malloc.h>
#include <stdlib.h>

#include <frameobject.h>

#define ERROR_TIMESTAMP_DELTA 600

/* some colors to make the output better visible*/
#define ANSI_COLOR_RED     "\x1b[31m"
#define ANSI_COLOR_GREEN   "\x1b[32m"
#define ANSI_COLOR_YELLOW  "\x1b[33m"
#define ANSI_COLOR_BLUE    "\x1b[34m"
#define ANSI_COLOR_MAGENTA "\x1b[35m"
#define ANSI_COLOR_CYAN    "\x1b[36m"
#define ANSI_COLOR_RESET   "\x1b[0m"

time_t old_error_timestamp=0;

/* cherub_mail sends mail to email address from config file */
int cherub_mail(struct cherub_content *content, char *subject, char *mailcontent)
{
	time_t error_timestamp;
	
	if(content->email==NULL || strcmp(content->email,"") == 0){
		syslog(LOG_DEBUG, "[MASTER] [cherub_mail] No mail send, because of no email address");
		return 0;
	}

	if(strcmp(subject,"CHERUB-ERROR")==0){
		syslog(LOG_INFO, "[MASTER] [cherub_mail] ERROR Mail is sent");
		/* for initialisation simulate, the last error message was send prior 700 seconds */
		if(old_error_timestamp==0){	
			old_error_timestamp=time(NULL);
			old_error_timestamp-=(time_t)(ERROR_TIMESTAMP_DELTA+100);
		}
		
		error_timestamp=time(NULL);
		/* if not older than one hour or has already send this message, dont sent it again */
		if(error_timestamp-old_error_timestamp<ERROR_TIMESTAMP_DELTA){
			/* syslog(LOG_INFO, "[MASTER] [cherub_mail] No mail send, because another error mail was already send in the last %d seconds!",ERROR_TIMESTAMP_DELTA); */
			/* syslog(LOG_INFO, "[MASTER] [cherub_mail] error_timestamp: %d old_error_timestamp: %d",(int)error_timestamp,(int)old_error_timestamp);               */
			return 0;
		}
		else{
			/* syslog(LOG_INFO, "[MASTER] [cherub_mail] NOT: error_timestamp-old_error_timestamp<ERROR_TIMESTAMP_DELTA");                                                                              */
			/* syslog(LOG_INFO, "[MASTER] [cherub_mail] error_timestamp: %d old_error_timestamp: %d < ERROR_TIMESTAMP_DELTA: %d",(int)error_timestamp,(int)old_error_timestamp,ERROR_TIMESTAMP_DELTA); */
			old_error_timestamp=error_timestamp;			
		}
	}

	char command[1024];
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] TRY  - print command in string");
	snprintf(command,1023,"/usr/bin/mail -s '%s' %s", subject, content->email);
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] DONE - print command in string");
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] TRY  - open streamer");
	FILE *mailer = popen(command , "w");
	if(mailer==NULL)
		{syslog(LOG_DEBUG, "[MASTER] [cherub_mail] ERROR- could not open streamer");
		return 1;}
	
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] DONE - open streamer");
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] TRY  - print content");
	fprintf(mailer, "CHERUB %s",mailcontent);
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] DONE - print content");
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] TRY  - close mailer");
	pclose(mailer);
	syslog(LOG_DEBUG, "[MASTER] [cherub_mail] DONE - close mailer");
	syslog(LOG_INFO, "[MASTER] [cherub_mail] SEND MAIL TO %s",content->email);
	return 0;
}

int send_problem_mails(struct cherub_content *content, struct cherub_cluster *cluster)
{
	int n,b=0,shut=0,r=0,sign=0, position_b=0, position_shut=0, position_r=0, position_sign=0;
	char *boot_p=NULL,*shutdown_p=NULL,*register_p=NULL,*signoff_p=NULL,*message=NULL;
	syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] Function Entered");
	
	boot_p     = realloc(boot_p    , strlen("Nodes:") + 1);
	shutdown_p = realloc(shutdown_p, strlen("Nodes:") + 1);
	register_p = realloc(register_p, strlen("Nodes:") + 1);
	signoff_p  = realloc(signoff_p , strlen("Nodes:") + 1);
	memcpy(boot_p    , "Nodes:", sizeof("Nodes"));
	memcpy(shutdown_p, "Nodes:", sizeof("Nodes"));
	memcpy(register_p, "Nodes:", sizeof("Nodes"));
	memcpy(signoff_p , "Nodes:", sizeof("Nodes"));
	position_b   =strlen(boot_p);
	position_shut=strlen(shutdown_p);
	position_r   =strlen(register_p);
	position_sign=strlen(signoff_p);
	
	for(n=0;n<cluster->number_of_nodes;n++)
	{
		syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] %s",boot_p);
		if(cluster->nodes[n].boot_problem == 1){
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] node has boot problems");
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] TRY  - realloc to %d",(int)(strlen(boot_p) + strlen(cluster->nodes[n].name) + 2));
			boot_p = realloc(boot_p, (strlen(boot_p) + strlen(cluster->nodes[n].name) + 2));
			if(boot_p == NULL) {
				syslog(LOG_ERR, "[MASTER] [send_problem_mails] COULD NOT REALLOCATE MEM");
				exit(1);
			}
			memcpy(boot_p+position_b, " ",strlen(" "));
			memcpy(boot_p+position_b+1, cluster->nodes[n].name, strlen(cluster->nodes[n].name) + 1);
			position_b = strlen(boot_p);
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] strlen boot_p: %d",(int)strlen(boot_p));
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] processing node %d",n);
			b=1;
		}	
		if(cluster->nodes[n].shutdown_problem == 1){
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] node has shutdown problems");
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] TRY  - realloc to %d",(int)(strlen(shutdown_p) + strlen(cluster->nodes[n].name) + 2));
			shutdown_p = realloc(shutdown_p, (strlen(shutdown_p) + strlen(cluster->nodes[n].name) + 2));
			if(shutdown_p == NULL) {
				syslog(LOG_ERR, "[MASTER] [send_problem_mails] COULD NOT REALLOCATE MEM");
				exit(1);
			}
			memcpy(shutdown_p+position_shut, " ",strlen(" "));
			memcpy(shutdown_p+position_shut+1, cluster->nodes[n].name, strlen(cluster->nodes[n].name) + 1);
			position_shut = strlen(shutdown_p);
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] strlen shutdown_p: %d",(int)strlen(shutdown_p));
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] processing node %d",n);
			shut=1;
		}
		if(cluster->nodes[n].register_problem == 1){
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] node has register problems");
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] TRY  - realloc to %d",(int)(strlen(register_p) + strlen(cluster->nodes[n].name) + 2));
			register_p = realloc(register_p, (strlen(register_p) + strlen(cluster->nodes[n].name) + 2));
			if(register_p == NULL) {
				syslog(LOG_ERR, "[MASTER] [send_problem_mails] COULD NOT REALLOCATE MEM");
				exit(1);
			}
			memcpy(register_p+position_r, " ",strlen(" "));
			memcpy(register_p+position_r+1, cluster->nodes[n].name, strlen(cluster->nodes[n].name) + 1);
			position_r = strlen(register_p);
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] strlen register_pp: %d",(int)strlen(register_p));
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] processing node %d",n);
			r=1;
		}
		if(cluster->nodes[n].signoff_problem == 1){
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] node has signoff problems");
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] TRY  - realloc to %d",(int)(strlen(signoff_p) + strlen(cluster->nodes[n].name) + 2));
			signoff_p = realloc(signoff_p, (strlen(signoff_p) + strlen(cluster->nodes[n].name) + 2));
			if(signoff_p == NULL) {
				syslog(LOG_ERR, "[MASTER] [send_problem_mails] COULD NOT REALLOCATE MEM");
				exit(1);
			}
			memcpy(signoff_p+position_sign, " ",strlen(" "));
			memcpy(signoff_p+position_sign+1, cluster->nodes[n].name, strlen(cluster->nodes[n].name) + 1);
			position_sign = strlen(signoff_p);
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] strlen signoff_p: %d",(int)strlen(signoff_p));
			syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] processing node %d",n);
			sign=1;
		}
	}
	message = realloc(message, strlen("Status:\n") + 1);
	memcpy(message, "Status:\n", strlen("Status:\n") + 1);
	if(b==1){
		message = realloc(message, strlen(message) + strlen("\nNodes with Bootproblems:\n") + 1);
		memcpy(message + strlen(message), "\nNodes with Bootproblems:\n", strlen("\nNodes with Bootproblems:\n") + 1);
		message = realloc(message, strlen(message) + strlen(boot_p) + 1);
		memcpy(message + strlen(message), boot_p, strlen(boot_p) + 1);
	}
	if(shut==1){
		message = realloc(message, strlen(message) + strlen("\nNodes with Shutdownproblems:\n") + 1);
		memcpy(message + strlen(message), "\nNodes with Shutdownproblems:\n", strlen("\nNodes with Shutdownproblems:\n") + 1);
		message = realloc(message, strlen(message) + strlen(shutdown_p) + 1);
		memcpy(message + strlen(message), shutdown_p, strlen(shutdown_p) + 1);
	}
	if(r==1){
		message = realloc(message, strlen(message) + strlen("\nNodes with Registerproblems:\n") + 1);
		memcpy(message + strlen(message), "\nNodes with Registerproblems:\n", strlen("\nNodes with Registerproblems:\n") + 1);
		message = realloc(message, strlen(message) + strlen(register_p) + 1);
		memcpy(message + strlen(message), register_p, strlen(register_p) + 1);
	}
	if(sign==1){
		message = realloc(message, strlen(message) + strlen("\nNodes with Signoffproblems:\n") + 1);
		memcpy(message + strlen(message), "\nNodes with Signoffproblems:\n", strlen("\nNodes with Signoffproblems:\n") + 1);
		message = realloc(message, strlen(message) + strlen(signoff_p) + 1);
		memcpy(message + strlen(message), signoff_p, strlen(signoff_p) + 1);
	}
	if(sizeof(message)>0)
	{
		cherub_mail(content, "CHERUB ERROR REPORT", message);
	}
	syslog(LOG_DEBUG, "[MASTER] [send_problem_mails] Leaving Function");
	free(boot_p);
	free(shutdown_p);
	free(register_p);
	free(signoff_p);
	free(message);
	return 0;
}


//parameter is the node_ipmi_ip_address to shutdown.
int cherub_shutdown(const char *node_ipmi_ip_address, struct cherub_content *content)
{
	assert(node_ipmi_ip_address != NULL);
	PyObject *funk, *prm, *ret;

	if(!(PyObject_HasAttrString(content->modul, "cherub_shutdown"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_shutdown was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}

	funk = PyObject_GetAttrString(content->modul, "cherub_shutdown");
	if(funk == NULL)
	{
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] CANT LOAD CHERUB_SHUTDOWN FROM %s.PY",content->scriptname);
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: SHUTDOWN can't load script (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	prm = Py_BuildValue("(s)", node_ipmi_ip_address);
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: SHUTDOWN can't build value (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	ret = PyObject_CallObject(funk, prm); 
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: SHUTDOWN can't call function (error in function) (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	if(PyInt_AsLong(ret)!=0)
	{
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] ERROR ON PyInt_AsLong (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);

	return EXIT_SUCCESS;
}
	
//parameter is the node_ipmi_ip_address to boot.
int cherub_boot(const char *node_ipmi_ip_address, struct cherub_content *content)
{
	assert(node_ipmi_ip_address != NULL);
	PyObject *funk, *prm, *ret;
	int returncode=0;

	if(!(PyObject_HasAttrString(content->modul, "cherub_boot"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_boot was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_boot");
	if(funk == NULL)
	{
		syslog(LOG_ERR, "[MASTER] [cherub_boot] CANT LOAD CHERUB_BOOT FROM %s.PY",content->scriptname);
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: BOOT can't load module (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	prm = Py_BuildValue("(s)", node_ipmi_ip_address); 
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_boot] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: BOOT can't build value (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	ret = PyObject_CallObject(funk, prm); 
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_boot] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: BOOT can't call function (error in function) (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	returncode = PyInt_AsLong(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);
	Py_CLEAR(ret);

	switch(returncode)
	{
		case	0: return EXIT_SUCCESS;
		case	2: return 2;
		case	3: return 3;
		default	 : syslog(LOG_ERR, "[MASTER] [cherub_boot] ERROR ON PyInt_AsLong (please have a look into the logfile)");
			   return(EXIT_FAILURE);
	}
} 
	
//parameter is the node_name to sign off from the RMS.
int cherub_sign_off(const char *node_name, struct cherub_content *content)
{
	assert(node_name != NULL);
	PyObject *funk, *prm, *ret;

	if(!(PyObject_HasAttrString(content->modul, "cherub_sign_off"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_sign_off was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_sign_off");
	if(funk == NULL)
	{
		syslog(LOG_ERR, "[MASTER] [cherub_sign_off] CANT LOAD CHERUB_SIGN_OFF FROM %s.PY",content->scriptname);
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: SIGNOFF can't load module (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	prm = Py_BuildValue("(s)", node_name);
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_sign_off] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: SIGNOFF can't build value (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	ret = PyObject_CallObject(funk, prm); 
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_sign_off] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: SIGNOFF can't call function (error in function) (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	if(PyInt_AsLong(ret)!=0)
	{
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_sign_off] ERROR ON PyInt_AsLong");
		return(EXIT_FAILURE);
	}
	
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);

	return EXIT_SUCCESS;
}

//parameter is the node_name to register him in the RMS.
int cherub_register(const char *node_name, struct cherub_content *content)
{
	assert(node_name != NULL);
	PyObject *funk, *prm, *ret;

	if(!(PyObject_HasAttrString(content->modul, "cherub_register"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_register was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_register");
	if(funk == NULL)
	{
		syslog(LOG_ERR, "[MASTER] [cherub_register] CANT LOAD CHERUB_REGISTER FROM %s.PY",content->scriptname);
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: REGISTER can't load module (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	prm = Py_BuildValue("(s)", node_name);
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_register] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: REGISTER can't build value (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	ret = PyObject_CallObject(funk, prm); 
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_register] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: REGISTER can't call function (error in function) (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	if(PyInt_AsLong(ret)!=0)
	{
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_register] ERROR ON PyInt_AsLong (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);

	return EXIT_SUCCESS;
} 

//parameter is the node_name by which the status should be reviewed.
int cherub_status(const char *node_name, struct cherub_content *content)
{
	assert(node_name != NULL);
	syslog(LOG_DEBUG, "[MASTER] [cherub_status] TRY  - get the new state");
	enum cherub_status error = CHERUB_UNKNOWN;
	syslog(LOG_DEBUG, "[MASTER] [cherub_status] make new PyObjects");
	PyObject *funk, *prm, *ret;
	
	syslog(LOG_DEBUG, "[MASTER] [cherub_status] get function with PyObject_GetAttrString");
	if(!(PyObject_HasAttrString(content->modul, "cherub_status"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_status was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_status");
	if(funk == NULL)
	{
		syslog(LOG_ERR, "[MASTER] [cherub_status] CANT LOAD CHERUB_STATUS FROM %s.PY",content->scriptname);
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: STATUS can't load module (please have a look into the logfile)");
		return(-1);
	}
	
	prm = Py_BuildValue("(s)", node_name);
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_status] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: STATUS can't build value (please have a look into the logfile)");
		return(-1);
	}
	
	ret = PyObject_CallObject(funk, prm); 
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_status] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: STATUS can't call function (error in function) (please have a look into the logfile)");
		return(-1);
	}
	error = PyInt_AsLong(ret);
	if(error < -2 || error > 3)
	{
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_status] ERROR ON PyInt_AsLong (please have a look into the logfile)");
		return(-1);
	}
	if(error == -2)
	{
		Py_CLEAR(ret);
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_status] ERROR: Could not connect to communicator.");
		return(-2);
	}
	
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);
	
	syslog(LOG_DEBUG, "[MASTER] [cherub_status] DONE - get the new state");
	return error; //-1 = CHERUB_UNKNWON // 0 = CHERUB_BUSY // 1 = CHERUB_ONLINE // 2 = CHERUB_OFFLINE // 3 = CHERUB_DOWN // 
} 
	
int* cherub_status_parallel(int *size, struct cherub_content *content) {
	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] entered cherub_status_parallel method");
	PyObject *funk, *prm, *ret;
	int i;
	int* list;

	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] TRY  - PyObject_GetAttrString");
	if(!(PyObject_HasAttrString(content->modul, "cherub_status_parallel"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_status_parallel was not found in the script, make sure every function of the Cherub API is implemented");
		return(NULL);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_status_parallel");
	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] DONE - PyObject_GetAttrString");
	if(funk == NULL)
	{
		syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] CANT LOAD CHERUB_STATUS_PARALLEL FROM %s.PY",content->scriptname);
		return NULL;
	}

	prm = Py_BuildValue("()");
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_status_parallel] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: CHERUB_STATUS_PARALLEL can't build value (please have a look into the logfile)");
		return NULL;
	}

	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] TRY  - PyObject_CallObject");
	ret = PyObject_CallObject(funk, prm);
	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] DONE - PyObject_CallObject");
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_status_parallel] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: CHERUB_STATUS_PARALLEL can't call function (error in function) (please have a look into the logfile)");
		return NULL;
	}

	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] TRY  - Convert PyList");
	*size = PyList_Size(ret);
	list = calloc(*size, sizeof(int));

	for (i=0; i<*size; i++) {
		list[i] = (int) PyInt_AsLong(PyList_GetItem(ret, i));
	}
	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] DONE  - Convert PyList");

	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] TRY  - Py_CLEAR");
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);
	syslog(LOG_DEBUG, "[MASTER] [cherub_status_parallel] DONE - Py_CLEAR");
	return list; //-1 = CHERUB_UNKNWON // 0 = CHERUB_BUSY // 1 = CHERUB_ONLINE // 2 = CHERUB_OFFLINE // 3 = CHERUB_DOWN //
}


//returns if this single node has load
int cherub_node_load(const char *node_name, struct cherub_content *content)
{
	assert(node_name != NULL);
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] entered cherub_node_load method");
	PyObject *funk, *prm, *ret;
	long error = -1;
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] TRY  - PyObject_GetAttrString");
	if(!(PyObject_HasAttrString(content->modul, "cherub_node_load"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_node_load was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_node_load");
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] DONE - PyObject_GetAttrString");
	if(funk == NULL)
	{
		syslog(LOG_ERR, "[MASTER] [cherub_node_load] CANT LOAD CHERUB_NODE_LOAD FROM %s.PY",content->scriptname);
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: NODE_LOAD can't load module (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] TRY  - Py_BuildValue");
	/*printf("-----------------------------------\n");*/
	prm = Py_BuildValue("(s)", node_name);
	/*if(PyString_Check(prm))
	{
		printf("--------is STRING\n");
	}else
	{
		printf("--------is KEIN!!! STRING\n");
	}


	printf("####### %s ########## %s ########\n",PyString_AsString(prm),node_name);*/
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] DONE - Py_BuildValue");
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_node_load] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: NODE_LOAD can't build value (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] TRY  - PyObject_CallObject");
	ret = PyObject_CallObject(funk, prm); 
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] DONE - PyObject_CallObject");
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_node_load] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: NODE_LOAD can't call function (error in function) (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] TRY  - PyInt_AsLong");
	error = PyInt_AsLong(ret);
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] DONE - PyInt_AsLong with error: %ld",error);
	if(error < 0 || error > 1)
	{
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_node_load] ERROR ON PyInt_AsLong, PyInt_AsLong(ret)=%ld",error);
		return(EXIT_FAILURE);
	}
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] TRY  - Py_CLEAR");
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);
	syslog(LOG_DEBUG, "[MASTER] [cherub_node_load] DONE - Py_CLEAR");
	//	 0 = the given node has no load
	//	 1 = the given node has load (to indicate that he must be started)	
	return error;
}

//returns which nodes have load
int* cherub_load_list(int *size, struct cherub_content *content)
{
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] entered cherub_load_list method");
	PyObject *funk, *prm, *ret;
	int i;
	int* list;

	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] TRY  - PyObject_GetAttrString");
	if(!(PyObject_HasAttrString(content->modul, "cherub_load_list"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_load_list was not found in the script, make sure every function of the Cherub API is implemented");
		return(NULL);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_load_list");
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] DONE - PyObject_GetAttrString");
	if(funk == NULL)
	{
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] CANT LOAD CHERUB_NODES_LOAD FROM %s.PY",content->scriptname);
		Py_CLEAR(funk);
		return NULL;
	}

	prm = Py_BuildValue("()");
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_load_list] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: NODES_LOAD can't build value (please have a look into the logfile)");
		return NULL;
	}

	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] TRY  - PyObject_CallObject");
	ret = PyObject_CallObject(funk, prm);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] DONE - PyObject_CallObject");
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_load_list] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: NODES_LOAD can't call function (error in function) (please have a look into the logfile)");
		return NULL;
	}

	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] TRY  - Convert PyList");
	*size = PyList_Size(ret);
	list = calloc(*size, sizeof(int));

	for (i=0; i<*size; i++) {
		list[i] = (int) PyInt_AsLong(PyList_GetItem(ret, i));
		printf("returning list: %d",list[i]);
	}
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] DONE  - Convert PyList");

	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] TRY  - Py_CLEAR");
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_list] DONE - Py_CLEAR");
	//	 0 = the given node has no load
	//	 1 = the given node has load (to indicate that he must be started)
	return list;
}

//returns the number of nodes to boot (explicit specified are not counted)
int cherub_global_load(struct cherub_content *content)
{	
	syslog(LOG_DEBUG, "[MASTER] [cherub_status] TRY  - get the new state");
	PyObject *funk, *prm, *ret;
	long error = -1;

	if(!(PyObject_HasAttrString(content->modul, "cherub_global_load"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_global_load was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_global_load");
	if(funk == NULL)
	{
		syslog(LOG_ERR, "[MASTER] [cherub_global_load] CANT LOAD CHERUB_GLOBAL_LOAD FROM %s.PY",content->scriptname);
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: GLOABAL_LOAD can't load module (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	prm = Py_BuildValue("()");
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_global_load] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: GLOBAL_LOAD can't build value (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	ret = PyObject_CallObject(funk, prm); 
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_global_load] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: GLOBAL_LOAD can't call function (error in function) (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	error = PyInt_AsLong(ret);
	if(error < -1 && PyErr_Occurred()!=NULL)
	{
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_global_load] ERROR ON PyInt_AsLong");
		return(EXIT_FAILURE);
	}
	
	Py_CLEAR(ret);
	Py_CLEAR(prm);
	Py_CLEAR(funk);
	
	syslog(LOG_DEBUG, "[MASTER] [cherub_global_load] DONE - get the new state");
	//	-1 = if an error occured
	//	0-n = for the number of nodes needet to run all jobs
	return error;
} 

int adjust_update_cycle(int current_cycle, struct cherub_content *content)
{
	PyObject *funk, *prm, *ret;

	if(!(PyObject_HasAttrString(content->modul, "cherub_adjust_update_cycle"))){
		syslog(LOG_ERR, "[MASTER] [cherub_shutdown] Critical Error: cherub_adjust_update_cycle was not found in the script, make sure every function of the Cherub API is implemented");
		return(EXIT_FAILURE);
	}
	funk = PyObject_GetAttrString(content->modul, "cherub_adjust_update_cycle");
	if(funk == NULL)
	{
		syslog(LOG_INFO, "[MASTER] [cherub_update_cycle] CANT LOAD CHERUB_ADJUST_UPDATE_CYCLE FROM %s.PY",content->scriptname);
		syslog(LOG_INFO, "[MASTER] [cherub_update_cycle] MAYBE IT'S NOT IMPLEMENTED!");
		return(current_cycle);
	}
	
	prm = Py_BuildValue("(s)", current_cycle);
	if(prm == NULL)
	{
		Py_CLEAR(funk);
		syslog(LOG_ERR, "[MASTER] [cherub_update_cycle] CANT BUILD VALUE");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: cherub_update_cycle can't build value (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	ret = PyObject_CallObject(funk, prm); 
	if(ret == NULL)
	{
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_update_cycle] CANT CALL FUNCTION OBJECT");
		cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: CHERUB_ADJUST_UPDATE_CYCLE can't call function (error in function) (please have a look into the logfile)");
		return(EXIT_FAILURE);
	}
	
	if(PyInt_AsLong(ret) <= -1)
	{
		Py_CLEAR(prm);
		Py_CLEAR(funk);
		Py_CLEAR(ret);
		syslog(LOG_ERR, "[MASTER] [cherub_update_cycle] ERROR ON PyInt_AsLong");
		return(EXIT_FAILURE);
	}
	
	return PyInt_AsLong(ret);
}

///////////////////////////////////////////////
//////////////// INIT /////////////////////////
///////////////////////////////////////////////
/* this function is executet AFTER the cherub_load_config function! It reads the cluster configuration  */
/* from the /etc/cherub_config.py and builds up the cluster datastructure which is used by cherub	*/
int cherub_init(struct cherub_cluster *cluster, struct cherub_content *content)
{
	assert(cluster !=NULL);
	syslog(LOG_DEBUG, "[MASTER] [cherub_init] TRY  - start of cherub_init");
	int n = 0, phase = 0, functiontest = 0;
	PyObject *ret, *cluster_node, *name, *ip_address, *power_phase, *always_on;
	if(content->config_modul!=NULL)
	{
		syslog(LOG_DEBUG, "[MASTER] [cherub_init] TRY  - get Object");
		if(!PyObject_HasAttrString(content->config_modul, "cluster")){
			syslog(LOG_ERR, "[MASTER] [cherub_init]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" Error: no cluster defined, please fix the configuration file!");
			return(EXIT_FAILURE);
		}
		ret = PyObject_GetAttrString(content->config_modul, "cluster");
		if(ret == NULL)			{goto ERROR_OCCURED;}
		if(PyList_Check(ret) != 1)	{goto ERROR_OCCURED;}
		syslog(LOG_DEBUG, "[MASTER] [cherub_init] DONE - get Object");	
		cluster->number_of_nodes = PyList_Size(ret);
		cluster->running_nodes = cluster->number_of_nodes;
		cluster->busy_nodes = 0;
		cluster->online_nodes = 0;
		cluster->offline_nodes = 0;
		cluster->down_nodes = 0;
		cluster->unknown_nodes = 0;
		
		/* check if parallel versions of the functions are implemented */
		
		functiontest = PyObject_HasAttrString(content->modul, "cherub_status_parallel");
		if(functiontest!=1){
			content->parallel_status=0;
			syslog(LOG_INFO, "[MASTER] [cherub_init]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" No parallel status determination implemented");	
			
		}
		else{
			content->parallel_status=1;
			syslog(LOG_INFO, "[MASTER] [cherub_init]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" Parallel status determination function was found");	
		}
		functiontest = PyObject_HasAttrString(content->modul, "cherub_load_list");
		if(functiontest!=1)
		{
			content->parallel_load=0;
			syslog(LOG_INFO, "[MASTER] [cherub_init]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" No parallel load determination implemented");	
		}
		else
		{
			content->parallel_load=1;
			syslog(LOG_INFO, "[MASTER] [cherub_init]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" Parallel load determination function was found");	
		}
		//subscripe every node in the array cluster.nodes[]
		for(n = 0; n < cluster->number_of_nodes; n++)
		{
			cluster_node = PyList_GetItem(ret, n);
			
			if(PyTuple_Check(cluster_node) != 1) 
				goto ERROR_OCCURED;
			
			//here you get only borrowed references, you must not free them with Py_CLEAR
			name 		= PyTuple_GetItem(cluster_node, 0);	
			ip_address	= PyTuple_GetItem(cluster_node, 1);
			power_phase = PyTuple_GetItem(cluster_node, 2);
			always_on	= PyTuple_GetItem(cluster_node, 3);
			
			if(PyString_Check(name) != 1 || PyString_Check(ip_address) != 1 || PyInt_Check(power_phase) != 1 || PyInt_Check(always_on) != 1)
				{goto ERROR_OCCURED;}
			
			cluster->nodes[n].name = strdup(PyString_AS_STRING(name));
			if (cluster->nodes[n].name == NULL)
			    {goto ERROR_OCCURED;}
				
			cluster->nodes[n].ip_address = strdup(PyString_AS_STRING(ip_address));
			if (cluster->nodes[n].ip_address == NULL) 
			    {goto ERROR_OCCURED;}
			
			if (strlen(cluster->nodes[n].name) > NI_MAXHOST || strlen(cluster->nodes[n].ip_address) > NI_MAXHOST)
				{goto ERROR_OCCURED;}
				
			phase 	= (int)PyLong_AsLong(power_phase);
			if (phase < 1 || phase > 3)
				{goto ERROR_OCCURED;}

			cluster->nodes[n].always_on = (int)PyLong_AsLong(always_on);
			if(cluster->nodes[n].always_on < 0 || cluster->nodes[n].always_on > 1)
				{goto ERROR_OCCURED;}
			//initialize the rest of the attributs
			cluster->nodes[n].power_phase = (int)phase;
			cluster->nodes[n].cycles_running = 0;
			cluster->nodes[n].cycles_under_load = 0;
			cluster->nodes[n].cycles_error = 0;
			cluster->nodes[n].boot_problem = 0;
			cluster->nodes[n].shutdown_problem = 0;
			cluster->nodes[n].signoff_problem = 0;
			cluster->nodes[n].emergency_routin = 0;
			cluster->nodes[n].register_problem = 0;
			cluster->nodes[n].state = CHERUB_UNKNOWN;
			cluster->nodes[n].last_state = CHERUB_UNKNOWN;
			cluster->nodes[n].load = 0;
			if(content->options & COMMANDLINE_INFO)
			{
				printf("\n##### INIT VALUES OF NODE NUMBER %i #####\n"\
					   "# NODE NAME:   %15s         #\n"\
					   "# NODE IP:     %15s         #\n"\
					   "# POWER PHASE: %15i         #\n"\
					   "# ALWAYS ON:   %15i         #\n"\
					   "# STATUS:      %15i         #\n"
					   "# RUNNING CYCLES: %12.0f         #\n"\
					   "########################################\n\n"\
					,n,cluster->nodes[n].name,cluster->nodes[n].ip_address,cluster->nodes[n].power_phase,cluster->nodes[n].always_on,cluster->nodes[n].state, cluster->nodes[n].cycles_running
					);
			}
		}//for each node
		syslog(LOG_INFO, "[MASTER] [cherub_init] following nodes are registered");
		if (cluster->number_of_nodes<1){
			syslog(LOG_INFO, ANSI_COLOR_RED"NONE"ANSI_COLOR_RESET);
		}
		for(n = 0; n < cluster->number_of_nodes; n++){
			syslog(LOG_INFO, "[Name: "ANSI_COLOR_MAGENTA"%10s"ANSI_COLOR_RESET"][IPMI:  "ANSI_COLOR_MAGENTA"%16s"ANSI_COLOR_RESET"][Status: "ANSI_COLOR_MAGENTA"%i"ANSI_COLOR_RESET"][PowerPhase: "ANSI_COLOR_MAGENTA"%i"ANSI_COLOR_RESET"][Always on: "ANSI_COLOR_MAGENTA"%i"ANSI_COLOR_RESET"]",cluster->nodes[n].name, cluster->nodes[n].ip_address, cluster->nodes[n].state, cluster->nodes[n].power_phase, cluster->nodes[n].always_on);
		}//for each node
		syslog(LOG_DEBUG, "[MASTER] [cherub_init] TRY  - Py_CLEAR");
		Py_CLEAR(ret);
		//Py_CLEAR(cluster_node); //DO NOT DELETE THIS REFERENCE!!!
		syslog(LOG_DEBUG, "[MASTER] [cherub_init] DONE - Py_CLEAR");

    } 
    else 
	{
		syslog(LOG_ERR, "[MASTER] [cherub_init] ERROR: Modul cherub_config could not be found - this error should not occure");
		return -1;
	}
	syslog(LOG_DEBUG, "[MASTER] [cherub_init] DONE - start of cherub_init");
	return 0;
	
	ERROR_OCCURED:
	{
		syslog(LOG_ERR, "[MASTER] [cherub_init] ERROR: Something wrong in the cherub_init");
		syslog(LOG_DEBUG, "[MASTER] [cherub_init] TRY  - Py_CLEAR");
		Py_CLEAR(ret);
		syslog(LOG_DEBUG, "[MASTER] [cherub_init] DONE - Py_CLEAR");
		syslog(LOG_ERR, "[MASTER] [cherub_init] HINT: the return value of the config file is apparently  not a valid python list");
		return -1;
	}
}

/* this function is executet BEFOR the cherub_init function! It initialize python and loads the */
/* configuration file /etc/cherub_config.py							*/
int cherub_load_config(struct cherub_content *content)
{
	assert(content != NULL);

	/* parsing configfile arg */ 
	char* config_path=NULL;
	char* config_name=NULL;
	if(content->options & COMMANDLINE_CONFIG){
		char* config_name_raw=NULL;
	        char* config_path_raw=NULL;
		int file_len=0, path_len=0,config_len=0;
		config_len = strlen(content->config);
		file_len = strlen(strrchr(content->config,'/')) - 1;
		/* DO NOT FORGET THE \0 TERMINATION IN THE END! ALLOCADE SPACE FOR IT*/
		config_name_raw = (char*) calloc(file_len + 1,sizeof(char));
		strncpy(config_name_raw, strrchr(content->config,'/') + sizeof(char), file_len + 1);
		path_len = config_len - file_len;
	
		config_path_raw = (char*) calloc(path_len + 1, sizeof(char));
		strncpy(config_path_raw, content->config, path_len);
		
		if(strrchr(config_name_raw,'.')!=NULL)
		{
			if(strncmp(strrchr(config_name_raw,'.'),".py",3)==0){
				config_name=(char*) calloc(file_len + 1, sizeof(char));
				strncpy(config_name, config_name_raw, file_len - 3);
			}
			else{
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] "ANSI_COLOR_RED"[X]"ANSI_COLOR_RESET" your config file must end either with suffix .py or must have no suffix");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
		}
		else{
			config_name=(char*) calloc(file_len + 1, sizeof(char));
			strncpy(config_name, config_name_raw, config_len);
		}
		config_path = config_path_raw;
	}	
	else{
		config_path = strdup("/etc/");
		config_name = strdup("cherub_config");
	}

	/* initiate python */
	int result;
	if(!Py_IsInitialized()){	
		/*ON START UP*/
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - Py_Initialize");
		result = setenv("PYTHONPATH", config_path, 1);
		if (result == -1) 
		{
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" failed to set PYTHONPATH");
			exit(EXIT_FAILURE);
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" set PYTHONPATH to %s",config_path);
		}
		Py_Initialize();
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE  - Py_Initialize");
	}
	else{	/* ELSEBRANCH NOT USED IN CURRENT VERSION */
		/*ON RELOAD*/
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" ELSE try reinitialization"ANSI_COLOR_RESET);
		Py_CLEAR(content->modul);
		Py_CLEAR(content->config_modul);
		Py_Finalize();
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] Py_Finalize done");
		result = setenv("PYTHONPATH", config_path, 1);
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] set PYTHONPATH to /etc/");
		if (result == -1) 
		{
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" failed to set PYTHONPATH");
			exit(EXIT_FAILURE);
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" set PYTHONPATH to %s",config_path);
		}
		Py_Initialize();
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] Py_Initialize done");
		free(content->scriptname);
		free(content->loglvl);
	}
	
	/* PYTHON IS INITIALIST */
	/* used with New references */
	PyObject *email=NULL, *pythonpath=NULL, *log_facility=NULL, *loglvl=NULL, *scriptname=NULL, *sequential_shutdown=NULL, *e_cycles=NULL, *boot_cycles=NULL, *shutdown_cycles=NULL, *register_cycles=NULL, *sign_off_cycles=NULL, *u_cycles=NULL, *d_update=NULL, *python_version=NULL, *sys_module=NULL, *backup=NULL;
	
	/* used with Borrowed references */
	PyObject *PyError;
	
	/* Check Python Version */
	sys_module = PyImport_ImportModule("sys");	//new Reference
	if(sys_module==NULL){
		syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" PYTHON_VERSION: could not load sys module");
		syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
		exit(EXIT_FAILURE);
	}
	if(PyObject_HasAttrString(sys_module, "version")){
		python_version = PyObject_GetAttrString(sys_module, "version");
		if(python_version == NULL){
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" python_version: could not get value (PyObject_GetAttrString failed)");
			syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
			exit(EXIT_FAILURE);
		}
		else{
			/* check for version >= 2.7 */
			char* python_version_string = PyString_AS_STRING(python_version);
			if(python_version_string[0] >= '2'){
				if(python_version_string[2] < '7' ) printf("OLD VERSION OF PYTHON: %c\n",python_version_string[2]);
				else syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" PYTHON_VERSION: %s",python_version_string);
			}
			else{
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" PYTHON_VERSION: %s",python_version_string);
				syslog(LOG_INFO, "[MASTER] [cherub_load_config] Used python version not supported");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
		}
	}
	else{
		syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" PYTHON_VERSION: could not get python version");
		syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
		exit(EXIT_FAILURE);
	}
	Py_CLEAR(sys_module);
	Py_CLEAR(python_version);
	
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - to load %s%s",config_path,config_name);
	
	/* give the config path and name as argument to the module, so it can also load the same config */
	/* like the main C compontents*/
	char * argv[2];
	argv[0] = config_name;
	argv[1] = config_path;
	PySys_SetArgv(2, argv);

	content->config_modul = PyImport_ImportModule(config_name);
	if(content->config_modul != NULL){
		/* Print the loaded config file to the user, he now can check if it was the right one */
		PyObject* tmp_config_name = PyImport_AddModule(config_name);	//Borrowed reference
		PyObject* globalDictionary = PyModule_GetDict(tmp_config_name);	//Borrowed reference
		const char* pythonScript_getfile = "result = __file__\n";
		PyRun_String(pythonScript_getfile, Py_file_input, globalDictionary, globalDictionary);
		const char* file_path = PyString_AsString(PyDict_GetItemString(globalDictionary, "result"));
		syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" CONFIG_FILE_LOADED: %s",file_path);
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - to load %s%s",config_path,config_name);
		
		//  loglvl
		//default is debug (is set in cherub.c (search for "setlogmask")
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load log_level");
		if(PyObject_HasAttrString(content->config_modul, "log_level")){
			loglvl = PyObject_GetAttrString(content->config_modul, "log_level");
			if(loglvl == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" log_level: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyString_Check(loglvl))
				{
					content->loglvl = strdup(PyString_AS_STRING(loglvl));
					/*XXX - TEST PURPOSE: content->loglvl = strdup("DEBUG");*/
					/*XXX - set loglevel hart to debug */
					//content->loglvl = strdup("DEBUG");
	
					if (content->loglvl == NULL){
						syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" log_level: strdup failed");
						syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
						exit(EXIT_FAILURE);
					}
					else{
						syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" log_level: loglevel entry set to %s",content->loglvl);
					}
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" log_level: must be a  string, loading default (DEBUG)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" log_level: not set, loading default (INFO)");
		}		
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load log_level");
		
		/* NOW set a the log lvl */
		if(strcmp(content->loglvl,"DEBUG") == 0)
		{
			setlogmask(LOG_UPTO(LOG_DEBUG));
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" loglvl: set upto DEBUG");
		}
	
		if(strcmp(content->loglvl,"INFO")== 0)
		{
			setlogmask(LOG_UPTO(LOG_INFO));
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" loglvl: set upto INFO");
		}
	
		if(strcmp(content->loglvl,"WARNING")== 0)
		{
			setlogmask(LOG_UPTO(LOG_WARNING));
			syslog(LOG_WARNING, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" loglvl: set upto WARNING");
		}
	
		if(strcmp(content->loglvl,"ERROR")== 0)
		{
			setlogmask(LOG_UPTO(LOG_ERR));
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" loglvl: set upto ERROR");
		}
		
		/* get email address for sending error messages */
                syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - to get email address");
                if(PyObject_HasAttrString(content->config_modul, "email")){
	                email = PyObject_GetAttrString(content->config_modul, "email");
			if(email == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" E-Mail: could not get address, this should not happen!");
			}
			else{
				if(PyString_Check(email) == 1){
		                        content->email = strdup(PyString_AS_STRING(email));
					if(content->email != NULL){
						syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" E-Mail: address set to: %s",content->email);
					}
					else{
						syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" E-Mail: strdup failed, no address set, e-mails function deactivated");
					}
	        	        }
		                else{
		                        syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" E-Mail: E-Mail must be a string, no address is set, e-mails function deactivated");
		                        content->email = strdup("");
		                }
			}
		}
                else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" E-Mail: no address set, e-mails function deactivated");
			content->email = strdup("");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - to get email address");
		
		/* check if mail program is available */
		const char *mail_filename = "/usr/bin/mail";
		if(access(mail_filename, F_OK)==0){
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" E-Mail: /usr/bin/mail found");
		}else
		{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" E-Mail: /usr/bin/mail not found, e-mails function deactivated");
		}
		
		//  log_facility
		//set the log facility, default DAEMON
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - to set log_facility");
		if(PyObject_HasAttrString(content->config_modul, "log_facility")){
			log_facility = PyObject_GetAttrString(content->config_modul, "log_facility");
			if(log_facility==NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" log_facility: could not get log_facility, this should not happen!");
			}
			else{
				int new_log_set = 0;
				if(PyString_Check(log_facility)){
					char *lf = strdup(PyString_AS_STRING(log_facility));
					if(!strncmp(lf,"LOCAL0",strlen("LOCAL0"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL0); new_log_set=1;}
					if(!strncmp(lf,"LOCAL1",strlen("LOCAL1"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL1); new_log_set=1;}
					if(!strncmp(lf,"LOCAL2",strlen("LOCAL2"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL2); new_log_set=1;}
					if(!strncmp(lf,"LOCAL3",strlen("LOCAL3"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL3); new_log_set=1;}
					if(!strncmp(lf,"LOCAL4",strlen("LOCAL4"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL4); new_log_set=1;}
					if(!strncmp(lf,"LOCAL5",strlen("LOCAL5"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL5); new_log_set=1;}
					if(!strncmp(lf,"LOCAL6",strlen("LOCAL6"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL6); new_log_set=1;}
					if(!strncmp(lf,"LOCAL7",strlen("LOCAL7"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_LOCAL7); new_log_set=1;}
					if(!strncmp(lf,"DAEMON",strlen("DAEMON"))) {openlog("cherub", LOG_PID | LOG_NDELAY | LOG_PERROR | LOG_CONS, LOG_DAEMON); new_log_set=1;}
					if(!new_log_set){
						syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" logfacility: %s is not supported! Please change to LOCAL0-LOCAL7 or DAEMON",lf);
					}else{
						syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" logfacility: set to %s",lf); 
					}
					free(lf);
				}
				else{
					syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" log_facility: must be a string, DEFAULT LOG FACILITY IS DAEMON (check /etc/cherub_config.py to change)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" log_facility: not set, DEFAULT LOG FACILITY IS DAEMON (check /etc/cherub_config.py to change)");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - to set log_facility");

		//  expand the pythonpath
		//there is no default
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - to get pythonpath");
		if(PyObject_HasAttrString(content->config_modul, "pythonpath")){
			pythonpath = PyObject_GetAttrString(content->config_modul, "pythonpath");
			if(pythonpath == NULL)
			{
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" pythonpath: could not get path, this should not occure!");
				cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: PYTHONPATH NOT SET/FOUND (please have a look into the logfile)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyString_Check(pythonpath)){
					syslog(LOG_DEBUG, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" pythonpath: set to %s",PyString_AsString(pythonpath));
				}
				else{
					syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" pythonpath: must be a string, check /etc/cherub_config.py");
					syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
					exit(EXIT_FAILURE);
				}
			}
		}
		else{
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" pythonpath: not set, please check /etc/cherub_config.py and set the pythonpath to your used python site-packages (e.g. /usr/lib64/python2.7/site-packages)");
			cherub_mail(content, "CHERUB-ERROR", "ERROR WAS LOGGED: PYTHONPATH NOT SET/FOUND (please have a look into the logfile)");
			syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
			exit(EXIT_FAILURE);
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - to get pythonpath");

		//  SET the pythonpath
		//no default here
		if(PyString_Check(pythonpath))
		{	
			syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - to SET pythonpath");
			char *ppath, *p = strdup(PyString_AS_STRING(pythonpath));
			if(p == NULL){
                                syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" pythonpath: strdup failed (could not allocade)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
                        }
			else{
				ppath = calloc(1,strlen(p)+strlen(":/etc/")+1);
				strncat(ppath, p, strlen(p));
				strncat(ppath, ":/etc/",6);
				int result = setenv("PYTHONPATH", ppath, 1);
				if (result == -1){
					syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" pythonpath: setenv failed");
					syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
					exit(EXIT_FAILURE);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" pythonpath: set to %s",ppath);
					free(p);
					free(ppath);
					syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - to SET pythonpath");
				}
			}
		}	
		else{
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" pythonpath: must be a string, check /etc/cherub_config.py (should not occure, due to earlier tests)");
			syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
			exit(EXIT_FAILURE);
		}

		//  WAIT_FOR_SIGN_OFF
		//default is 10 minutes
		content->WAIT_FOR_SIGN_OFF = 10;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load wait_for_sign_off");
		if(PyObject_HasAttrString(content->config_modul, "wait_for_sign_off")){
			sign_off_cycles = PyObject_GetAttrString(content->config_modul, "wait_for_sign_off");
			if(sign_off_cycles == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_sign_off: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(sign_off_cycles)){
					content->WAIT_FOR_SIGN_OFF = (int)PyLong_AsLong(sign_off_cycles);
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" wait_for_sign_off: set to %d min",content->WAIT_FOR_SIGN_OFF);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_sign_off: must be a number, loading default (10 minutes)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_sign_off: not set, loading default (10 minutes), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load wait_for_sign_off");
		
		//  WAIT_FOR_REGISTER
		//default is 0 minutes
		content->WAIT_FOR_REGISTER = 0;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load wait_for_register");
		if(PyObject_HasAttrString(content->config_modul, "wait_for_register")){
			register_cycles = PyObject_GetAttrString(content->config_modul, "wait_for_register");
			if(register_cycles == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_register: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(register_cycles) == 1)
				{
					content->WAIT_FOR_REGISTER = (int)PyLong_AsLong(register_cycles);
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" wait_for_register: set to %d min",content->WAIT_FOR_REGISTER);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_register: must be a number, loading default (0 minutes)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_register: not set, loading default (0 minutes), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load wait_for_register");
		
		//  WAIT_FOR_SHUTDOWN
		//default is 0 minutes
		content->WAIT_FOR_SHUTDOWN = 0;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load wait_for_shutdown");
		if(PyObject_HasAttrString(content->config_modul, "wait_for_shutdown")){
			shutdown_cycles = PyObject_GetAttrString(content->config_modul, "wait_for_shutdown");
			if(shutdown_cycles == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_shutdown: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(shutdown_cycles)){
					content->WAIT_FOR_SHUTDOWN = (int)PyLong_AsLong(shutdown_cycles);
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" wait_for_shutdown: set to %d min",content->WAIT_FOR_SHUTDOWN);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_shutdown: must be a number, loading default (0 minutes)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_shutdown: not set, loading default (0 minutes), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load wait_for_shutdown");
		
		//  WAIT_FOR_BOOT
		//default is 0 minutes
		content->WAIT_FOR_BOOT = 0;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load wait_for_boot");
		if(PyObject_HasAttrString(content->config_modul, "wait_for_boot")){
			boot_cycles = PyObject_GetAttrString(content->config_modul, "wait_for_boot");
			if(boot_cycles == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_boot: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(boot_cycles)){
					content->WAIT_FOR_BOOT = (int)PyLong_AsLong(boot_cycles);
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" wait_for_boot: set to %d min",content->WAIT_FOR_BOOT);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_boot: must be a number, loading default (0 minutes)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_boot: not set, loading default (0 minutes), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load wait_for_boot");
		
		//  UPDATE_CYCLE
		//default is 60 seconds
		content->UPDATE_CYCLE = 60;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load update_cycle");
		if(PyObject_HasAttrString(content->config_modul, "update_everything")){
			u_cycles = PyObject_GetAttrString(content->config_modul, "update_everything");
			if(u_cycles == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" update_everything: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(u_cycles))
				{
					content->UPDATE_CYCLE = (int)PyLong_AsLong(u_cycles);
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" update_cycle: set to %d sec",content->UPDATE_CYCLE);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" update_cycle: must be a number, loading default (60 seconds)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" update_cycle: not set, loading default (60 seconds), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load update_cycle");

		//  dynamic_update
		//default is 0 (No/Off)
		content->dynamic_update = 0;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load dynamic_update");
		if(PyObject_HasAttrString(content->config_modul, "dynamic_update")){
			d_update = PyObject_GetAttrString(content->config_modul, "dynamic_update");
			if(d_update == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" dynamic_update: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(d_update)){
					content->dynamic_update = (int)PyLong_AsLong(d_update);
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" dynamic_update: set to %d sec",content->dynamic_update);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" dynamic_update: must be a number, loading default (0 / no dynamic update)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" dynamic_update: not set, loading default (0 / no dynamic update), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load dynamic_update");
		
		//  ERROR_THRESHOLD_MINUTES
		//default is 10 mintues
		content->ERROR_THRESHOLD_MINUTES = 10;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load wait_for_error");
		if(PyObject_HasAttrString(content->config_modul, "wait_for_error")){
			e_cycles = PyObject_GetAttrString(content->config_modul, "wait_for_error");
			if(e_cycles == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_error: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(e_cycles)){
					content->ERROR_THRESHOLD_MINUTES = (int)PyLong_AsLong(e_cycles);
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" wait_for_error: set to %d min",content->ERROR_THRESHOLD_MINUTES);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_error: must be a number, loading default (10 minutes)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" wait_for_error: not set, loading default (10 minutes), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load wait_for_error");
		
		//  sequential_shutdown
		//default is off
		content->sequential_shutdown = 0;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load sequential_shutdown");
		if(PyObject_HasAttrString(content->config_modul, "sequential_shutdown")){
			sequential_shutdown = PyObject_GetAttrString(content->config_modul, "sequential_shutdown");
			if(sequential_shutdown == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" sequential_shutdown: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(sequential_shutdown)){	
					content->sequential_shutdown = (int)PyLong_AsLong(sequential_shutdown);
					if(content->sequential_shutdown<0){
						content->sequential_shutdown = 0;
						syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" sequential_shutdown: value less than zero is set to 0 / Off");
					}
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" sequential_shutdown: set to %d",content->sequential_shutdown);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" sequential_shutdown: must be a number, loading default (0 / Off)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" sequential_shutdown: not set, loading default (0 / Off), check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load sequential_shutdown");

		//  Get the Scriptname for the local RMS
		//no default here
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load name for local RMS script");
		if(PyObject_HasAttrString(content->config_modul, "scriptname")){
			scriptname = PyObject_GetAttrString(content->config_modul, "scriptname");
			if(scriptname == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" scriptname: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyString_Check(scriptname)){	
					content->scriptname = strdup(PyString_AS_STRING(scriptname));
					if(content->scriptname == NULL){
						syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" scriptname: strdup failed ");
						syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
						exit(EXIT_FAILURE);
					}
				}
				else{
					syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" scriptname: must be a string");
					syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" scriptname: please make sure, to provide a script for the local RMS which implements the Cherub API");
					syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
					exit(EXIT_FAILURE);
				}
			}
		}
		else{
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" scriptname: not set");
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" scriptname: please make sure, to provide a script for the local RMS which implements the Cherub API");
			syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
			exit(EXIT_FAILURE);
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load name for local RMS script");
		
		//  modul
		//try to load the scriptname for the local RMS (cherubim_script_xxx)
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load module with name: %s",content->scriptname);
		content->modul = PyImport_ImportModule(content->scriptname);
		if(content->modul == NULL){
			PyError = PyErr_Occurred();
			if(PyError != NULL){
				PyErr_Print();
			}
			syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" load module: failed to load %s.py",content->scriptname);
			syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
			exit(EXIT_FAILURE);
		}
		else{
			/* Print the loaded cherubim_script_xxx file to the user, he now can check if it was the right one */
			PyObject* tmp_config_name = PyImport_AddModule(content->scriptname);	//Borrowed reference
			PyObject* globalDictionary = PyModule_GetDict(tmp_config_name);		//Borrowed reference
			const char* pythonScript = "result = __file__\n";
			PyRun_String(pythonScript, Py_file_input, globalDictionary, globalDictionary);
			const char* result = PyString_AsString(PyDict_GetItemString(globalDictionary, "result"));
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" CHERUBIM_SCRIPT_LOADED: %s",result);
			syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load module with name: %s",content->scriptname);
		}
		
		//  Backup
		//default is 0% backup
		content->backup = 0;
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] TRY  - load backup");
		if(PyObject_HasAttrString(content->config_modul, "backup")){
			backup = PyObject_GetAttrString(content->config_modul, "backup");
			if(sequential_shutdown == NULL){
				syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" backup: could not get value (PyObject_GetAttrString failed)");
				syslog(LOG_ERR, "[MASTER] [cherub_load_config] --- EXITING at line %d in %s ---", __LINE__, __FILE__);
				exit(EXIT_FAILURE);
			}
			else{
				if(PyInt_Check(backup)){	
					content->backup = (int)PyLong_AsLong(backup);
					if(content->backup<0){
						content->backup = 0;
						syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" backup: value less than 0 is set to 0 / Off");
					}
					if(content->backup>100){
						content->backup = 100;
						syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" backup: value more than 100 is set to 100 (all nodes on, why do you have installed cherub?)");
					}
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_GREEN" [OK]"ANSI_COLOR_RESET" backup: set to %d",content->backup);
				}
				else{
					syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" backup: must be a number, loading default (0 / Off)");
				}
			}
		}
		else{
			syslog(LOG_INFO, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" backup: not set, loading default (0 / Off), if you like to change check /etc/cherub_config.py");
		}
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] DONE - load backup");
		
	}
	else
	{
		PyErr_Print();
		syslog(LOG_ERR, "[MASTER] [cherub_load_config]"ANSI_COLOR_RED" [X]"ANSI_COLOR_RESET" FAILED - to load %s%s (file can be changed via the -c option)", config_path , config_name);
		exit(EXIT_FAILURE);
	}
	
	if(d_update!=NULL) Py_CLEAR(d_update);
	if(log_facility!=NULL) Py_CLEAR(log_facility);
	if(pythonpath!=NULL) Py_CLEAR(pythonpath);
	if(email!=NULL) Py_CLEAR(email);
	if(u_cycles !=NULL) Py_CLEAR(u_cycles);
	if(sign_off_cycles !=NULL) Py_CLEAR(sign_off_cycles);
	if(register_cycles !=NULL) Py_CLEAR(register_cycles);
	if(shutdown_cycles !=NULL) Py_CLEAR(shutdown_cycles);
	if(boot_cycles !=NULL) Py_CLEAR(boot_cycles);
	if(e_cycles !=NULL) Py_CLEAR(e_cycles);
	if(sequential_shutdown !=NULL) Py_CLEAR(sequential_shutdown);
	if(scriptname !=NULL) Py_CLEAR(scriptname);
	if(loglvl !=NULL) Py_CLEAR(loglvl);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] UPDATE_CYCLE (update_everything) is: %d",content->UPDATE_CYCLE);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] WAIT_FOR_SIGN_OFF is: %d",content->WAIT_FOR_SIGN_OFF);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] WAIT_FOR_REGISTER is: %d",content->WAIT_FOR_REGISTER);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] WAIT_FOR_SHUTDOWN is: %d",content->WAIT_FOR_SHUTDOWN);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] WAIT_FOR_BOOT is: %d",content->WAIT_FOR_BOOT);
	if(content->sequential_shutdown == 0)
	{
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] sequential_shutdown is: OFF");
	}
	else
	{
		syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] sequential_shutdown is: %d",content->sequential_shutdown);
	}

	/* load all thresholds */
	
	content->MINUTES_NORMALYSER = (int)(60/content->UPDATE_CYCLE);
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] MINUTES_NORMALYSER is: %d",content->MINUTES_NORMALYSER);
	
	content->SIGN_OFF_THRESHOLD = content->WAIT_FOR_SIGN_OFF*content->MINUTES_NORMALYSER;
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] SIGN_OFF_THRESHOLD is: %d",content->SIGN_OFF_THRESHOLD);
	
	content->REGISTER_THRESHOLD = content->WAIT_FOR_REGISTER*content->MINUTES_NORMALYSER;
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] REGISTER_THRESHOLD is: %d",content->REGISTER_THRESHOLD);
	
	content->SHUTDOWN_THRESHOLD = content->WAIT_FOR_SHUTDOWN*content->MINUTES_NORMALYSER;
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] SHUTDOWN_THRESHOLD is: %d",content->SHUTDOWN_THRESHOLD);
	
	content->BOOT_THRESHOLD = content->WAIT_FOR_BOOT*content->MINUTES_NORMALYSER;
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] BOOT_THRESHOLD is: %d",content->BOOT_THRESHOLD);
	
	content->ERROR_THRESHOLD = content->ERROR_THRESHOLD_MINUTES*content->MINUTES_NORMALYSER;
	syslog(LOG_DEBUG, "[MASTER] [cherub_load_config] ERROR_THRESHOLD (wait_for_error) is: %d",content->ERROR_THRESHOLD);

	content->mails_to_send = 0;

	return 1;
}

int cherub_finalize(struct cherub_cluster *cluster, struct cherub_content *content, int noaction_flag)
{
	syslog(LOG_DEBUG, "[MASTER] [cherub_finalize] TRY  - Py_Finalize");
	if(content->modul!=NULL) Py_CLEAR(content->modul);
	if(content->config_modul!=NULL) Py_CLEAR(content->config_modul);
	Py_Finalize();
	syslog(LOG_DEBUG, "[MASTER] [cherub_finalize] DONE - Py_Finalize");
	syslog(LOG_DEBUG, "[MASTER] [cherub_finalize] TRY  - free all memory");
	if(!noaction_flag)
	{
		int n;
		for(n=0;n < cluster->number_of_nodes; n++)
		{
			if(cluster->nodes[n].name != NULL)
			{
				free(cluster->nodes[n].name);
			}
			if(cluster->nodes[n].ip_address != NULL)
			{
				free(cluster->nodes[n].ip_address);
			}
		}
	}
	else
	{
		syslog(LOG_INFO, "[MASTER] NO ACTION");
	}
	free(content->scriptname);
	free(content->loglvl);
	syslog(LOG_DEBUG, "[MASTER] [cherub_finalize] DONE - free all memory");
	return 0;
}
