//Copyright (C) 2010 Simon Kiertscher
//
//This program is free software; you can redistribute it and/or modify it 
//under the terms of the GNU General Public License as published by the 
//Free Software Foundation; either version 3 of the License, or (at your option) 
//any later version.
//
//This program is distributed in the hope that it will be useful, but WITHOUT 
//ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS 
//FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
//
//You should have received a copy of the GNU General Public License along with 
//this program; if not, see <http://www.gnu.org/licenses/>.

#ifndef __CHERUBIM_API__
#define __CHERUBIM_API__

#include <Python.h>
#define MAX_NODES 8192

#define COMMANDLINE_HELP        1
#define COMMANDLINE_INFO        2
#define COMMANDLINE_NOACTION    4
#define COMMANDLINE_TEST        8
#define COMMANDLINE_PIDFILE     16
#define COMMANDLINE_CONFIG      32

#define L1 0
#define L2 1
#define L3 2

//possible states for the nodes
//-1 = CHERUB_UNKNOWN // 0 = CHERUB_BUSY // 1 = CHERUB_ONLINE // 2 = CHERUB_OFFLINE // 3 = CHERUB_DOWN // 
enum cherub_status {
	CHERUB_ERROR	=-2,
	CHERUB_UNKNOWN	=-1,
	CHERUB_BUSY	= 0,
	CHERUB_ONLINE	= 1,
	CHERUB_OFFLINE	= 2,
	CHERUB_DOWN	= 3
};	

//data structur for the compute nodes
struct cherub_node{
	char *name;			//the name of the node
	char *ip_address;		//the ipmi ip-address of the node
	enum cherub_status state;	//0 = CHERUB_BUSY // 1 = CHERUB_ONLINE // 2 = CHERUB_OFFLINE // 3 = CHERUB_DOWN // 4 = CHERUB_UNKNWON
	enum cherub_status last_state;	//0 = CHERUB_BUSY // 1 = CHERUB_ONLINE // 2 = CHERUB_OFFLINE // 3 = CHERUB_DOWN // 4 = CHERUB_UNKNWON
	int load;				//0 = node has no load // 1 = node has load and should be booted
	int power_phase;			//what power_phase is used by the node? 1 ,2 or 3
	int always_on;				//should the node be alway_on? 0 = FALSE // 1 = TRUE
	double cycles_running;			//counts how long the node is in the current state
	double cycles_under_load;		//counts how long the node has his load value set to 1
	double cycles_error;			//count the cycles till an error is reported
	int boot_problem;			/* set to 1 if the node has problems while booting */
	int shutdown_problem;			/* set to 1 if the node has problems while shutdown */
	int signoff_problem;			/* set to 1 if the node has problems while signoff */
	int register_problem;			/* set to 1 if the node has problems while register */
	int emergency_routin;			/* set to 1 if the emergency routin was startet*/
};

struct cherub_cluster{
	struct cherub_node nodes[MAX_NODES];	//the compute_nodes
	int number_of_nodes;			//overall nodes
	int running_nodes;			//holds the number of currently running nodes (BUSY AND ONLINE)
	int busy_nodes, online_nodes, offline_nodes, down_nodes, unknown_nodes; //holds the corresponding numbers
	
};

struct cherub_content{
	int UPDATE_CYCLE;	//After how many seconds a update cycle is run
	int dynamic_update;	/*should the update_cycle adjust automatically yes or no*/
	int WAIT_FOR_SIGN_OFF;	//After how many minuts the nodes gets the sign_off command //should be named WAIT_FOR_SIGN_OFF
	int WAIT_FOR_REGISTER;	//After how many minuts the nodes gets the register command 
	int WAIT_FOR_SHUTDOWN;	//After how many minuts the nodes gets the shutdown command while offline and without load
	int WAIT_FOR_BOOT;	//After how many minuts the nodes gets the boot command if he has load
	int MINUTES_NORMALYSER;
	int SIGN_OFF_THRESHOLD;	// <-- WAIT_FOR_SIGN_OFF * MINUTES_NORMALYSER
	int REGISTER_THRESHOLD;	// <-- WAIT_FOR_REGISTER * MINUTES_NORMALYSER
	int SHUTDOWN_THRESHOLD;	// <-- WAIT_FOR_SHUTDOWN * MINUTES_NORMALYSER	
	int BOOT_THRESHOLD;		// <-- WAIT_FOR_BOOT * MINUTES_NORMALYSER
	int ERROR_THRESHOLD;    // <-- ERROR_THRESHOLD_MINUTES * MINUTES_NORMALYSER
	int ERROR_THRESHOLD_MINUTES; 	//After how many minutes a error is reported
	int sequential_shutdown;		//how many nodes can shutdown at the same time 0 = off
	int sequential_from_online_counter;//the corresponding counter
	int sequential_from_offline_counter;//the corresponding counter
	int backup;		/* Backup in percentage, how many nodes has to be ON for backup/overprovisioning */
	
	char *scriptname;				//name of the RMS specific script
	char *loglvl;					//the log lvl 0 = DEBUG, 1 = INFO, 2 = WARNING, 3 = ERROR
	unsigned int options;			//Commandline options
	PyObject *modul,*config_modul;		//the module cherubim_script_[RMS] as python object
	char *email;		/* E-Mail address for sending error and information mails */
	char *config;		/* path to a configfile, other then /etc/cherub_config.py */
	int mails_to_send;	/* do we have mails to send? */

	int parallel_load;	/* 1 if parallel load determination is implemented, 0 if not */
	int parallel_status;	/* 1 if parallel status determination is implemented, 0 if not */
};

//parameter is the node_ipmi_address to shutdown. this function calls a python-script called "cherubim_shutdown.py"
//this should be located in the folder where the programm is run
int cherub_shutdown(const char *node_ipmi_address, struct cherub_content *content);

//parameter is the node_ipmi_address to boot. this function calls a python-script called "cherubim_boot.py"
//this should be located in the folder where the programm is run
int cherub_boot(const char *node_ipmi_address, struct cherub_content *content);
	
//parameter is the node_name to sign off from the RMS. this function calls a python-script called "cherubim_script.py"
//this should be located in the folder where the programm is run
int cherub_sign_off(const char *node_name, struct cherub_content *content);

//parameter is the node_name to register him in the RMS. this function calls a python-script called "cherubim_script.py"
//this should be located in the folder where the programm is run
int cherub_register(const char *node_name, struct cherub_content *content);

//parameter is the node_name by which the status should be reviewed. this function calls a python-script called "cherubim_script.py"
//this should be located in the folder where the programm is run
//returns: 0 = free // 1 = offline // 2 = down // 3 = reserve // 4 = job-exclusive // 5 = job-sharing // 6 = busy // 7 = time-shared // 8 = state-unknown
int cherub_status(const char *node_name, struct cherub_content *content);

//returns a list of node status, the list can be processed in parallel.
//The status of the nodes in the list, must follow the same order like the nodes are
//given in the configuration file!
int* cherub_status_parallel(int *size, struct cherub_content *content);

//Initialize the cherub_config.py and fills the cluster struct named cluster
int cherub_init(struct cherub_cluster* , struct cherub_content *content);

//load the config from the python file
int cherub_load_config(struct cherub_content *content);

//finds out if a node has load and must be booted
int cherub_node_load(const char *node_name, struct cherub_content *content);

//finds out which nodes have load and must be booted
//The nodes in the list, must follow the same order like the nodes are
//given in the configuration file!
int* cherub_load_list(int *size, struct cherub_content *content);

//returns the number of nodes to boot (explicit specified are not counted)
int cherub_global_load(struct cherub_content *content);

//free all the memory and finalize the python interpretor
int cherub_finalize(struct cherub_cluster *cluster, struct cherub_content *content, int noaction_flag);

/*send email to address specified in email from config file*/
int cherub_mail(struct cherub_content *content, char *subject, char *mailcontent);

/*wrapper for multiproblem mails, especially for the boot problems*/
int send_problem_mails(struct cherub_content *content, struct cherub_cluster *cluster);

/*vor a dynamic update cycle*/
int adjust_update_cycle(int current_cycle, struct cherub_content *content);

#endif
