/**************************************************************************/
/*!
@file     MQ135.cpp
@author   G.Krocker (Mad Frog Labs)
@license  GNU GPLv3

First version of an Arduino Library for the MQ135 gas sensor
TODO: Review the correction factor calculation. This currently relies on
the datasheet but the information there seems to be wrong.

@section  HISTORY

v1.0 - First release
v 1.1 - extensions and corrections by Mareen Przybylla with input provided by
the community, esp. Vilius Kraujutis (https://github.com/ViliusKraujutis) and
Andreas Grillenberger

*/
/**************************************************************************/

#include "MQ135.h"

/**************************************************************************/
/*!
@brief  Default constructor

@param[in] pin  The analog input pin for the readout of the sensor
@param[in] rZero  The calibrated value for rZero or 0 if not calibration
*/
/**************************************************************************/

MQ135::MQ135(uint8_t pin, float rZero) {
  _pin = pin;
  _rZero = rZero;
}


/**************************************************************************/
/*!
@brief  Get the correction factor to correct for temperature and humidity

@param[in] t  The ambient air temperature (in °C)
@param[in] h  The relative humidity (in %)

@return The calculated correction factor
*/
/**************************************************************************/
float MQ135::getCorrectionFactor(float t, float h) {
  // Linearization of the temperature dependency curve under and above 20 degree C
    // below 20degC: fact = a * t * t - b * t - (h - 33) * d
    // above 20degC: fact = a * t + b * h + c
    // this assumes a linear dependency on humidity
  if(t < 20){
      return CORA * t * t - CORB * t + CORC - (h-33.)*CORD;
  } else {
      return CORE * t + CORF * h + CORG;
  }
}

/**************************************************************************/
/*!
@brief  Get the resistance of the sensor, ie. the measurement value

@return The sensor resistance in kOhm
*/
/**************************************************************************/
float MQ135::getResistance() {
  int val = analogRead(_pin);
  return ((1023./(float)val) - 1.)*RLOAD;
}

/**************************************************************************/
/*!
@brief  Get the resistance of the sensor, ie. the measurement value corrected
        for temp/hum

@param[in] t  The ambient air temperature (in °C)
@param[in] h  The relative humidity (in %)

@return The corrected sensor resistance kOhm
*/
/**************************************************************************/
float MQ135::getCorrectedResistance(float t, float h) {
  return getResistance()/getCorrectionFactor(t, h);
}

/**************************************************************************/
/*!
@brief  Get the ppm of CO2 sensed (neglecting concentration of other gases than
CO2 in the air)

@return CO2 concentration in ppm in the air or 0 if uncalibrated
*/
/**************************************************************************/
float MQ135::getPPM() {
  if(!isCalibrated())
    return 0;
  return PARA * pow((getResistance()/_rZero), -PARB);
}

/**************************************************************************/
/*!
@brief  Get the ppm of CO2 sensed (neglecting concentration of other gases than
CO2 in the air), corrected for temp/hum

@param[in] t  The ambient air temperature (in °C)
@param[in] h  The relative humidity (in %)

@return CO2 concentration in ppm in the air or 0 if uncalibrated
*/
/**************************************************************************/
float MQ135::getCorrectedPPM(float t, float h) {
  if(!isCalibrated())
    return 0;
  return PARA * pow((getCorrectedResistance(t, h)/_rZero), -PARB);
}

/**************************************************************************/
/*!
@brief  Get the resistance RZero of the sensor for calibration purposes

@return The sensor resistance RZero in kOhm
*/
/**************************************************************************/
float MQ135::getRZero() {
  return getResistance() * pow((ATMOCO2/PARA), (1./PARB));
}

/**************************************************************************/
/*!
@brief  Get the resistance RZero of the sensor for calibration purposes

@return The sensor resistance RZero in kOhm
*/
/**************************************************************************/
float MQ135::getRZeroWithTargetPPM(float ppm) {
  return getResistance() * pow((ppm/PARA), (1./PARB));
}

/**************************************************************************/
/*!
@brief  Get the corrected resistance RZero of the sensor for calibration
        purposes

@param[in] t  The ambient air temperature (in °C)
@param[in] h  The relative humidity (in %)

@return The corrected sensor resistance RZero in kOhm
*/
/**************************************************************************/
float MQ135::getCorrectedRZero(float t, float h) {
  return getCorrectedResistance(t, h) * pow((ATMOCO2/PARA), (1./PARB));
}

/**************************************************************************/
/*!
@brief  Checks if rZero for calibration was set in contructor

@return Boolean for whether calibrated rZero was set
*/
/**************************************************************************/
bool MQ135::isCalibrated() {
  if(_rZero != 0)
    return true;
  return false;
}

/**************************************************************************/
/*!
@brief  Sets rZero after calibration

@return void
*/
/**************************************************************************/
void MQ135::setRZero(float rZero) {
  _rZero = rZero;
}
